
package plf;

/**
*  This class allows the user to define a piecewise linear function over
*  a 2-dimensional domain.  Each dimension is divided into equally
*  spaced intervals, but the separate dimensions can have different
*  spacing and different numbers of points.
*/
public class Function_2D
   {
   /**
   *  The name of the function represented.
   */
   String name;

   /**
   *  The number of sample points in each dimension.
   */
   int[] dim_counts = new int[2];

   /**
   *  The extents of the x-value domain.
   */
   double[] min_x_values;
   double[] max_x_values;

   /**
   *  The array of y_values in the function representation.  These are
   *  assumed to be in the linearized order of "last dimension varies fastest".
   *  It must also be of size n^2.
   */
   double[] y_values = new double[2];

   /**
   *  The extents of the y-value range.
   */
   Double min_y_value;
   Double max_y_value;

   /**
   *  The index of the last point in each dimension, and also the number of
   *  intervals in each dimension.
   */
   int[] last_x_indices = new int[2];

   /**
   *  The increment between indices for the x_values in each dimension.
   */
   Double[] x_increments = new Double[2];

   /**
   *  The scale factor from the x_values to the indices in each dimension.
   */
   Double[] x_scale_factors = new Double[2];

   /**
   *  The constructor stores the function value defintions.  The
   *  min_x-values and max_x_values represent the domain extents
   *  in each dimension for the function.  The y_values must be the
   *  values for equally spaced x_values, including the bounds of the
   *  domain extents. This function assumes that the sequence of value pairs
   *  defines an adequate approximation of a continuous function over the
   *  intervals of x_values in all dimensions.
   *  @param name The name of the function represented.
   *  @param dim_counts The number of points per dimension axis.
   *  @param min_x_values The minimum x_values.
   *  @param max_x_values The maximum x_values.
   *  @param y_values The array of y-axis values.
   *  @throws Exception if discrepencies are identified.
   */
   public Function_2D(String name,
                      int[] dim_counts,
                      double[] min_x_values,
                      double[] max_x_values,
                      double[] y_values)
      throws Exception
      {
      if (dim_counts.length != 2)
         {
         throw(new Exception("Function_2D(name,"
                                        + " dim_counts,"
                                        + " min_x_values, max_x_values,"
                                        + " y_values)"
                           + " y_values)"
                           + " constructor ERROR detected:\n"
                           + "   The dim_counts array does not"
                           + " have 2 entries.\n"
                            ));
         }
      if (min_x_values.length != 2)
         {
         throw(new Exception("Function_2D(name,"
                                        + " dim_counts,"
                                        + " min_x_values, max_x_values,"
                                        + " y_values)"
                           + " y_values)"
                           + " constructor ERROR detected:\n"
                           + "   The min_x_values array does not"
                           + " have 2 entries.\n"
                            ));
         }
      if (max_x_values.length != 2)
         {
         throw(new Exception("Function_2D(name,"
                                        + " dim_counts,"
                                        + " min_x_values, max_x_values,"
                                        + " y_values)"
                           + " y_values)"
                           + " constructor ERROR detected:\n"
                           + "   The max_x_values array does not"
                           + " have 2 entries.\n"
                            ));
         }

      if (y_values.length <= 1)
         {
         throw(new Exception("Function_2D(name,"
                                        + " dim_counts,"
                                        + " min_x_values, max_x_values,"
                                        + " y_values)"
                           + " y_values)"
                           + " constructor ERROR detected:\n"
                           + "   The y_values array has 1 or fewer entries.\n"
                            ));
         }
      this.name = name;
      this.dim_counts = dim_counts;
      this.min_x_values = min_x_values;
      this.max_x_values = max_x_values;
      this.y_values = y_values;

      int num_points = 1;
      for (int dim_index = 0;
           dim_index < 2;
           dim_index++)
         {
         double min_x_value = min_x_values[dim_index];
         double max_x_value = max_x_values[dim_index];

         double dim_x_diff = (max_x_value - min_x_value);
         if (dim_x_diff <= 0.0)
            {
            throw(new Exception("Function_2D(name,"
                                           + " dim_counts,"
                                           + " min_x_values, max_x_values,"
                                           + " y_values)"
                              + " constructor ERROR detected:\n"
                              + "   The min_x_value is not less than the"
                              + " max_x_value for dimension ("
                              + dim_index
                              + ").\n"
                               ));
            }

         int dim_count = dim_counts[dim_index];
         if (dim_count < 2)
            {
            throw(new Exception("Function_2D(name,"
                                           + " dim_counts,"
                                           + " min_x_values, max_x_values,"
                                           + " y_values)"
                              + " constructor ERROR detected:\n"
                              + "   The dim_count doesn't allow for at least"
                              + " 2 values per axis.\n"
                               ));
            }
         num_points *= dim_count;

         Integer last_x_index = dim_count - 1;
         this.last_x_indices[dim_index] = last_x_index;

         Double interval_count = last_x_index.doubleValue();
         Double x_increment = interval_count / dim_x_diff;
         double x_scale_factor = 1.0 / x_increment;

         this.x_scale_factors[dim_index] = x_scale_factor;
         this.x_increments[dim_index] = x_increment;
         }
      if (y_values.length != num_points)
         {
         throw(new Exception("Function_2D(name,"
                                        + " dim_counts,"
                                        + " min_x_values, max_x_values,"
                                        + " y_values)"
                           + " y_values)"
                           + " constructor ERROR detected:\n"
                           + "   The y_values array does not have"
                           + " dim_counts[0] * dim_counts[1] entries."
                           + "      y_values.length = "
                              + y_values.length
                              + "\n"
                           + "      dim_counts[0] = "
                              + dim_counts[0]
                              + "\n"
                           + "      dim_counts[1] = "
                              + dim_counts[1]
                              + "\n"
                           + "      product = "
                              + num_points
                           + "\n"
                            ));
         }

      /*
      *  Get the extents of the y_values.
      */
      this.min_y_value = y_values[0];
      this.max_y_value = y_values[0];
      for (int point_index = 1;
           point_index < y_values.length;
           point_index++)
         {
         if (this.min_y_value > y_values[point_index])
            {
            this.min_y_value = y_values[point_index];
            }
         if (this.max_y_value < y_values[point_index])
            {
            this.max_y_value = y_values[point_index];
            }
         }
      }

   /**
   *  Get the function value for a given input value.
   *  @param x_values the arguments in each dimension to the function.
   *  @return the y_value approximation.
   *  @throws Exception
   */
   public double getValue(double[] x_values)
      throws Exception
      {
      double y_value;

      if (x_values.length != 2)
         {
         throw(new Exception("plf.getValue(x_values)"
                           + " ERROR detected:\n"
                           + "   The x_values array has the"
                           + " wrong number of values ("
                           + x_values.length
                           + ")."
                           + "   It should have one for each dimension (2"
                           + ") in the function.\n"
                            ));
         }

      int[] x_low_indices = new int[2];

      double[] x_low_ratios = new double[2];
      double[] x_high_ratios = new double[2];

      for (int dim_index = 0;
           dim_index < 2;
           dim_index++)
         {
         double x_value = x_values[dim_index];
         double min_x_value = this.min_x_values[dim_index];
         double max_x_value = this.max_x_values[dim_index];

         double x_scale_factor = this.x_scale_factors[dim_index];
         double x_increment = this.x_increments[dim_index];

         if (x_value < min_x_value)
            {
            throw(new Exception("plf.getValue(x_values)"
                              + " ERROR detected:\n"
                              + "   The x_values["
                              + dim_index
                              + "] ("
                              + x_value
                              + ") is less than the"
                              + " min_x_value["
                              + dim_index
                              + "] ("
                              + min_x_value
                              + ").\n"
                               ));
            }
         if (x_value > max_x_value)
            {
            throw(new Exception("plf.getValue(x_values)"
                              + " ERROR detected:\n"
                              + "   The x_values["
                              + dim_index
                              + "] ("
                              + x_value
                              + ") is greater than the"
                              + " max_x_values["
                              + dim_index
                              + "] ("
                              + max_x_value
                              + ").\n"
                               ));
            }
         double x_index = (x_value - min_x_value) * x_scale_factor;

         Integer x_low_index = (int) Math.floor(x_index);
         Double x_offset = x_index - x_low_index.doubleValue();

         double low_ratio = 0.0;
         double high_ratio = 1.0;
         if (x_offset <= 0.0)
            {
            /*
            *  No need to interpolate.
            */
            low_ratio = 0.0;
            high_ratio = 1.0;
            }
         else if (x_low_index >= this.dim_counts[dim_index])
            {
            /*
            *  No need to interpolate.
            */
            x_low_index = this.dim_counts[dim_index] - 2;
            low_ratio = 0.0;
            high_ratio = 1.0;
            }
         else
            {
            /*
            *  It is between x_values, so we need to interpolate.
            */
            high_ratio = x_offset / x_increment;
            low_ratio = 1.0 - high_ratio;
            }

         x_low_indices[dim_index] = x_low_index;

         x_low_ratios[dim_index] = low_ratio;
         x_high_ratios[dim_index] = high_ratio;
         }

      int y00_index = x_low_indices[0] * this.dim_counts[0] +
                      x_low_indices[1];
      int y01_index = x_low_indices[0] * this.dim_counts[0] +
                      (x_low_indices[1] + 1);
      int y10_index = (x_low_indices[0] + 1) * this.dim_counts[0] +
                      x_low_indices[1];
      int y11_index = (x_low_indices[0] + 1) * this.dim_counts[0] +
                      (x_low_indices[1] + 1);

      double y00_coef = x_low_ratios[0] * x_low_ratios[1];
      double y01_coef = x_low_ratios[0] * x_high_ratios[1];
      double y10_coef = x_high_ratios[0] * x_low_ratios[1];
      double y11_coef = x_high_ratios[0] * x_high_ratios[1];

      y_value = this.y_values[y00_index] * y00_coef +
                this.y_values[y01_index] * y01_coef +
                this.y_values[y10_index] * y10_coef +
                this.y_values[y11_index] * y11_coef;

      return(y_value);
      }
   

   /**
   *  Formats the function definition into a string.
   */
   public String toString()
      {
      String out_string = new String();
      String.format("Function_2D: %s() (%3d X %3d) using %4d points.\n"
                  + "        y_range = (%9.3f, %9.3f)\n",
                    this.name,
                    this.dim_counts[0],
                    this.dim_counts[1],
                    this.y_values.length,
                    this.min_y_value,
                    this.max_y_value
                   );
      int point_index = 0;
      String out_line = new String();
      for (int d0_index = 0;
           d0_index < this.dim_counts[0];
           d0_index++)
         {
         String.format("   %3d: %9.3f --\n",
                       d0_index,
                       (this.x_scale_factors[0] * (double) d0_index +
                        this.min_x_values[0])
                       );
         out_string += out_line;
         for (int d1_index = 0;
              d1_index < this.dim_counts[1];
              d1_index++)
            {
            String.format("      (%3d: %9.3f) - %9.3f\n",
                          d1_index,
                          (this.x_scale_factors[1] * (double) d1_index +
                           this.min_x_values[1]),
                          this.y_values[point_index]
                         );
            out_string += out_line;
            point_index++;
            }
         }
      out_string += "\n";
      return(out_string);
      }

   /**
   *  Formats the function definition into a string and prints it.
   */
   public void Print()
      {
      System.out.format("Function_2D: %s() (%3d X %3d) using %4d points.\n"
                      + "        y_range = (%9.3f, %9.3f)\n",
                        this.name,
                        this.dim_counts[0],
                        this.dim_counts[1],
                        this.y_values.length,
                        this.min_y_value,
                        this.max_y_value
                       );
      int point_index = 0;
      for (int d0_index = 0;
           d0_index < this.dim_counts[0];
           d0_index++)
         {
         System.out.format("   %3d: %9.3f --\n",
                           d0_index,
                           (this.x_scale_factors[0] * (double) d0_index +
                            this.min_x_values[0])
                           );
         for (int d1_index = 0;
              d1_index < this.dim_counts[1];
              d1_index++)
            {
            System.out.format("      (%3d: %9.3f) - %9.3f\n",
                              d1_index,
                              (this.x_scale_factors[1] * (double) d1_index +
                               this.min_x_values[1]),
                              this.y_values[point_index]
                             );
            point_index++;
            }
         }
      System.out.println("\n");
      }
   }

