
/**
* The PlotAxes class creates an image of a pair of axes on which a plot
* may later be displayed or which can be used for one of the three
* planes for 3-D plots.
*/

package plot;


import java.util.*;
import java.awt.*;
import java.awt.font.*;
import java.awt.geom.*;
import java.awt.image.*;

import util.*;

/**
*  The PlotAxes definition.  This class generates the plane of two axes as
*  an image, allowing the placement of
*  <ol type=1>
*     <li>The axes box,
*     <li>The labels for each axis,
*     <li>The major and minor tics,
*     <li>The value labels (corresponding to major tics),
*     <li>Grid lines for each axis.
*     </ol>
*/
public class PlotAxes
   {

   enum Location
      {
      TOP,
      BOTTOM,
      LEFT,
      RIGHT
      };

   String h_axis_title = "H-Axis";
   String v_axis_title = "V-Axis";

   Axis h_axis = null;
   Axis v_axis = null;

   BufferedImage plot_image = null;

   double h_border = 10.0;
   double v_border = 10.0;

   FontRenderContext font_render_context = null;
   private Font axis_label_font = new Font("Serif", Font.BOLD, 16);
   private Font tic_label_font = new Font("Serif", Font.BOLD, 12);

   double major_tic_length = 5.0;
   double minor_tic_length = 2.0;

   Rectangle2D.Double plot_box = null;

   Color grid_color = Color.darkGray;

   boolean display_top_label = false;
   boolean display_bottom_label = false;
   boolean display_top_tic_labels = false;
   boolean display_bottom_tic_labels = false;
   boolean display_top_major_tics = false;
   boolean display_bottom_major_tics = false;
   boolean display_top_minor_tics = false;
   boolean display_bottom_minor_tics = false;
   boolean display_h_grid = false;
   double h_margin = 10.0;

   ArrayList<String> h_tic_labels = null;
   ArrayList<Double> h_major_tics = null;
   ArrayList<Double> h_minor_tics = null;

   boolean display_left_label = false;
   boolean display_right_label = false;
   boolean display_left_tic_labels = false;
   boolean display_right_tic_labels = false;
   boolean display_left_major_tics = false;
   boolean display_right_major_tics = false;
   boolean display_left_minor_tics = false;
   boolean display_right_minor_tics = false;
   boolean display_v_grid = false;
   double v_margin = 10.0;
   
   ArrayList<String> v_tic_labels = null;
   ArrayList<Double> v_major_tics = null;
   ArrayList<Double> v_minor_tics = null;


   double h_values_min = 0.0;
   double h_values_range = 1.0;
   double h_values_scale = 1.0;

   double v_values_min = 0.0;
   double v_values_range = 1.0;
   double v_values_scale = 1.0;

   /**
   *  Create the plot object with the given titles/labels.
   *  @param h_axis the horizontal axis.
   *  @param h_border the space to allow for the border.
   *  @param v_axis the vertical axis.
   *  @param v_border the space to allow for the border.
   */
   public PlotAxes(Axis h_axis,
                   double h_border,
                   Axis v_axis,
                   double v_border)
      {
      this.h_axis = h_axis;
      this.v_axis = v_axis;

      this.h_border = h_border;
      this.v_border = v_border;

      this.h_axis_title = h_axis.getTitle();
      this.v_axis_title = v_axis.getTitle();

/*
      System.out.println("PlotAxes Constructor on ENTRY:\n"
                       + "   title = " + this.title + "\n"
                       + "   h_axis_title = " + this.h_axis_title + "\n"
                       + "   v_axis_title = " + this.v_axis_title + "\n"
                        );
*/

      this.updatePlotValueStats();
/*
      System.out.println("Plot2D Constructor on EXIT.");
*/
      }

   /**
   * Update the min and range for the plot axes offsets and scale factors.
   */
   public void updatePlotValueStats()
      {
      /*
      *  Scale the data range to the plot area size.
      */
      this.h_values_min = this.h_axis.getAdjustedValueMin();
      this.h_values_range = this.h_axis.getAdjustedValueRange();

      this.v_values_min = this.v_axis.getAdjustedValueMin();
      this.v_values_range = this.v_axis.getAdjustedValueRange();

/*
      System.out.println("PlotAxes.updatePlotValueStats() on EXIT:\n"
                       + "   (min, range) = ("
                       + this.h_values_min
                       + ", "
                       + this.h_values_range
                       + "), ("
                       + this.v_values_min
                       + ", "
                       + this.v_values_range
                       + ")"
                        );
*/
      }


   /**
   *  Set the plot options for the horizontal axes labeling and tics.
   *  @param display_top_label true to display the label at the top.
   *  @param display_bottom_label true to display the label at the bottom.
   *  @param display_top_tic_labels true to display the tic labels at the
   *       top.
   *  @param display_bottom_tic_labels true to display the tic labels at
   *       the bottom.
   *  @param display_top_major_tics true to display the major tics at the
   *       top.
   *  @param display_bottom_major_tics true to display the major tics at
   *       the bottom.
   *  @param display_top_minor_tics true to display the minor tics at the
   *       top (if the major are displayed on the top).
   *  @param display_bottom_minor_tics true to display the minor tics at
   *       the bottom (if the major are displayed on the bottom).
   *  @param display_h_grid true to display this orientation of the grid
   *       aligned with major tic positions.
   */
   public void setHorizontalOptions(boolean display_top_label,
                                    boolean display_bottom_label,
                                    boolean display_top_tic_labels,
                                    boolean display_bottom_tic_labels,
                                    boolean display_top_major_tics,
                                    boolean display_bottom_major_tics,
                                    boolean display_top_minor_tics,
                                    boolean display_bottom_minor_tics,
                                    boolean display_h_grid)
      {
      this.display_top_label = display_top_label;
      this.display_bottom_label = display_bottom_label;
      this.display_top_tic_labels = display_top_tic_labels;
      this.display_bottom_tic_labels = display_bottom_tic_labels;
      this.display_top_major_tics = display_top_major_tics;
      this.display_bottom_major_tics = display_bottom_major_tics;
      this.display_top_minor_tics = display_top_minor_tics;
      this.display_bottom_minor_tics = display_bottom_minor_tics;
      this.display_h_grid = display_h_grid;
      }


   /**
   *  Set the plot options for the vertical axes labeling and tics.
   *  @param display_left_label true to display the label at the left.
   *  @param display_right_label true to display the label at the right.
   *  @param display_left_tic_labels true to display the tic labels at the
   *       left.
   *  @param display_right_tic_labels true to display the tic labels at the
   *       right.
   *  @param display_left_major_tics true to display the major tics at the
   *       left.
   *  @param display_right_major_tics true to display the major tics at the
   *       right.
   *  @param display_left_minor_tics true to display the minor tics at the
   *       left (if the major are displayed on the left).
   *  @param display_right_minor_tics true to display the minor tics at
   *       the right (if the major are displayed on the right).
   *  @param display_v_grid true to display this orientation of the grid
   *       aligned with major tic positions.
   */
   public void setVerticalOptions(boolean display_left_label,
                                  boolean display_right_label,
                                  boolean display_left_tic_labels,
                                  boolean display_right_tic_labels,
                                  boolean display_left_major_tics,
                                  boolean display_right_major_tics,
                                  boolean display_left_minor_tics,
                                  boolean display_right_minor_tics,
                                  boolean display_v_grid)
      {
      this.display_left_label = display_left_label;
      this.display_right_label = display_right_label;
      this.display_left_tic_labels = display_left_tic_labels;
      this.display_right_tic_labels = display_right_tic_labels;
      this.display_left_major_tics = display_left_major_tics;
      this.display_right_major_tics = display_right_major_tics;
      this.display_left_minor_tics = display_left_minor_tics;
      this.display_right_minor_tics = display_right_minor_tics;
      this.display_v_grid = display_v_grid;
      }

   /**
   *  Get the plot box.
   *  @return the current plot box reference.
   */
   public Rectangle2D.Double getPlotBox()
      {
      return(this.plot_box);
      }

   /**
   *  Get the horizontal minimum data value.
   *  @return the horizontal minimum data value.
   */
   public double getHorizontalValuesMin()
      {
      return(this.h_values_min);
      }

   /**
   *  Get the vertical minimum data value.
   *  @return the vertical minimum data value.
   */
   public double getVerticalValuesMin()
      {
      return(this.v_values_min);
      }

   /**
   *  Get the horizontal data values range.
   *  @return the horizontal data values range
   */
   public double getHorizontalValuesRange()
      {
      return(this.h_values_range);
      }

   /**
   *  Get the vertical data values range.
   *  @return the vertical data values range
   */
   public double getVerticalValuesRange()
      {
      return(this.v_values_range);
      }

   /**
   *  Get the horizontal scale coefficient.
   *  @return the horizontal scale value.
   */
   public double getHorizontalValuesScale()
      {
      return(this.h_values_scale);
      }

   /**
   *  Get the vertical scale coefficient.
   *  @return the vertical scale value.
   */
   public double getVerticalValuesScale()
      {
      return(this.v_values_scale);
      }


   /**
   *  Draw the contents of the Plot2D image, including plot area, axis labels,
   *  grid, tic marks, and tic labels.
   *  @param image_box the box giving the image sized rectangle.
   */
   public BufferedImage drawContents(Rectangle2D.Double image_box,
                                     Color background_color)
      {
      double width = image_box.width;
      double height = image_box.height;

/*
      System.out.println("PlotPanel::drawContents() on ENTRY:\n"
                       + "   image size = ("
                       + width
                       + ", "
                       + height
                       + ")\n"
                        );
*/

      this.plot_image = new BufferedImage((int) image_box.getWidth(),
                                          (int) image_box.getHeight(),
                                          BufferedImage.TYPE_INT_ARGB);
      Graphics2D graphics_2d = plot_image.createGraphics();

      if (background_color!= null)
         {
         graphics_2d.setBackground(background_color);
         }

      this.font_render_context = graphics_2d.getFontRenderContext();

      /*
      *  Get the actual plot area size by first adding in all the decorations
      *  and allowing space for them.
      *     Main title
      *     Horizontal and Vertical axis titles.
      *     Horizontal and Vertical tics and tic labels.
      */

      /*
      *  Create a rectangle with all zeros for later changes to the plot box.
      */
      Rectangle2D.Double plot_box_alterations = new Rectangle2D.Double(0.0,
                                                                       0.0,
                                                                       0.0,
                                                                       0.0);

      /*
      *  The axis labels, tics, and tic labels must be checked to see how
      *  much the plot box must be adjusted.
      */

      /*
      *  Preprocess the axis labels for later display.
      */
      TextLayout top_title_layout = null;
      TextLayout bottom_title_layout = null;
      TextLayout left_title_layout = null;
      TextLayout right_title_layout = null;
      if (this.display_top_label)
         {
         top_title_layout = this.prepAxisLabel(Location.TOP,
                                               this.h_axis_title,
                                               plot_box_alterations);
         }
      if (this.display_bottom_label)
         {
         bottom_title_layout = this.prepAxisLabel(Location.BOTTOM,
                                                  this.h_axis_title,
                                                  plot_box_alterations);
         }
      if (this.display_left_label)
         {
         left_title_layout = this.prepAxisLabel(Location.LEFT,
                                                this.v_axis_title,
                                                plot_box_alterations);
         }
      if (this.display_right_label)
         {
         right_title_layout = this.prepAxisLabel(Location.RIGHT,
                                                 this.v_axis_title,
                                                 plot_box_alterations);
         }

      /*
      *  Get tic positions if they are needed.
      */
      if (this.display_top_tic_labels ||
          this.display_bottom_tic_labels ||
          this.display_top_major_tics ||
          this.display_bottom_major_tics ||
          this.display_v_grid)
         {
         this.h_major_tics = this.h_axis.getMajorTics();
         if (this.display_top_minor_tics ||
             this.display_bottom_minor_tics)
            {
            this.h_minor_tics = this.h_axis.getMinorTics();
            }
         }
      if (this.display_left_tic_labels ||
          this.display_right_tic_labels ||
          this.display_left_major_tics ||
          this.display_right_major_tics ||
          this.display_h_grid)
         {
         this.v_major_tics = this.v_axis.getMajorTics();
         if (this.display_left_minor_tics ||
             this.display_right_minor_tics)
            {
            this.v_minor_tics = this.v_axis.getMinorTics();
            }
         }

      /*
      *  Get tic labels if they are needed.
      */
      if (this.display_top_tic_labels ||
          this.display_bottom_tic_labels)
         {
         this.h_tic_labels = this.h_axis.getTicLabels();
         }
      if (this.display_left_tic_labels ||
          this.display_right_tic_labels)
         {
         this.v_tic_labels = this.v_axis.getTicLabels();
         }

      /*
      *  Preprocess the tic labels for later display.
      */
      ArrayList<TextLayout> top_tic_layouts = null;
      ArrayList<TextLayout> bottom_tic_layouts = null;
      ArrayList<TextLayout> left_tic_layouts = null;
      ArrayList<TextLayout> right_tic_layouts = null;
      if (this.display_top_tic_labels)
         {
         top_tic_layouts = this.prepAxisTicLabels(Location.TOP,
                                                  this.h_tic_labels,
                                                  plot_box_alterations);
         }
      if (this.display_bottom_tic_labels)
         {
         bottom_tic_layouts = this.prepAxisTicLabels(Location.BOTTOM,
                                                     this.h_tic_labels,
                                                     plot_box_alterations);
         }

      if (this.display_left_tic_labels)
         {
         left_tic_layouts = this.prepAxisTicLabels(Location.LEFT,
                                                   this.v_tic_labels,
                                                   plot_box_alterations);
         }
      if (this.display_right_tic_labels)
         {
         right_tic_layouts = this.prepAxisTicLabels(Location.RIGHT,
                                                    this.v_tic_labels,
                                                    plot_box_alterations);
         }

      /*
      *  Preprocess the tics for later display.
      */
      if (this.display_top_major_tics)
         {
         this.prepAxisTics(Location.TOP,
                           this.h_major_tics,
                           plot_box_alterations);
         }
      if (this.display_bottom_major_tics)
         {
         this.prepAxisTics(Location.BOTTOM,
                           this.h_major_tics,
                           plot_box_alterations);
         }

      if (this.display_left_major_tics)
         {
         this.prepAxisTics(Location.LEFT,
                           this.v_major_tics,
                           plot_box_alterations);
         }
      if (this.display_right_major_tics)
         {
         this.prepAxisTics(Location.RIGHT,
                           this.v_major_tics,
                           plot_box_alterations);
         }

      /*
      *  Set the corners of the plot region.
      *  The bottom left will be the origin offset.
      *  The offsets are set to place items outside the plot area,
      *  including tic and tic label clearances.
      */
      this.plot_box = new Rectangle2D.Double(this.h_border,
                                             this.v_border,
                                             width - 2.0 * this.h_border,
                                             height - 2.0 * this.v_border);
      /*
      *  Get the adjusted plot area dimensions to allow for the placement
      *  of the various labels and tics.
      */
      this.plot_box.x +=  plot_box_alterations.x;
      this.plot_box.y +=  plot_box_alterations.y;
      this.plot_box.width += plot_box_alterations.width;
      this.plot_box.height += plot_box_alterations.height;

      /*
      *  Draw the axis labels that are indicated.
      */
      if (this.display_top_label)
         {
         this.drawAxisLabel(graphics_2d, Location.TOP, top_title_layout);
         }
      if (this.display_bottom_label)
         {
         this.drawAxisLabel(graphics_2d, Location.BOTTOM, bottom_title_layout);
         }
      if (this.display_left_label)
         {
         this.drawAxisLabel(graphics_2d, Location.LEFT, left_title_layout);
         }
      if (this.display_right_label)
         {
         this.drawAxisLabel(graphics_2d, Location.RIGHT, right_title_layout);
         }


      /*
      *  Set up the scaling for the tic placement on the horizontal and
      *  vertical axes.
      */
      this.h_values_scale = this.plot_box.width / this.h_values_range;
      this.v_values_scale = this.plot_box.height / this.v_values_range;

/*
      System.out.println("PlotAxes.drawContents() STATUS:\n"
                       + "   this.h_values_range = "
                       + this.h_values_range
                       + "\n"
                       + "   this.v_values_range = "
                       + this.v_values_range
                       + "\n"
                        );
*/

      /*
      *  Draw the tic labels that are indicated.
      */
      if (this.display_top_tic_labels)
         {
         this.drawAxisTicLabels(graphics_2d,
                                Location.TOP,
                                this.h_major_tics,
                                top_tic_layouts);
         }
      if (this.display_bottom_tic_labels)
         {
         this.drawAxisTicLabels(graphics_2d,
                                Location.BOTTOM,
                                this.h_major_tics,
                                bottom_tic_layouts);
         }
      if (this.display_left_tic_labels)
         {
         this.drawAxisTicLabels(graphics_2d,
                                Location.LEFT,
                                this.v_major_tics,
                                left_tic_layouts);
         }
      if (this.display_right_tic_labels)
         {
         this.drawAxisTicLabels(graphics_2d,
                                Location.RIGHT,
                                this.v_major_tics,
                                right_tic_layouts);
         }

      /*
      *  Draw the tics that are indicated.
      */
      if (this.display_top_major_tics)
         {
         this.drawAxisTics(graphics_2d,
                           Location.TOP,
                           this.display_top_minor_tics,
                           this.h_major_tics,
                           this.h_minor_tics);
         }
      if (this.display_bottom_major_tics)
         {
         this.drawAxisTics(graphics_2d,
                           Location.BOTTOM,
                           this.display_bottom_minor_tics,
                           this.h_major_tics,
                           this.h_minor_tics);
         }
      if (this.display_left_major_tics)
         {
         this.drawAxisTics(graphics_2d,
                           Location.LEFT,
                           this.display_left_minor_tics,
                           this.v_major_tics,
                           this.v_minor_tics);
         }
      if (this.display_right_major_tics)
         {
         this.drawAxisTics(graphics_2d,
                           Location.RIGHT,
                           this.display_right_minor_tics,
                           this.v_major_tics,
                           this.v_minor_tics);
         }

      /*
      *  Draw the grid lines that are indicated.
      */
      if (this.display_v_grid)
         {
         this.drawAxisGrid(graphics_2d, Location.TOP, this.h_major_tics);
         }
      if (this.display_h_grid)
         {
         this.drawAxisGrid(graphics_2d, Location.LEFT, this.v_major_tics);
         }

      /*
      *  The plot region box is now fixed and can be outlined.
      */
      graphics_2d.setColor(Color.BLACK);
      graphics_2d.draw(this.plot_box);

      return(this.plot_image);
      }

   /**
   *  Get the preparatory data for altering the plot box and drawing the
   *  label.
   *  @param location the location for the label --
   *     TOP -- horizontal above the box.
   *     BOTTOM -- horizontal below the box.
   *     LEFT -- vertical on the left of the box.
   *     RIGHT -- vertical on the right of the box.
   *  @param label the label to prep for.
   *  @param plot_box_alterations the changes to add to the plot box
   *     to compensate for this label's size and position, to be updated
   *     each time this routine is called.
   *  @return the layout for this label to be drawn.
   */
   public TextLayout prepAxisLabel(Location location,
                                   String label,
                                   Rectangle2D.Double plot_box_alterations)
      {
      TextLayout layout = new TextLayout(label,
                                         this.axis_label_font,
                                         this.font_render_context);
      Rectangle2D bounds = layout.getBounds();
      double text_width = bounds.getWidth();
      double text_height = bounds.getHeight();

/*
      System.out.println("PlotAxes.prepAxisLabel(location, label,"
                       + " plot_box_alterations) on ENTRY:\n"
                       + "   location = " + location + "\n"
                       + "   label = " + label + "\n"
                       + "   plot_box_alterations = "
                       + plot_box_alterations
                       + "\n"
                        );
*/

      switch (location)
         {
         case TOP:
            plot_box_alterations.y += text_height + 2.0 * this.v_margin;
            plot_box_alterations.height -= text_height + 2.0 * this.v_margin;
            break;
         case BOTTOM:
            plot_box_alterations.height -= text_height + 2.0 * this.v_margin;
            break;
         case LEFT:
            plot_box_alterations.x += text_height + 2.0 * this.h_margin;
            plot_box_alterations.width -= text_height + 2.0 * this.h_margin;
            break;
         case RIGHT:
            plot_box_alterations.width -= text_height + 2.0 * this.h_margin;
            break;
         }

/*
      System.out.println("PlotAxes.prepAxisLabel(location, label,"
                       + " plot_box_alterations) on ENTRY:\n"
                       + "   plot_box_alterations = "
                       + plot_box_alterations
                       + "\n"
                        );
*/

      return(layout);
      }

   /**
   *  Get the preparatory data for altering the plot box and drawing the
   *  tic labels.
   *  @param location the location for the label --
   *     TOP -- horizontal above the box.
   *     BOTTOM -- horizontal below the box.
   *     LEFT -- vertical on the left of the box.
   *     RIGHT -- vertical on the right of the box.
   *  @param labels the labels to prep for.
   *  @param plot_box_alterations the changes to add to the plot box
   *     to compensate for this label's size and position, to be updated
   *     each time this routine is called.
   *  @return the layout for this label to be drawn.
   */
   public ArrayList<TextLayout>
                     prepAxisTicLabels(Location location,
                                       ArrayList<String> labels,
                                       Rectangle2D.Double plot_box_alterations)
      {
      ArrayList<TextLayout> layouts = new ArrayList<TextLayout>();

      Double max_width = 0.0;
      Double max_height = 0.0;
      for (String label: labels)
         {
         label.trim();
         TextLayout layout = new TextLayout(label,
                                            this.tic_label_font,
                                            this.font_render_context);
         Rectangle2D bounds = layout.getBounds();
         double text_width = bounds.getWidth();
         double text_height = bounds.getHeight();

         if (max_width < text_width)
            {
            max_width = text_width;
            }
         if (max_height < text_height)
            {
            max_height = text_height;
            }
         layouts.add(layout);
         }

      if (layouts.size() > 0)
         {
         switch (location)
            {
            case TOP:
               plot_box_alterations.x += 0.5 * max_height
                                       + this.h_margin;
               plot_box_alterations.width -= max_height
                                           + 2.0 * this.h_margin;
               plot_box_alterations.y += max_width
                                       + 2.0 * this.v_margin;
               plot_box_alterations.height -= max_width
                                            + 2.0 * this.v_margin;
               break;
            case BOTTOM:
               plot_box_alterations.x += 0.5 * max_height
                                       + this.h_margin;
               plot_box_alterations.width -= max_height
                                           + 2.0 * this.h_margin;
               plot_box_alterations.height -= max_width
                                            + 2.0 * this.v_margin;
               break;
            case LEFT:
               plot_box_alterations.x += max_width
                                       + 2.0 * this.h_margin;
               plot_box_alterations.width -= max_width
                                           + 2.0 * this.h_margin;
               plot_box_alterations.y += 0.5 * max_height
                                       + this.v_margin;
               plot_box_alterations.height -= max_height
                                       + 2.0 * this.v_margin;
               break;
            case RIGHT:
               plot_box_alterations.width -= max_width
                                           + 2.0 * this.h_margin;
               plot_box_alterations.y += 0.5 * max_height
                                       + this.v_margin;
               plot_box_alterations.height -= max_height
                                       + 2.0 * this.v_margin;
               break;
            }
         }

      return(layouts);
      }


   /**
   *  Get the preparatory data for altering the plot box and drawing the
   *  major tics.
   *  @param location the location for the label --
   *     TOP -- horizontal above the box.
   *     BOTTOM -- horizontal below the box.
   *     LEFT -- vertical on the left of the box.
   *     RIGHT -- vertical on the right of the box.
   *  @param major_tics the labels to prep for.
   *  @param plot_box_alterations the changes to add to the plot box
   *     to compensate for this label's size and position, to be updated
   *     each time this routine is called.
   */
   public void prepAxisTics(Location location,
                            ArrayList<Double> major_tics,
                            Rectangle2D.Double plot_box_alterations)
      {
      switch (location)
         {
         case TOP:
            plot_box_alterations.height -= this.major_tic_length;
            break;
         case BOTTOM:
            plot_box_alterations.height -= this.major_tic_length;
            break;
         case LEFT:
            plot_box_alterations.width -= this.major_tic_length;
            break;
         case RIGHT:
            plot_box_alterations.width -= this.major_tic_length;
            break;
         }
      }


   /**
   *  Get the preparatory data for altering the plot box and drawing the
   *  label.
   *  @param graphics_2d the graphics to draw to.
   *  @param location the location for the label --
   *     TOP -- horizontal above the box.
   *     BOTTOM -- horizontal below the box.
   *     LEFT -- vertical on the left of the box.
   *     RIGHT -- vertical on the right of the box.
   *  @param layout the layout for this text.
   */
   public void drawAxisLabel(Graphics2D graphics_2d,
                             Location location,
                             TextLayout layout)
      {
      Rectangle2D bounds = layout.getBounds();
      double text_width = bounds.getWidth();
      double text_height = bounds.getHeight();

      Shape text_shape = null;
      AffineTransform text_at = null;

      double blc_x = 0.0;
      double blc_y = 0.0;

      graphics_2d.setColor(Color.BLACK);

      switch (location)
         {
         case TOP:
            blc_x = this.plot_box.x
                  + 0.5 * (this.plot_box.width - text_width);
            blc_y = this.v_border
                  + this.v_margin
                  + text_height;
            layout.draw(graphics_2d, (float) blc_x, (float) blc_y);
            break;
         case BOTTOM:
            blc_x = this.plot_box.x
                  + 0.5 * (this.plot_box.width - text_width);
            blc_y = this.plot_image.getHeight()
                  - this.v_border
                  - this.v_margin;
            layout.draw(graphics_2d, (float) blc_x, (float) blc_y);
            break;
         case LEFT:
            blc_x = this.h_border
                  + this.h_margin
                  + text_height;
            blc_y = this.plot_box.y
                  + 0.5 * (this.plot_box.height + text_width);
            text_at = new AffineTransform();
            text_at.setToIdentity();
            text_at.translate(blc_x, blc_y);
            text_at.rotate(Math.toRadians(-90));
            text_shape = layout.getOutline(text_at);

            graphics_2d.setColor(Color.BLACK);
            graphics_2d.fill(text_shape);
            break;
         case RIGHT:
            blc_x = this.plot_image.getWidth()
                  - this.h_border
                  - this.h_margin;
            blc_y = this.plot_box.y
                  + 0.5 * (this.plot_box.height + text_width);
            text_at = new AffineTransform();
            text_at.setToIdentity();
            text_at.translate(blc_x, blc_y);
            text_at.rotate(Math.toRadians(-90));
            text_shape = layout.getOutline(text_at);

            graphics_2d.setColor(Color.BLACK);
            graphics_2d.fill(text_shape);
            break;
         }
      }


   /**
   *  Draw the tic labels in the correct positions.
   *  @param graphics_2d the graphics to draw to.
   *  @param location the location for the label --
   *     TOP -- horizontal above the box.
   *     BOTTOM -- horizontal below the box.
   *     LEFT -- horizontal on the left of the box.
   *     RIGHT -- horizontal on the right of the box.
   *  @param layouts the layouts for the text strings.
   */
   public void drawAxisTicLabels(Graphics2D graphics_2d,
                                 Location location,
                                 ArrayList<Double> major_tics,
                                 ArrayList<TextLayout> layouts)
      {
      if (major_tics.size() > 0 &&
          layouts.size() == major_tics.size())
         {
         graphics_2d.setStroke(new BasicStroke(1.0F));
         graphics_2d.setColor(Color.BLACK);

         Shape text_shape = null;
         AffineTransform text_at = null;

         double max_text_width = 0.0;
         double max_text_height = 0.0;

         int tic_index = 0;
         for (Double major_tic: major_tics)
            {
            TextLayout layout = layouts.get(tic_index);
            Rectangle2D bounds = layout.getBounds();

            double text_width = bounds.getWidth();
            double text_height = bounds.getHeight();
            if (max_text_width < text_width)
               {
               max_text_width = text_width;
               }
            if (max_text_height < text_height)
               {
               max_text_height = text_height;
               }
            tic_index++;
            }

         tic_index = 0;
         for (Double major_tic: major_tics)
            {
            TextLayout layout = layouts.get(tic_index);
            Rectangle2D bounds = layout.getBounds();

            /*
            *  These are the offsets for the blc of the text.
            */
            double blc_x = 0.0;
            double blc_y = 0.0;

            Double tic_value = major_tic;
            switch (location)
               {
               case TOP:
                  tic_value -= this.h_values_min;
                  blc_x = this.plot_box.x
                        + tic_value * this.h_values_scale
                        + 0.5 * max_text_height;
                  blc_y = plot_box.y
                        - this.major_tic_length
                        - this.v_margin;
                  text_at = new AffineTransform();
                  text_at.setToIdentity();
                  text_at.translate(blc_x, blc_y);
                  text_at.rotate(Math.toRadians(-90));
                  text_shape = layout.getOutline(text_at);

                  graphics_2d.setColor(Color.BLACK);
                  graphics_2d.fill(text_shape);
                  break;
               case BOTTOM:
                  tic_value -= this.h_values_min;
                  blc_x = this.plot_box.x
                        + tic_value * this.h_values_scale
                        + 0.5 * max_text_height;
                  blc_y = plot_box.y
                        + plot_box.height
                        + this.major_tic_length
                        + this.v_margin
                        + max_text_width;
                  text_at = new AffineTransform();
                  text_at.setToIdentity();
                  text_at.translate(blc_x, blc_y);
                  text_at.rotate(Math.toRadians(-90));
                  text_shape = layout.getOutline(text_at);

                  graphics_2d.setColor(Color.BLACK);
                  graphics_2d.fill(text_shape);
                  break;
               case LEFT:
                  tic_value -= this.v_values_min;
                  blc_x = plot_box.x
                        - this.major_tic_length
                        - this.h_margin
                        - max_text_width;
                  blc_y = this.plot_box.y + this.plot_box.height
                        - tic_value * this.v_values_scale
                        + 0.5 * max_text_height;
                  layout.draw(graphics_2d, (float) blc_x, (float) blc_y);
                  break;
               case RIGHT:
                  tic_value -= this.v_values_min;
                  blc_x = plot_box.x
                        + plot_box.width
                        + this.major_tic_length
                        + this.h_margin;
                  blc_y = this.plot_box.y + this.plot_box.height
                        - tic_value * this.v_values_scale
                        + 0.5 * max_text_height;
                  layout.draw(graphics_2d, (float) blc_x, (float) blc_y);
                  break;
               }

            tic_index++;
            }
         }
      }


   /**
   *  Draw the tics in the correct positions.
   *  @param graphics_2d the graphics to draw to.
   *  @param location the location for the tics --
   *     TOP -- horizontal above the box.
   *     BOTTOM -- horizontal below the box.
   *     LEFT -- vertical on the left of the box.
   *     RIGHT -- vertical on the right of the box.
   *  @param include_minor_tics if true include the minor tics.
   *  @param major_tics the locations for the major tics along the axis.
   *  @param minor_tics the locations for the minor tics along the axis.
   */
   public void drawAxisTics(Graphics2D graphics_2d,
                            Location location,
                            boolean include_minor_tics,
                            ArrayList<Double> major_tics,
                            ArrayList<Double> minor_tics)
      {
      if (major_tics.size() > 0)
         {
         graphics_2d.setStroke(new BasicStroke(1.0F));
         graphics_2d.setColor(Color.BLACK);

         Point2D start_pos = null;
         Point2D end_pos = null;

         for (Double major_tic: major_tics)
            {
            Double tic_value = major_tic;
            Double tic_pos = null;

            switch (location)
               {
               case TOP:
                  tic_value -= this.h_values_min;
                  tic_pos = this.plot_box.x + tic_value * this.h_values_scale;
                  start_pos = new Point2D.Double(tic_pos, plot_box.y);
                  end_pos = new Point2D.Double(tic_pos,
                                               plot_box.y
                                             - this.major_tic_length);
                  break;
               case BOTTOM:
                  tic_value -= this.h_values_min;
                  tic_pos = this.plot_box.x + tic_value * this.h_values_scale;
                  start_pos = new Point2D.Double(tic_pos,
                                                 plot_box.y
                                               + plot_box.height);
                  end_pos = new Point2D.Double(tic_pos,
                                               plot_box.y
                                             + plot_box.height
                                             + this.major_tic_length);
                  break;
               case LEFT:
                  tic_value -= this.v_values_min;
                  tic_pos = this.plot_box.y
                          + tic_value * this.v_values_scale;
                  start_pos = new Point2D.Double(plot_box.x, tic_pos);
                  end_pos = new Point2D.Double(plot_box.x
                                             - this.major_tic_length,
                                               tic_pos);
                  break;
               case RIGHT:
                  tic_value -= this.v_values_min;
                  tic_pos = this.plot_box.y
                          + tic_value * this.v_values_scale;
                  start_pos = new Point2D.Double(plot_box.x
                                               + plot_box.height,
                                                 tic_pos);
                  end_pos = new Point2D.Double(plot_box.x
                                             + plot_box.width
                                             + this.major_tic_length,
                                               tic_pos);
                  break;
               }

            /*
            *  Draw a major tic line (5 long) on the plot axis.
            */
            Line2D line = new Line2D.Double(start_pos, end_pos);
            graphics_2d.draw(line);
            }

        if (include_minor_tics &&
            minor_tics.size() > 0)
            {
            for (Double minor_tic: minor_tics)
               {
               Double tic_value = minor_tic;
               Double tic_pos = null;

               switch (location)
                  {
                  case TOP:
                     tic_value -= this.h_values_min;
                     tic_pos = this.plot_box.x
                             + tic_value * this.h_values_scale;
                     start_pos = new Point2D.Double(tic_pos, plot_box.y);
                     end_pos = new Point2D.Double(tic_pos,
                                                  plot_box.y
                                                - minor_tic_length);
                     break;
                  case BOTTOM:
                     tic_value -= this.h_values_min;
                     tic_pos = this.plot_box.x
                             + tic_value * this.h_values_scale;
                     start_pos = new Point2D.Double(tic_pos,
                                                    plot_box.y
                                                  + plot_box.height);
                     end_pos = new Point2D.Double(tic_pos,
                                                  plot_box.y
                                                + plot_box.height
                                                + minor_tic_length);
                     break;
                  case LEFT:
                     tic_value -= this.v_values_min;
                     tic_pos = this.plot_box.y
                             + tic_value * this.v_values_scale;
                     start_pos = new Point2D.Double(plot_box.x, tic_pos);
                     end_pos = new Point2D.Double(plot_box.x
                                                - minor_tic_length,
                                                  tic_pos);
                     break;
                  case RIGHT:
                     tic_value -= this.v_values_min;
                     tic_pos = this.plot_box.y
                             + tic_value * this.v_values_scale;
                     start_pos = new Point2D.Double(plot_box.x
                                                  + plot_box.width,
                                                    tic_pos);
                     end_pos = new Point2D.Double(plot_box.x
                                                + plot_box.width
                                                + minor_tic_length,
                                                  tic_pos);
                     break;
                  }

               /*
               *  Draw a minor tic line (2 long) on the plot axis.
               */
               Line2D line = new Line2D.Double(start_pos, end_pos);
               graphics_2d.draw(line);
               }
            }
         }
      }


   /**
   *  Draw the grid lines in the correct positions.
   *  @param graphics_2d the graphics to draw to.
   *  @param location the location for the tics --
   *     TOP -- horizontal above the box.
   *     BOTTOM -- horizontal below the box.
   *     LEFT -- vertical on the left of the box.
   *     RIGHT -- vertical on the right of the box.
   *  @param major_tics the locations for the major tics along the axis.
   */
   public void drawAxisGrid(Graphics2D graphics_2d,
                            Location location,
                            ArrayList<Double> major_tics)
      {
      if (major_tics.size() > 0)
         {
         graphics_2d.setStroke(new BasicStroke(1.0F));
         graphics_2d.setColor(Color.LIGHT_GRAY);

         Point2D start_pos = null;
         Point2D end_pos = null;

         for (Double major_tic: major_tics)
            {
            Double tic_value = major_tic;
            Double tic_pos = null;

            switch (location)
               {
               case TOP:
                  tic_value -= this.h_values_min;
                  tic_pos = this.plot_box.x + tic_value * this.h_values_scale;
                  start_pos = new Point2D.Double(tic_pos, plot_box.y);
                  end_pos = new Point2D.Double(tic_pos,
                                               plot_box.y
                                             + plot_box.height);
                  break;
               case BOTTOM:
                  tic_value -= this.h_values_min;
                  tic_pos = this.plot_box.x + tic_value * this.h_values_scale;
                  start_pos = new Point2D.Double(tic_pos, plot_box.y);
                  end_pos = new Point2D.Double(tic_pos,
                                               plot_box.y
                                             + plot_box.height);
                  break;
               case LEFT:
                  tic_value -= this.v_values_min;
                  tic_pos = this.plot_box.y
                          + tic_value * this.v_values_scale;
                  start_pos = new Point2D.Double(plot_box.x, tic_pos);
                  end_pos = new Point2D.Double(plot_box.x
                                             + plot_box.width,
                                               tic_pos);
                  break;
               case RIGHT:
                  tic_value -= this.v_values_min;
                  tic_pos = this.plot_box.y
                          + tic_value * this.v_values_scale;
                  start_pos = new Point2D.Double(plot_box.x, tic_pos);
                  end_pos = new Point2D.Double(plot_box.x
                                             + plot_box.width,
                                               tic_pos);
                  break;
               }

            /*
            *  Draw a grid line across the plot area.
            */
            Line2D line = new Line2D.Double(start_pos, end_pos);
            graphics_2d.draw(line);
            }
         }
      }
   }

