
/*
*  The Panel for displaying a Plot2D.
*/

package plot;


import java.awt.*;
import java.awt.print.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.awt.font.*;
import java.awt.image.BufferedImage;

import java.util.*;
import java.io.*;

import javax.print.attribute.*;
import javax.imageio.*;

import javax.swing.*;
import javax.swing.event.*;

import util.*;
import util.geom.*;
import plot.*;


/**
*  This class is a panel for displaying the plot data.  It can be 2-D plots
*  3-D 3-view (2-D) plots, 3-D rendering plot, or the 3-D rendering and 3-view
*  plots.
*/
public class PlotPanel extends JPanel implements Printable
   {
   static final long serialVersionUID = 1L;

   private PlotFrame plot_frame = null;
   private PlotPanel panel = null;

   private boolean markers_submenu = false;
   private PlotItem curr_plot_item = null;
   private JCheckBox curr_marker_cb = null;


   private JPopupMenu popup_menu = null;
   private JMenu menu = null;


   /**
   *  File access information for the plot pictures.
   */
   String plot_file_path = null;
   String plot_file_name = null;

   /*
   *  Overall plot data.
   */
   private String title = null;

   double x_border = 10.0;
   double y_border = 10.0;
   double z_border = 10.0;

   private double last_width = 0.0;
   private double last_height = 0.0;

   Color background_color = Color.white;
   Color foreground_color = Color.black;

   private Font font_title = new Font("Serif", Font.BOLD, 18);
   private Font font_axis_title = new Font("Serif", Font.BOLD, 14);
   private Font font = new Font("Serif", Font.BOLD, 10);

   /*
   *  The plot area box and within the current window.
   */
   Rectangle2D.Double outline_box = null;
   Rectangle2D.Double image_box = null;
   Rectangle2D.Double plot_box = null;

   Rectangle2D.Double selection_box = null;

   Color key_box_background_color = null;

   /*
   *  The plot 2D and plot3D key to the plotted lines.
   */
   boolean key_dragging = false;
   double key_box_x_offset = 0.0;
   double key_box_y_offset = 0.0;

   Rectangle2D.Double key_box =  null;
   BufferedImage key_box_image =  null;

   /*
   *  The objects, and the points associated with each object for a 2D plot.
   */
   private Plot2D plot_2d = null;

   /*
   *  The objects, and the points associated with each object for a 3D plot.
   */
   private Plot3D plot_3d = null;

   /*
   *  This plots blocks in color along the rows of the plot area as in
   * spectral plots.
   */
   private ScrolledPlot scrolled_plot = null;


   private Axis axis_x = null;
   private Axis axis_y = null;
   private Axis axis_z = null;
   private PlotAxes plot_axes = null;

   private BufferedImage plot_image = null;


   /**
   *  These are the flags for enablind and disabling the various plot
   *  labeling and display options.
   */
   boolean display_h_grid = true;
   boolean display_h_bottom = true;

   boolean display_top_label = false;
   boolean display_top_tic_labels = false;
   boolean display_top_major_tics = false;
   boolean display_top_minor_tics = false;

   boolean display_bottom_label = true;
   boolean display_bottom_tic_labels = true;
   boolean display_bottom_major_tics = true;
   boolean display_bottom_minor_tics = true;


   boolean display_v_grid = true;
   boolean display_v_left = true;

   boolean display_left_label = true;
   boolean display_left_tic_labels = true;
   boolean display_left_major_tics = true;
   boolean display_left_minor_tics = true;

   boolean display_right_label = false;
   boolean display_right_tic_labels = false;
   boolean display_right_major_tics = false;
   boolean display_right_minor_tics = false;

   boolean display_key_box = true;

   /*
   *  These are only associated with the ScrolledPlot form.
   */
   private PlotFrame column_lines_plot = null;
   private PlotFrame row_lines_plot = null;

   double col_width = 0.0;
   double col_height = 0.0;



   private Point2D initial_mouse_position = null;
   private Point2D last_mouse_position = null;
   private Point2D position_offset = null;
   private double old_x_pos;

   private double last_aspect_ratio = 0.0;


   private MarkerActionListener marker_action_listener =
                                         new MarkerActionListener();
   private PopupActionListener action_listener = new PopupActionListener();
   private PopupMenuListener menu_listener = new PopupMenuListener();

   /**
   *  Create the panel for a 2-D plot and connect the event handlers to
   *  the mouse events.
   *  @param plot_frame the parent frame for this panel.
   *  @param plot the 2-D plot to be displayed in the panel.
   */
   public PlotPanel(PlotFrame plot_frame, Plot2D plot)
      {
/*
      System.out.println("PlotPanel::Constructor(PlotFrame, Plot2D)"
                       + " on ENTRY.");
*/

      this.plot_frame = plot_frame;
      this.plot_2d = plot;

      this.title = this.plot_2d.title;
      this.axis_x = plot_2d.axis_x;
      this.axis_y = plot_2d.axis_y;

      this.plot_axes = new PlotAxes(this.axis_x,
                                    this.x_border,
                                    this.axis_y,
                                    this.y_border);
      this.plot_axes.setHorizontalOptions(this.display_top_label,
                                          this.display_bottom_label,
                                          this.display_top_tic_labels,
                                          this.display_bottom_tic_labels,
                                          this.display_top_major_tics,
                                          this.display_bottom_major_tics,
                                          this.display_top_minor_tics,
                                          this.display_bottom_minor_tics,
                                          this.display_h_grid);
      this.plot_axes.setVerticalOptions(this.display_left_label,
                                        this.display_right_label,
                                        this.display_left_tic_labels,
                                        this.display_right_tic_labels,
                                        this.display_left_major_tics,
                                        this.display_right_major_tics,
                                        this.display_left_minor_tics,
                                        this.display_right_minor_tics,
                                        this.display_v_grid);

      setupPlotPanel();
      }

   /**
   *  Create the panel for a 3-D plot and connect the event handlers to the
   *  mouse events.
   *  @param plot_frame the parent frame for this panel.
   *  @param plot the 3-D plot to be displayed in the panel.
   */
   public PlotPanel(PlotFrame plot_frame, Plot3D plot)
      {
/*
      System.out.println("PlotPanel::Constructor(PlotFrame, Plot3D)"
                       + " on ENTRY.");
*/

      this.plot_frame = plot_frame;
      this.plot_3d = plot;

      this.title = this.plot_3d.title;
      this.axis_x = plot_3d.axis_x;
      this.axis_y = plot_3d.axis_y;
      this.axis_z = plot_3d.axis_z;

      this.plot_axes = new PlotAxes(this.axis_x,
                                    this.x_border,
                                    this.axis_y,
                                    this.y_border);
      this.plot_axes.setHorizontalOptions(this.display_top_label,
                                          this.display_bottom_label,
                                          this.display_top_tic_labels,
                                          this.display_bottom_tic_labels,
                                          this.display_top_major_tics,
                                          this.display_bottom_major_tics,
                                          this.display_top_minor_tics,
                                          this.display_bottom_minor_tics,
                                          this.display_h_grid);
      this.plot_axes.setVerticalOptions(this.display_left_label,
                                        this.display_right_label,
                                        this.display_left_tic_labels,
                                        this.display_right_tic_labels,
                                        this.display_left_major_tics,
                                        this.display_right_major_tics,
                                        this.display_left_minor_tics,
                                        this.display_right_minor_tics,
                                        this.display_v_grid);

      setupPlotPanel();
      }


   /**
   *  Create the panel and connect the event handlers to the mouse events.
   *  @param plot_frame the parent frame for this panel.
   *  @param scrolled_plot the plot to be displayed in the panel.
   */
   public PlotPanel(PlotFrame plot_frame, ScrolledPlot scrolled_plot)
      {
/*
      System.out.println("PlotPanel::Constructor(PlotFrame, ScrolledPlot)"
                       + " on ENTRY:"
                        );
      StackTraceElement [] trace = Thread.currentThread().getStackTrace();
      for (StackTraceElement elm: trace)
         {
         System.out.println(elm);
         }
*/

      this.plot_frame = plot_frame;
      this.scrolled_plot = scrolled_plot;

      this.title = this.scrolled_plot.title;
      this.axis_x = this.scrolled_plot.axis_x;
      this.axis_y = this.scrolled_plot.axis_y;

      this.plot_axes = new PlotAxes(this.axis_x,
                                    this.x_border,
                                    this.axis_y,
                                    this.y_border);
      this.plot_axes.setHorizontalOptions(this.display_top_label,
                                          this.display_bottom_label,
                                          this.display_top_tic_labels,
                                          this.display_bottom_tic_labels,
                                          this.display_top_major_tics,
                                          this.display_bottom_major_tics,
                                          this.display_top_minor_tics,
                                          this.display_bottom_minor_tics,
                                          this.display_h_grid);
      this.plot_axes.setVerticalOptions(this.display_left_label,
                                        this.display_right_label,
                                        this.display_left_tic_labels,
                                        this.display_right_tic_labels,
                                        this.display_left_major_tics,
                                        this.display_right_major_tics,
                                        this.display_left_minor_tics,
                                        this.display_right_minor_tics,
                                        this.display_v_grid);

      setupPlotPanel();
      }


   /**
   *  Set up the panel and connect the popup menu handlers and
   *  the handlers for the mouse events.
   */
   public void setupPlotPanel()
      {
      /*
      *  Since these are not available from the panel until AFTER its
      *  parent in created these are saved here.
      */
      this.panel = this;

      popup_menu = new JPopupMenu();
      buildPopupMenu(popup_menu);

      JMenu [] menus = 
         {
         new JMenu("Options")
         };
/*
      menu = new JPopupMenu();
*/
      menu = menus[0];
      buildPopupMenu(menu.getPopupMenu());

      /*
      *  Build the menu bar with the above menus.
      */
      JMenuBar menu_bar = new JMenuBar();
      for (int menu_index = 0;
           menu_index < menus.length;
           menu_index++)
         {
         menu_bar.add(menus[menu_index]);
         }

      if (this.plot_frame != null)
         {
         this.plot_frame.setJMenuBar(menu_bar);
         }

      addMouseListener(new PlotMouseAdapter());
      addMouseMotionListener(new PlotMouseMotionAdapter());

/*
      System.out.println("PlotPanel::Constructor on EXIT.");
*/
      }

   /**
   *  Build the popupmenu for the plot panel display.
   */
   private void buildPopupMenu(JPopupMenu menu)
      {
      /*
      *  Build the popup menu for operations on the plot.
      */
      JMenu sm = null;    /* A submenu. */
      JMenuItem m = null; /* A menu item. */
      JMenu ssm = null;    /* A subsubmenu. */
      JCheckBox cb = null;    /* A checkbox. */

      m = new JMenuItem("Print This Plot");
      m.addActionListener(action_listener);
      menu.add(m);

      m = new JMenuItem("Save This Plot");
      m.addActionListener(action_listener);
      menu.add(m);

      menu.addSeparator();

      cb = new JCheckBox("Show Key ");
      cb.addActionListener(action_listener);
      cb.setSelected(this.display_key_box);
      menu.add(cb);

      menu.addSeparator();

      sm = new JMenu("Item Plots");
      sm.addMenuListener(menu_listener);
      menu.add(sm);

      sm = new JMenu("Markers");
      sm.addMenuListener(menu_listener);
      menu.add(sm);

      menu.addSeparator();

      sm = new JMenu("Plot Flags");
      menu.add(sm);

      ssm = new JMenu("X-Axis");
      ssm.addMenuListener(menu_listener);
      sm.add(ssm);

      cb = new JCheckBox("X Tics");
      cb.addActionListener(action_listener);
      cb.setSelected(this.axis_x.getDisplayTics());
      ssm.add(cb);

      cb = new JCheckBox("X Grid");
      cb.addActionListener(action_listener);
      cb.setSelected(this.axis_x.getDisplayGrid());
      ssm.add(cb);

      cb = new JCheckBox("X Minor Tics");
      cb.addActionListener(action_listener);
      cb.setSelected(this.axis_x.getDisplayMinorTics());
      ssm.add(cb);

      cb = new JCheckBox("X Log");
      cb.setEnabled(false);
      cb.addActionListener(action_listener);
      cb.setSelected(this.axis_x.isLog());
      ssm.add(cb);

      ssm = new JMenu("Y-Axis");
      ssm.addMenuListener(menu_listener);
      sm.add(ssm);

      cb = new JCheckBox("Y Tics");
      cb.addActionListener(action_listener);
      cb.setSelected(this.axis_y.getDisplayTics());
      ssm.add(cb);

      cb = new JCheckBox("Y Grid");
      cb.addActionListener(action_listener);
      cb.setSelected(this.axis_y.getDisplayGrid());
      ssm.add(cb);

      cb = new JCheckBox("Y Minor Tics");
      cb.addActionListener(action_listener);
      cb.setSelected(this.axis_y.getDisplayMinorTics());
      ssm.add(cb);

      cb = new JCheckBox("Y Log");
      cb.setEnabled(false);
      cb.addActionListener(action_listener);
      cb.setSelected(this.axis_y.isLog());
      ssm.add(cb);
      }


   /**
   *  Mouse button press interaction controller.
   */
   class PlotMouseAdapter extends MouseAdapter
      {
      public void mousePressed(MouseEvent event)
         {
         int button = event.getButton();
         Point pt = event.getPoint();

         String new_text = null;
/*
         System.out.println("PlotPanel::mousePressed()\n"
                          + "   loc = " + pt + "\n"
                          + "   button = " + button);
*/

         if (button == MouseEvent.BUTTON1)
            {
            Component comp = event.getComponent();
            String comp_type = comp.getClass().
                               getSuperclass().
                               getSimpleName();

            double x_pos = pt.getX();
            double y_pos = pt.getY();

            /*
            *  If it is inside the key box set up for relocating the key box.
            *  DUMMIED key box drag for now.
            */
            if (display_key_box &&
                key_box != null &&
                key_box.contains(x_pos, y_pos))
               {
/*
               System.out.println("PlotPanel.mousePressed() BUTTON1:\n"
                                + "button press in the key_box.");
*/
               key_box_x_offset = x_pos - key_box.x;
               key_box_y_offset = y_pos - key_box.y;
               key_dragging = true;
               }
/*
            new_text = "Button1Pressed event: comp_type = \""
                     + comp_type
                     + "\"";
            plot_gui.setTextField(new_text);
            System.out.println(new_text);
*/
            }
         else if (button == MouseEvent.BUTTON2)
            {
            Component comp = event.getComponent();
            String comp_type = comp.getClass().
                               getSuperclass().
                               getSimpleName();
/*
            new_text = "Button2Pressed event: comp_type = \""
                     + comp_type
                     + "\"";
            plot_gui.setTextField(new_text);
            System.out.println(new_text);
*/
            }
         else if (button == MouseEvent.BUTTON3)
            {
/*
            if (event.isPopupTrigger())
*/
               {
               /*
               *  This only works on the component that is a JPanel.
               */
               Component comp = event.getComponent();
               String comp_type = comp.getClass().
                                  getSuperclass().
                                  getSimpleName();
/*
               new_text = "Button3Pressed event: comp_type = \""
                        + comp_type
                        + "\"";
               plot_gui.setTextField(new_text);
               System.out.println(new_text);
*/

               if (comp_type.equals("JPanel"))
                  {
                  popup_menu.show((JPanel) comp,
                                  event.getX(),
                                  event.getY());
                  }
               else if (comp_type.equals("JApplet"))
                  {
                  popup_menu.show((JApplet) comp,
                                  event.getX(),
                                  event.getY());
                  }
               }
            }
         }

      public void mouseReleased(MouseEvent event)
         {
         int button = event.getButton();
         Point pt = event.getPoint();
/*
         System.out.println("PlotPanel::mouseReleased()\n"
                          + "   loc = " + pt + "\n"
                          + "   button = " + button);
*/

         if (button == MouseEvent.BUTTON1)
            {
            Component comp = event.getComponent();
            String comp_type = comp.getClass().
                               getSuperclass().
                               getSimpleName();

            double x_pos = pt.getX();
            double y_pos = pt.getY();

            /*
            *  If the key box is being dragged drop it here!
            *  DUMMIED key box drag for now.
            */
            if (display_key_box &&
                key_dragging &&
                key_box != null)
               {
/*
               System.out.println("PlotPanel.mouseReleased() BUTTON1:\n"
                                + "button released dropping the key_box.");
*/
               key_box.x = x_pos - key_box_x_offset;
               key_box.y = y_pos - key_box_y_offset;
               key_dragging = false;
               repaint();
               }

/*
            String new_text = "Button1Released event: comp_type = \""
                            + comp_type
                            + "\"";
            plot_gui.setTextField(new_text);
            System.out.println(new_text);
*/
            }
         else if (button == MouseEvent.BUTTON2)
            {
            Component comp = event.getComponent();
            String comp_type = comp.getClass().
                               getSuperclass().
                               getSimpleName();
/*
            String new_text = "Button2Released event: comp_type = \""
                            + comp_type
                            + "\"";
            plot_gui.setTextField(new_text);
            System.out.println(new_text);
*/
            }
         else if (button == MouseEvent.BUTTON3)
            {
            Component comp = event.getComponent();
            String comp_type = comp.getClass().
                               getSuperclass().
                               getSimpleName();
/*
            String new_text = "Button3Released event: comp_type = \""
                            + comp_type
                            + "\"";
            plot_gui.setTextField(new_text);
            System.out.println(new_text);
*/
            }
         }
      }

   /**
   *  Mouse motion interaction controller.
   */
   class PlotMouseMotionAdapter extends MouseMotionAdapter
      {
      public void mouseDragged(MouseEvent event)
         {
         int button = event.getButton();
         Point pt = event.getPoint();

         double x_pos = pt.getX();
         double y_pos = pt.getY();

         /*
         *  If the key box is being dragged reposition it and redisplay,
         *  otherwise just draw a box.
         *  DUMMIED key box drag for now.
         */
         if (display_key_box &&
             key_dragging &&
             key_box != null)
            {
/*
            System.out.println("PlotPanel.mouseDragged() BUTTON1:\n"
                             + "mouse moved dragging the key_box.");
*/
            key_box.x = x_pos - key_box_x_offset;
            key_box.y = y_pos - key_box_y_offset;
            repaint();
            }

/*
         System.out.println("PlotPanel::mouseDragged()\n"
                          + "   loc = " + pt + "\n"
                          + "   button = " + button);
*/


         if (x_pos < outline_box.x)
            {
            x_pos = outline_box.x;
            }
         else if (x_pos > outline_box.x + outline_box.width)
            {
            x_pos = outline_box.x + outline_box.width;
            }

         double x_offset = Math.abs(old_x_pos - x_pos);
         int cell_offset = (int) Math.floor(x_offset);

         if (cell_offset > 0)
            {
            /*
            *  Get the current box.
            */
            selection_box = new Rectangle2D.Double(outline_box.y,
                                                   outline_box.x,
                                                   outline_box.width,
                                                   outline_box.height);
            last_mouse_position = pt;
            old_x_pos = x_pos;
            }
         }
      }


   /**
   *  The markers menu action listener for the image.
   */
   class MarkerActionListener implements ActionListener
      {

      public void actionPerformed(ActionEvent event)
         {
         String selected_text = ((AbstractButton) event.getSource()).getText();
/*
         System.out.println("MarkerActionListener::actionPerformed()"
                          + " STATUS:\n"
                          + "   selected_text = \""
                          + selected_text
                          + "\n"
                          + "   curr_plot_item = "
                          + (curr_plot_item == null?
                             "null":
                             "\""
                             + curr_plot_item.getTitle()
                             + "\""
                            )
                          + "\n"
                           );
*/
         ArrayList<String> marker_types = Plot2D.getMarkerTypes();

         boolean found = false;

         if (plot_2d != null &&
             curr_plot_item != null)
            {
            for (String marker_type: marker_types)
               {
               if (selected_text.equals(marker_type))
                  {
                  JCheckBox cb = (JCheckBox) event.getSource();

/*
                  System.out.println("MarkerActionListener::actionPerformed()"
                                   + " STATUS:"
                                   + "   curr_plot_item = \""
                                   + curr_plot_item.getTitle()
                                   + "\n"
                                   + "   marker_type\" = \""
                                   + marker_type
                                   + "\""
                                    );
*/
                  if (cb.isSelected())
                     {
                     if (curr_marker_cb != null)
                        {
                        curr_marker_cb.setSelected(false);
                        }
                     curr_plot_item.setMarkerType(plot_2d, marker_type);
                     curr_marker_cb = cb;
                     }
                  else
                     {
                     if (curr_marker_cb != null)
                        {
                        curr_marker_cb.setSelected(false);
                        }
                     curr_plot_item.setMarkerType(plot_2d, null);
                     curr_marker_cb = null;
                     }
                  repaint();
                  }
               }
            }
         }

      public void menuCanceled(MenuEvent event)
         {
         }

      public void menuDeselected(MenuEvent event)
         {
         }
      }


   /**
   *  The popup menu action listener for the image.
   */
   class PopupActionListener implements ActionListener
      {

      public void actionPerformed(ActionEvent event)
         {
         String selected_text = ((AbstractButton) event.getSource()).getText();
         JCheckBox cb = null;

/*
         String new_text = "PopupAction event: selected_text = \""
                         + selected_text
                         + "\"";
         plot_gui.setTextField(new_text);
         System.out.println(new_text);
*/

         if (selected_text.equals("Print This Plot"))
            {
            /*
            *  Print the contents of the PlotPanel.
            */
            PrinterJob job = PrinterJob.getPrinterJob();
            job.setPrintable(panel);

            HashPrintRequestAttributeSet attributes = 
                                  new HashPrintRequestAttributeSet();
            if (job.printDialog(attributes))
               {
               try
                  {
                  job.print(attributes);
                  }
               catch (PrinterException except)
                  {
                  System.out.println("PlotPanel::Print ERROR:\n"
                                   + "   \"" + except + "\"");
                  except.printStackTrace(System.out);
                  }
               }
            }
         else if (selected_text.equals("Save This Plot"))
            {
            /*
            *  Save the contents of the PlotPanel to a file.
            */
            System.out.println("PlotPanel::Save ERROR:\n"
                             + "   \"" + "Operation not implemented" + "\"");
            JFileChooser chooser = null;
            if (plot_gui.dir_path != null)
               {
               chooser = new JFileChooser(plot_gui.dir_path);
               }
            else
               {
               chooser = new JFileChooser();
               }

            int rVal = chooser.showSaveDialog(PlotPanel.this);
            if (rVal == JFileChooser.APPROVE_OPTION)
               {
               plot_file_name = chooser.getSelectedFile().getName();
/*
               plot_gui.filename_text.setText(plot_file_name);
*/

               plot_gui.dir_path = chooser.getCurrentDirectory().toString();
/*
               plot_gui.dir_text.setText(plot_gui.dir_path);
*/

               plot_file_path = plot_gui.dir_path
                              + "/"
                              + plot_file_name
                               ;

               System.out.println("SaveAsL(): Outputting to:\n"
                                + "   plot_file_path = \""
                                + plot_file_path + "\"");

               PlotPanel.this.writePlotImage(plot_file_path);
               }

            if (rVal == JFileChooser.CANCEL_OPTION)
               {
/*
               plot_gui.filename_text.setText("SaveAsL(): You pressed CANCEL.");
               plot_gui.dir_text.setText("");
*/
               }
            }
         else if (selected_text.equals("Show Key "))
            {
            cb = (JCheckBox) event.getSource();
/*
            System.out.println("PopupActionListener::actionPerformed()"
                             + " STATUS:\n"
                             + "   \"Show Key \" toggled to ("
                             + cb.isSelected()
                             + ")");
*/

            display_key_box = cb.isSelected();
            repaint();
            }
         else if (selected_text.equals("X Tics"))
            {
            cb = (JCheckBox) event.getSource();
/*
            System.out.println("PopupActionListener::actionPerformed()"
                             + " STATUS:\n"
                             + "   \"X Tics\" toggled to ("
                             + cb.isSelected()
                             + ")");
*/

            axis_x.setDisplayTics(cb.isSelected());
            if (display_h_bottom)
               {
               display_bottom_major_tics = !display_bottom_major_tics;
               }
            else
               {
               display_top_major_tics = !display_top_major_tics;
               }

            plot_axes.setHorizontalOptions(display_top_label,
                                           display_bottom_label,
                                           display_top_tic_labels,
                                           display_bottom_tic_labels,
                                           display_top_major_tics,
                                           display_bottom_major_tics,
                                           display_top_minor_tics,
                                           display_bottom_minor_tics,
                                           display_h_grid);
            repaint();
            }
         else if (selected_text.equals("X Grid"))
            {
            cb = (JCheckBox) event.getSource();
/*
            System.out.println("PopupActionListener::actionPerformed()"
                             + " STATUS:\n"
                             + "   \"X Grid\" toggled to ("
                             + cb.isSelected()
                             + ")");
*/

            axis_x.setDisplayGrid(cb.isSelected());

            display_h_grid = !display_h_grid;
            plot_axes.setHorizontalOptions(display_top_label,
                                           display_bottom_label,
                                           display_top_tic_labels,
                                           display_bottom_tic_labels,
                                           display_top_major_tics,
                                           display_bottom_major_tics,
                                           display_top_minor_tics,
                                           display_bottom_minor_tics,
                                           display_h_grid);
            repaint();
            }
         else if (selected_text.equals("X Minor Tics"))
            {
            cb = (JCheckBox) event.getSource();
/*
            System.out.println("PopupActionListener::actionPerformed()"
                             + " STATUS:\n"
                             + "   \"X MinorTics\" toggled to ("
                             + cb.isSelected()
                             + ")");
*/

            axis_x.setDisplayMinorTics(cb.isSelected());
            if (display_h_bottom)
               {
               display_bottom_minor_tics = !display_bottom_minor_tics;
               }
            else
               {
               display_top_minor_tics = !display_top_minor_tics;
               }
            plot_axes.setHorizontalOptions(display_top_label,
                                           display_bottom_label,
                                           display_top_tic_labels,
                                           display_bottom_tic_labels,
                                           display_top_major_tics,
                                           display_bottom_major_tics,
                                           display_top_minor_tics,
                                           display_bottom_minor_tics,
                                           display_h_grid);
            repaint();
            }
         else if (selected_text.equals("X Log"))
            {
            cb = (JCheckBox) event.getSource();
/*
            System.out.println("PopupActionListener::actionPerformed()"
                             + " STATUS:\n"
                             + "   \"X Log toggled to ("
                             + cb.isSelected()
                             + ")");
*/

            boolean axis_x_is_log = cb.isSelected();
            axis_x.setLog(axis_x_is_log);
            if (axis_x_is_log)
               {
               axis_x.setupLog();
               }
            else
               {
               axis_x.setupLinear();
               }
            plot_axes.updatePlotValueStats();
            repaint();
            }
         else if (selected_text.equals("Y Tics"))
            {
            cb = (JCheckBox) event.getSource();
/*
            System.out.println("PopupActionListener::actionPerformed()"
                             + " STATUS:\n"
                             + "   \"Y Tics toggled to ("
                             + cb.isSelected()
                             + ")");
*/

            axis_y.setDisplayTics(cb.isSelected());

            if (display_v_left)
               {
               display_left_major_tics = !display_left_major_tics;
               }
            else
               {
               display_right_major_tics = !display_right_major_tics;
               }
            plot_axes.setVerticalOptions(display_left_label,
                                         display_right_label,
                                         display_left_tic_labels,
                                         display_right_tic_labels,
                                         display_left_major_tics,
                                         display_right_major_tics,
                                         display_left_minor_tics,
                                         display_right_minor_tics,
                                         display_v_grid);
            repaint();
            }
         else if (selected_text.equals("Y Grid"))
            {
            cb = (JCheckBox) event.getSource();
/*
            System.out.println("PopupActionListener::actionPerformed()"
                             + " STATUS:\n"
                             + "   \"Y Grid toggled to ("
                             + cb.isSelected()
                             + ")");
*/

            axis_y.setDisplayGrid(cb.isSelected());

            display_v_grid = !display_v_grid;
            plot_axes.setVerticalOptions(display_left_label,
                                         display_right_label,
                                         display_left_tic_labels,
                                         display_right_tic_labels,
                                         display_left_major_tics,
                                         display_right_major_tics,
                                         display_left_minor_tics,
                                         display_right_minor_tics,
                                         display_v_grid);
            repaint();
            }
         else if (selected_text.equals("Y Minor Tics"))
            {
            cb = (JCheckBox) event.getSource();
/*
            System.out.println("PopupActionListener::actionPerformed()"
                             + " STATUS:\n"
                             + "   \"Y Minor Tics toggled to ("
                             + cb.isSelected()
                             + ")");
*/

            axis_y.setDisplayMinorTics(cb.isSelected());

            if (display_v_left)
               {
               display_left_minor_tics = !display_left_minor_tics;
               }
            else
               {
               display_right_minor_tics = !display_right_minor_tics;
               }
            plot_axes.setVerticalOptions(display_left_label,
                                         display_right_label,
                                         display_left_tic_labels,
                                         display_right_tic_labels,
                                         display_left_major_tics,
                                         display_right_major_tics,
                                         display_left_minor_tics,
                                         display_right_minor_tics,
                                         display_v_grid);
            repaint();
            }
         else if (selected_text.equals("Y Log"))
            {
            cb = (JCheckBox) event.getSource();
/*
            System.out.println("PopupActionListener::actionPerformed()"
                             + " STATUS:\n"
                             + "   \"Y Log toggled to ("
                             + cb.isSelected()
                             + ")");
*/

            boolean axis_y_is_log = cb.isSelected();
            axis_y.setLog(axis_y_is_log);
            if (axis_y_is_log)
               {
               axis_y.setupLog();
               }
            else
               {
               axis_y.setupLinear();
               }
            plot_axes.updatePlotValueStats();
            repaint();
            }
         else if (selected_text.equals("Plot Column Lines"))
            {
/*
            System.out.println("PopupActionListener::actionPerformed()"
                             + " STATUS:\n"
                             + "   \"Plot Column Lines\""
                              );
*/
            if (scrolled_plot != null)
               {
               if (column_lines_plot == null)
                  {
                  /*
                  *  Create it.
                  */
                  Plot2D plot = scrolled_plot.getColumnLinesPlot();
                  column_lines_plot = new PlotFrame(title + " Columns", plot);
                  }

               if (column_lines_plot != null)
                  {
                  /*
                  *  Make it visible.
                  */
                  column_lines_plot.setVisible(true);
                  }
               }
            }
         else if (selected_text.equals("Plot Row Lines"))
            {
/*
            System.out.println("PopupActionListener::actionPerformed()"
                             + " STATUS:\n"
                             + "   \"Plot Row Lines\""
                              );
*/
            if (scrolled_plot != null)
               {
               if (row_lines_plot == null)
                  {
                  /*
                  *  Create it.
                  */
                  Plot2D plot = scrolled_plot.getRowLinesPlot();
                  row_lines_plot = new PlotFrame(title + " Rows", plot);
                  }

               if (row_lines_plot != null)
                  {
                  /*
                  *  Make it visible.
                  */
                  row_lines_plot.setVisible(true);
                  }
               }
            }
         else
            {
/*
            System.out.println("PopupActionListener::actionPerformed()"
                             + " STATUS:\n"
                             + "   \"Plot Item\" toggled.");
*/

            /*
            *  It is not one of the hard-coded menu entry texts, so it must be
            *  a plot item title.
            */
            if (plot_2d != null)
               {
               for (PlotItem plot_item: plot_2d.getPlotItems())
                  {
                  if (selected_text.equals(plot_item.getTitle()))
                     {
                     cb = (JCheckBox) event.getSource();
/*
                     System.out.println("PopupActionListener::actionPerformed()"
                                      + " STATUS:\n"
                                      + "   \"Plot Item\" "
                                      + plot_item.getTitle()
                                      + " toggled to ("
                                      + cb.isSelected()
                                      + ")");
*/

                     plot_item.is_selected = cb.isSelected();
                     repaint();
                     }
                  }
               }
            }
         }
      }

   /**
   *  The popup menu listener for the image.
   */
   class PopupMenuListener implements MenuListener
      {
      public void menuSelected(MenuEvent event)
         {
         String selected_text = ((AbstractButton) event.getSource()).getText();

         JMenu sm = (JMenu) event.getSource();
         JCheckBox cb = null;
         JMenuItem mi = null;

/*
         String new_text = "PopupMenu event: selected_text = \""
                         + selected_text
                         + "\"";
         System.out.println(new_text);
*/

         if (selected_text.equals("Item Plots"))
            {
/*
            System.out.println("PopupMenuListener::menuSelected() Status:\n"
                             + "   \"Item Plots\" submenu activated.");
*/
            sm.removeAll();
            if (plot_2d != null)
               {
               for (PlotItem plot_item: plot_2d.getPlotItems())
                  {
                  cb = new JCheckBox(plot_item.getTitle());
                  cb.addActionListener(action_listener);
                  cb.setSelected(plot_item.isSelected());
                  sm.add(cb);
                  }
               }

            if (scrolled_plot != null)
               {
               mi = new JMenuItem("Plot Column Lines");
               mi.addActionListener(action_listener);
               sm.add(mi);

               mi = new JMenuItem("Plot Row Lines");
               mi.addActionListener(action_listener);
               sm.add(mi);
               }
            }
         else if (selected_text.equals("Markers"))
            {
            markers_submenu = true;
/*
            System.out.println("PopupMenuListener::menuSelected() Status:\n"
                             + "   \"Markers\" submenu activated.");
*/
            sm.removeAll();
            if (plot_2d != null)
               {
               for (PlotItem plot_item: plot_2d.getPlotItems())
                  {
                  JMenu ssm = new JMenu(plot_item.getTitle());
                  ssm.addMenuListener(menu_listener);
                  sm.add(ssm);
                  }
               }
            }
         else if (selected_text.equals("X-Axis"))
            {
/*
            System.out.println("PopupMenuListener::menuSelected() Status:\n"
                             + "   \"X-Axis\" submenu activated.");
*/
            sm.removeAll();

            cb = new JCheckBox("X Tics");
            cb.addActionListener(action_listener);
            cb.setSelected(axis_x.getDisplayTics());
            sm.add(cb);

            cb = new JCheckBox("X Grid");
            cb.addActionListener(action_listener);
            cb.setSelected(axis_x.getDisplayGrid());
            sm.add(cb);

            cb = new JCheckBox("X Minor Tics");
            cb.addActionListener(action_listener);
            cb.setSelected(axis_x.getDisplayMinorTics());
            sm.add(cb);

            cb = new JCheckBox("X Log");
            cb.addActionListener(action_listener);
            cb.setSelected(axis_x.isLog());
            sm.add(cb);
            }
         else if (selected_text.equals("Y-Axis"))
            {
/*
            System.out.println("PopupMenuListener::menuSelected() Status:\n"
                             + "   \"Y-Axis\" submenu activated.");
*/
            sm.removeAll();

            cb = new JCheckBox("Y Tics");
            cb.addActionListener(action_listener);
            cb.setSelected(axis_x.getDisplayTics());
            sm.add(cb);

            cb = new JCheckBox("Y Grid");
            cb.addActionListener(action_listener);
            cb.setSelected(axis_x.getDisplayGrid());
            sm.add(cb);

            cb = new JCheckBox("Y Minor Tics");
            cb.addActionListener(action_listener);
            cb.setSelected(axis_x.getDisplayMinorTics());
            sm.add(cb);

            cb = new JCheckBox("Y Log");
            cb.addActionListener(action_listener);
            cb.setSelected(axis_y.isLog());
            sm.add(cb);
            }
         else if (markers_submenu && plot_2d != null)
            {
            ArrayList<String> marker_types = Plot2D.getMarkerTypes();
            HashMap<String, Boolean> marker_available =
                                         plot_2d.getMarkerAvailabilities();
            for (PlotItem plot_item: plot_2d.getPlotItems())
               {
               if (selected_text.equals(plot_item.getTitle()))
                  {
                  curr_plot_item = plot_item;
                  sm.removeAll();
                  for (String marker_type: marker_types)
                     {
                     boolean is_available = marker_available.get(marker_type);
                     cb = new JCheckBox(marker_type);
                     cb.addActionListener(marker_action_listener);
                     cb.setSelected(!is_available);
                     sm.add(cb);
                     if (plot_item.marker_type != null &&
                         plot_item.marker_type.equals(marker_type))
                        {
                        cb.setSelected(true);
                        curr_marker_cb = cb;
                        }
                     else
                        {
                        cb.setEnabled(is_available);
                        }
                     }
                  }
               }
            repaint();
            }
         }

      public void menuCanceled(MenuEvent event)
         {
         JMenu sm = (JMenu) event.getSource();
         String selected_text = sm.getText();

         if (selected_text.equals("Markers"))
            {
            markers_submenu = false;
            }
         else if (markers_submenu && curr_plot_item != null)
            {
            if (selected_text.equals(curr_plot_item.getTitle()))
               {
               curr_plot_item = null;
               }
            curr_marker_cb = null;
            }
         }

      public void menuDeselected(MenuEvent event)
         {
         JMenu sm = (JMenu) event.getSource();
         String selected_text = sm.getText();

         if (selected_text.equals("Markers"))
            {
            markers_submenu = false;
            }
         }
      }

   /**
   *  Clear the current display (of obstructions).
   */
   public void clear()
      {
      repaint();
      }


   /**
   *  Set the Plot2D which this PlotPanel is to display.
   *  @param plot the Plot2D instance to display.
   */
   public void setPlot(Plot2D plot)
      {
/*
      System.out.println("PlotPanel::setPlot on ENTRY.");
*/

      this.plot_2d = plot;

      if (this.plot_2d != null)
         {
         /*
         *  Get the plot colors and links ready for viewing.
         */
         repaint();
         }
      }


   /**
   *  Set the Plot3D which this PlotPanel is to display.
   *  @param plot the Plot3D instance to display.
   */
   public void setPlot(Plot3D plot)
      {
/*
      System.out.println("PlotPanel::setPlot on ENTRY.");
*/

      this.plot_3d = plot;

      if (this.plot_3d != null)
         {
         /*
         *  Get the plot colors and links ready for viewing.
         */
         repaint();
         }
      }


   /**
   *  Set the ScrolledPlot which this PlotPanel is to display.
   *  @param scrolled_plot the Plot2D instance to display.
   */
   public void setPlot(ScrolledPlot scrolled_plot)
      {
/*
      System.out.println("PlotPanel::setPlot on ENTRY.");
*/

      this.scrolled_plot = scrolled_plot;

      if (this.scrolled_plot != null)
         {
         /*
         *  Get the plot colors and links ready for viewing.
         */
         repaint();
         }
      }


   /**
   *  Paint the components of the displayed graphic panel whenever the
   *  system requests it.  This may be done to the displayed panel, to
   *  a printer page, or to an image destined for a file.
   *  @param graphics the graphics area to be redrawn.
   */
   public void paintComponent(Graphics graphics)
      {
      super.paintComponent(graphics);

      int curr_width = this.getWidth();
      int curr_height = this.getHeight();

/*
      System.out.println("PlotPanel::paintComponent() on ENTRY:\n"
                       + "   curr size = ("
                       + curr_width
                       + ", "
                       + curr_height
                       + ")\n"
                        );
*/

      this.last_aspect_ratio = (double) curr_height / (double) curr_width;

      if (this.plot_2d != null || this.scrolled_plot != null)
         {
         this.drawContents(graphics, curr_width, curr_height);
         }
      }


   /**
   *  Draw the contents of the PlotPanel, including plot area, titles,
   *  grid, tic marks, and tic labels.
   *  @param graphics the graphics context for this rendering.
   *  @param curr_width the width of the available rendering area.
   *  @param curr_height the height of the available rendering area.
   */
   private void drawContents(Graphics graphics,
                             int curr_width,
                             int curr_height)
      {
      if (this.last_width != curr_width ||
          this.last_height != curr_height)
         {
         this.key_box = null;
         }
/*
      System.out.println("PlotPanel::drawContents() on ENTRY:\n"
                       + "   curr size = ("
                       + curr_width
                       + ", "
                       + curr_height
                       + ")\n"
                       + "   plot_2d is null = "
                       + (this.plot_2d == null)
                       + "\n"
                       + "   plot_3d is null = "
                       + (this.plot_3d == null)
                       + "\n"
                       + "   scrolled_plot is null = "
                       + (this.scrolled_plot == null)
                       + "\n"
                        );
*/
      Graphics2D graphics_2d = (Graphics2D) graphics;
      graphics_2d.setBackground(this.background_color);

      FontRenderContext font_render_context = 
                                          graphics_2d.getFontRenderContext();

      /*
      *  Get the actual plot area size by first adding in all the decorations
      *  and allowing space for them.
      *     Main title
      *     Horizontal and Vertical axis titles.
      *     Horizontal and Vertical tics and tic labels.
      */
      double x_size = curr_width - 2.0 * this.x_border;
      double y_size = curr_height - 2.0 * this.y_border;

      /*
      *  Frame the entire available drawing area in the panel.
      */
      this.outline_box = new Rectangle2D.Double(this.x_border,
                                                this.y_border,
                                                x_size,
                                                y_size);
      graphics_2d.setColor(Color.BLACK);
      graphics_2d.setStroke(new BasicStroke(1.0F));
      graphics_2d.draw(outline_box);

      /*
      *  Draw the plot title, centering it horizontally in the
      *  outline and above the plot box.
      */
      String header = "Plot Title";
      if (this.title != null)
         {
         header = this.title;
         }
/*
      System.out.println("");
      System.out.println("PlotPanel.drawContents(): plot title = " + header);
*/
      TextLayout title_layout = new TextLayout(header,
                                               font_title,
                                               font_render_context);

      Rectangle2D bounds = title_layout.getBounds();
      double title_width = bounds.getWidth();
      double title_height = bounds.getHeight();

      /*
      *  Set the corners of the plot region.  This will be adjusted to
      *  contain the plot image.
      *
      *  The bottom left will be the plot origin offset.
      *
      *  The offsets are used to place items outside the plot area,
      *  including tic and tic label clearances.
      */
      this.image_box = new Rectangle2D.Double(this.x_border,
                                              this.y_border,
                                              x_size - 2.0,
                                              y_size);
      /*
      *  The reference point for the text is its "baseline" (bottom,
      *  ignoring descenders) at the left end of the text.
      */
      float x_text_blc = (float) (this.image_box.x
                                  + 0.5 * (this.image_box.width - title_width));
      float y_text_blc = (float) (this.image_box.y + 2.0 * title_height);

      graphics_2d.setColor(Color.BLACK);
      title_layout.draw(graphics_2d, x_text_blc, y_text_blc);

      /*
      *  Adjust the plot box to exclude the plot title.
      */
      this.image_box.y += 2.4 * title_height;
      this.image_box.height -= 2.4 * title_height;

      this.plot_image = this.plot_axes.drawContents(this.image_box,
                                                    this.background_color);
      Rectangle2D.Double image_plot_box = this.plot_axes.getPlotBox();

      this.plot_box = new Rectangle2D.Double(this.image_box.x
                                           + image_plot_box.x,
                                             this.image_box.y
                                           + image_plot_box.y,
                                             image_plot_box.width,
                                             image_plot_box.height);

      AffineTransform image_at = new AffineTransform();
      image_at.setToIdentity();
      image_at.translate(this.image_box.x, this.image_box.y);
      graphics_2d.drawImage(this.plot_image, image_at, null);


      double x_plot_right = this.plot_box.x + this.plot_box.width;
      double y_plot_bottom = this.plot_box.y + this.plot_box.height;

      /*
      *  Scale the data range to the plot area size.
      */
      double x_data_min = this.plot_axes.getHorizontalValuesMin();
      double y_data_min = this.plot_axes.getVerticalValuesMin();

      double x_data_range = this.plot_axes.getHorizontalValuesRange();
      double y_data_range = this.plot_axes.getVerticalValuesRange();

      double x_data_scale = this.plot_axes.getHorizontalValuesScale();
      double y_data_scale = this.plot_axes.getVerticalValuesScale();


/*
      System.out.println("   (min, range, scale) = ("
                         + x_data_min
                         + ", "
                         + y_data_min
                         + "), ("
                         + x_data_range
                         + ", "
                         + y_data_range
                         + "), ("
                         + x_data_scale
                         + ", "
                         + y_data_scale
                         + ")"
                        );
*/
      /*
      *  Draw the graph lines content of the panel.
      */

      if (this.plot_2d != null)
         {
         /*
         *  These values must be transformed to the plot coordinates before use.
         */
         for (PlotItem plot_item: this.plot_2d.plot_items)
            {
            if (plot_item.is_selected)
               {
/*
 * Preparing to separate this block of code into another routine.
 *
               this.drawPlotItem(graphics_2d,
                                 plot_item,
                                 this.plot_box.x,  // X-origin offset
                                 y_plot_bottom,    // Y-origin offset
                                 x_scale,
                                 y_scale,
                                 axis_x_is_log,
                                 axis_y_is_log);
*/

               graphics_2d.setColor(plot_item.color);

               Double[] x_values = plot_item.x_values;
               Double[] y_values = plot_item.y_values;

               Point2D.Double prev_point = null;
               for (int point_index = 0;
                    point_index < plot_item.num_points;
                    point_index++)
                  {
                  double x_value = x_values[point_index];
                  double y_value = y_values[point_index];

                  boolean skip_point = false;
                  if (axis_x.isLog())
                     {
                     /*
                     * The value needs converted to a displayable log of
                     * the value.
                     */
                     if (x_value <= 0.0)
                        {
                        skip_point = true;
                        }
                     else
                        {
                        x_value = Math.log10(x_value);
                        }
                     }
                  if (axis_y.isLog())
                     {
                     /*
                     * The value needs converted to a displayable log of
                     * the value.
                     */
                     if (y_value <= 0.0)
                        {
                        skip_point = true;
                        }
                     else
                        {
                        y_value = Math.log10(y_value);
                        }
                     }

                  if (x_value < x_data_min
                      || x_value > x_data_min + x_data_range
                      || y_value < y_data_min
                      || y_value > y_data_min + y_data_range)
                     {
                     skip_point = true;
                     }

                  if (!skip_point)
                     {
                     /*
                     *  Scale the data point to the plot area range.
                     */
                     double x_plot_value = (x_value
                                          - x_data_min) * x_data_scale;
                     double y_plot_value = (y_value
                                          - y_data_min) * y_data_scale;

                     /*
                     *  Translate the point to the plot area in the panel.
                     */
                     double x_pos = this.plot_box.x + x_plot_value;
                     double y_pos = y_plot_bottom - y_plot_value;

                     Point2D.Double curr_point = new Point2D.Double(x_pos,
                                                                    y_pos);
                     if (prev_point != null)
                        {
                        graphics_2d.setColor(plot_item.color);
                        Line2D.Double line = new Line2D.Double(curr_point,
                                                               prev_point);
                        graphics_2d.draw(line);
                        }

                     if (plot_item.marker_type != null &&
                         plot_item.radius > 0)
                        {
                        drawMarker(graphics_2d,
                                   plot_item.marker_type,
                                   curr_point,
                                   plot_item.radius,
                                   plot_item.color);
                        }

                     prev_point = curr_point;
                     }
                  }
               }
            }

         if (this.display_key_box)
            {
            this.key_box_image =  drawKeyBoxImage(font_render_context);
            if (this.key_box == null)
               {
               this.key_box =
                        new Rectangle2D.Double(this.plot_box.x
                                             + this.plot_box.width
                                             - this.key_box_image.getWidth(),
                                               this.plot_box.y,
                                               this.key_box_image.getWidth(),
                                               this.key_box_image.getHeight());
               }

            image_at = new AffineTransform();
            image_at.setToIdentity();
            image_at.translate(this.key_box.x, this.key_box.y);
            graphics_2d.drawImage(this.key_box_image, image_at, null);
            }
         }
      else if (this.scrolled_plot != null)
         {
         paintScrolledPlot(graphics_2d);
         }

/*
      if (this.selection_box != null)
         {
         graphics_2d.setColor(Color.BLACK);
         graphics_2d.draw(selection_box);
         }
*/

      this.last_width = curr_width;
      this.last_height = curr_height;
      }



   /**
   *  Draw a marker at the current position.
   *  @param graphics_2d the graphic to draw on.
   *  @param marker_type the type of marker to use.
   *  @param marker_center the location for its center.
   *  @param marker_radius the extents around the center point.
   *  @param marker_color the color for the marker.
   */
   public void drawMarker(Graphics2D graphics_2d,
                          String marker_type,
                          Point2D marker_center,
                          double marker_radius,
                          Color marker_color)
      {
      Point2D point1 = null;
      Point2D point2 = null;
      Line2D.Double line = null;
      /*
      *  Draw a marker object for the point.
      */
      Vector2D x_extents = new Vector2D();
      x_extents.x = marker_center.getX() - marker_radius;
      x_extents.y = marker_center.getX() + marker_radius;

      Vector2D y_extents = new Vector2D();
      y_extents.x = marker_center.getY() - marker_radius;
      y_extents.y = marker_center.getY() + marker_radius;

      Point2D ulc = new Point2D.Double(x_extents.x,
                                       y_extents.x);
      Point2D urc = new Point2D.Double(x_extents.y,
                                       y_extents.x);
      Point2D llc = new Point2D.Double(x_extents.x,
                                       y_extents.y);
      Point2D lrc = new Point2D.Double(x_extents.y,
                                       y_extents.y);
      double width_height = 2.0 * marker_radius;

      graphics_2d.setStroke(new BasicStroke(1.0F));
      graphics_2d.setColor(marker_color);
/*
      System.out.println("drawMarker() STATUS: marker_type = \""
                       + marker_type
                       + "\"\n"
                       + "   marker_radius = " + marker_radius
                        );
*/
      if (marker_type.matches(".*triangle"))
         {
         /*
         * The y-values increase going up, opposite of the x-y axis
         * and their polar angle for the Vector2D.
         */
         Vector2D upper = new Vector2D(marker_radius, -90.0 *
                                                   Math.PI / 180.0);
         Vector2D ll = new Vector2D(marker_radius, (-90.0 + 120.0) *
                                                   Math.PI / 180.0);
         Vector2D lr = new Vector2D(marker_radius, (-90.0 + 240.0) *
                                                   Math.PI / 180.0);
         upper = upper.getRect().addTo(marker_center);
         ll = ll.getRect().addTo(marker_center);
         lr = lr.getRect().addTo(marker_center);

         Polygon poly = new Polygon();
         poly.addPoint((int) upper.x, (int) upper.y);
         poly.addPoint((int) ll.x, (int) ll.y);
         poly.addPoint((int) lr.x, (int) lr.y);

         if (marker_type.matches("fill.*"))
            {
            graphics_2d.fill(poly);
            }

         graphics_2d.draw(poly);
         }
      else if (marker_type.matches(".*delta"))
         {
         Vector2D lower = new Vector2D(marker_radius, 90.0 *
                                                   Math.PI / 180.0);
         Vector2D ul = new Vector2D(marker_radius, (90.0 + 120.0) *
                                                   Math.PI / 180.0);
         Vector2D ur = new Vector2D(marker_radius, (90.0 + 240.0) *
                                                   Math.PI / 180.0);

         lower = lower.getRect().addTo(marker_center);
         ul = ul.getRect().addTo(marker_center);
         ur = ur.getRect().addTo(marker_center);

         Polygon poly = new Polygon();
         poly.addPoint((int) lower.x, (int) lower.y);
         poly.addPoint((int) ul.x, (int) ul.y);
         poly.addPoint((int) ur.x, (int) ur.y);

         if (marker_type.matches("fill.*"))
            {
            graphics_2d.fill(poly);
            }

         graphics_2d.draw(poly);
         }
      else if (marker_type.matches(".*square"))
         {
         Rectangle2D square =
                       new Rectangle2D.Double(x_extents.x,
                                              y_extents.x,
                                              width_height,
                                              width_height);
         if (marker_type.matches("fill.*"))
            {
            graphics_2d.fill(square);
            }

         graphics_2d.draw(square);
         }
      else if (marker_type.matches("cross"))
         {
         point1 = new Point2D.Double(marker_center.getX(),
                                     y_extents.x);
         point2 = new Point2D.Double(marker_center.getX(),
                                     y_extents.y);
         line = new Line2D.Double(point1, point2);
         graphics_2d.draw(line);

         point1 = new Point2D.Double(x_extents.x,
                                     marker_center.getY());
         point2 = new Point2D.Double(x_extents.y,
                                     marker_center.getY());
         line = new Line2D.Double(point1, point2);
         graphics_2d.draw(line);
         }
      else if (marker_type.matches("ex"))
         {
         line = new Line2D.Double(ulc, lrc);
         graphics_2d.draw(line);

         line = new Line2D.Double(urc, llc);
         graphics_2d.draw(line);
         }
      else
         {
         /*
         *  The default actual marker is a circle.
         */
         Ellipse2D ell = new Ellipse2D.Double();
         ell.setFrameFromDiagonal(ulc, lrc);

         if (marker_type.matches("fill.*"))
            {
            graphics_2d.fill(ell);
            }

         graphics_2d.draw(ell);
         }
      }

   /**
   *  Paint the key box image.
   *  @param font_render_context font rendering details for the parent
   *     target image.
   */
   private BufferedImage drawKeyBoxImage(FontRenderContext font_render_context)
      {
      BufferedImage key_image = null;

      if (this.key_box_background_color == null)
         {
         /*
         *  Set it transparent. Later there may be an opaque option here.
         */
         this.key_box_background_color = new Color(0, 0, 0, 0);
         }

/*
      System.out.println("PlotPanel::drawKeyBoxImage() on ENTRY:\n"
                       + "   key_box_background_color = "
                       + key_box_background_color
                       + "\n"
                        );
*/
      ArrayList<TextLayout> key_layouts = new ArrayList<TextLayout>();
      ArrayList<Double> key_offsets = new ArrayList<Double>();

      double left_margin = 8.0;
      double line_to_text_margin = 8.0;
      double right_margin = 8.0;

      double top_margin = 2.0;
      double vertical_pad = 2.0;
      double bottom_margin = 3.0;

      double marker_radius = -1.0;

      if (this.plot_2d != null)
         {
         /*
         * Scan the plot items getting the extents needed to display each
         * entry, count the number of entries to display, and get the extents
         * for the markers to be included (if any).
         */
         int item_count = 0;

         /*
         * The key entry size max extents is calculated into these.
         */
         double max_text_width = 0.0;
         double max_entry_height = 0.0;

         for (PlotItem plot_item: this.plot_2d.plot_items)
            {
            if (plot_item.is_selected)
               {
               /*
               *  The font_render_context must be set before this routine can
               * be used to get the text sizes.
               */
               TextLayout key_layout = new TextLayout(plot_item.title,
                                                      font_axis_title,
                                                      font_render_context);
               key_layouts.add(key_layout);

               Rectangle2D bounds = key_layout.getBounds();
               double string_width = bounds.getWidth();
               double string_height = bounds.getHeight();

               if (max_text_width < string_width)
                  {
                  max_text_width = string_width;
                  }

               if (max_entry_height < string_height)
                  {
                  max_entry_height = string_height;
                  }

               /*
               * Include the marker size, if any, in making space in the key.
               */
               if (marker_radius < plot_item.radius &&
                   plot_item.marker_type != null)
                  {
                  marker_radius = plot_item.radius;
                  }
               item_count++;
               }
            }

         /*
         *  Get the width for placing the lines, and any marker symbols
         *  in the lines.
         */
         double line_length = 15.0;
         if (marker_radius > 0.0)
            {
            /*
            *  Uses markers on at least one line.
            */
            double marker_width = 2.0 * marker_radius;
            if (max_entry_height < 2.0 * marker_radius)
               {
               max_entry_height = 2.0 * marker_radius;
               }
            line_length += marker_width;
            }

         /*
         * Calculate the extents of the key box and key entry offsets into the
         * box.
         * The text_x_offset can now be calculated, as well as the keys box
         * width.
         * The keys_height is incrementally used to calculate the text
         * base locations, and updated for the rest later.
         */
         double text_x_offset = left_margin +
                                line_length +
                                line_to_text_margin;
         double keys_width = text_x_offset +
                             max_text_width +
                             right_margin;
         double keys_height = top_margin;

         for (int item_index = 0;
              item_index < item_count;
              item_index++)
            {
            /*
            *  Update the keys block height for the next entry inclusion.
            */
            keys_height += vertical_pad + max_entry_height;

            /*
            *  Get the location for the entry's text string "base point".
            */
            key_offsets.add(keys_height);
            }
         /*
         * Update the overall box height to include the bottom margin.
         */
         keys_height += bottom_margin;

         /*
         *  Create the border rectangle and image, then start drawing in
         *  the contents.
         */
         Rectangle2D.Double box = new Rectangle2D.Double(1.0,
                                                         1.0,
                                                         keys_width + 1.0,
                                                         keys_height + 1.0);

         key_image = new BufferedImage((int) keys_width + 4,
                                       (int) keys_height + 4,
                                       BufferedImage.TYPE_INT_ARGB);
         Graphics2D graphics_2d = key_image.createGraphics();

         graphics_2d.setBackground(this.key_box_background_color);
         graphics_2d.setColor(Color.BLACK);
         graphics_2d.setStroke(new BasicStroke(2.0F));
         graphics_2d.draw(box);

         /*
         *  Write/draw the key information into the key box image.
         */
         int key_index = 0;
         for (PlotItem plot_item: this.plot_2d.plot_items)
            {
            if (plot_item.is_selected)
               {
               graphics_2d.setColor(plot_item.color);

               double entry_y_offset = key_offsets.get(key_index);

               double key_ctr_x = left_margin + 0.5 * line_length;
               double key_ctr_y = entry_y_offset - 0.5 * max_entry_height;

               Point2D.Double start_point = new Point2D.Double(left_margin,
                                                               key_ctr_y);
               Point2D.Double end_point = new Point2D.Double(left_margin +
                                                             line_length,
                                                             key_ctr_y);
               /*
               *  Draw the line.
               */
               Line2D.Double line = new Line2D.Double(start_point,
                                                      end_point);
               graphics_2d.setStroke(new BasicStroke(2.0F));
               graphics_2d.setColor(plot_item.color);
               graphics_2d.draw(line);

               if (plot_item.marker_type != null &&
                   plot_item.radius > 0)
                  {
                  Point2D center_point = new Point2D.Double(key_ctr_x,
                                                            key_ctr_y);
                  drawMarker(graphics_2d,
                             plot_item.marker_type,
                             center_point,
                             plot_item.radius,
                             plot_item.color);
                  }

               float key_blc_x = (float) (left_margin +
                                          line_length +
                                          line_to_text_margin);
               float key_blc_y = (float) (entry_y_offset);

               /*
               *  Add the text to the entry.
               */
               graphics_2d.setColor(Color.black);

               TextLayout key_layout = key_layouts.get(key_index);
               key_layout.draw(graphics_2d, key_blc_x, key_blc_y);

               key_index++;
               }
            }
/*
         System.out.println("PlotPanel::drawKeyBoxImage() STATUS:\n"
                          + "   box.width = " + box.width + "\n"
                          + "   box.height = " + box.height + "\n"
                           );
*/
         }
/*
      System.out.println("PlotPanel::drawKeyBoxImage() on EXIT.");
*/
      return(key_image);
      }


   /**
   *  Paint a plot item.
   *  @param graphics_2d the graphics area in which to draw the node.
   *  @param plot_item the reference to the plot_item (plot line) to repaint.
   */
   private void paintPlotItem(Graphics2D graphics_2d,
                              PlotItem plot_item)
      {
      Color background_color = plot_item.color;
/*
      System.out.println("PlotPanel::paintPlotItem() on ENTRY:\n"
                       + "   plot_item: \""
                       + plot_item.getTitle()
                       + "\"  "
                       + background_color
                       + "\n"
                       + "   plot_box.x = " + this.plot_box.x + "\n"
                       + "   plot_box.y = " + this.plot_box.y + "\n"
                       + "   plot_box.width = " + this.plot_box.width + "\n"
                       + "   plot_box.height = " + this.plot_box.height + "\n"
                        );
*/


      /*
      *  These values must be transformed to the plot coordinates before use.
      */
      Double[] x_values = plot_item.x_values;
      Double[] y_values = plot_item.y_values;

      /*
      *  These are the values for translating and scaling the axes
      *  to the plot area and its size.
      */
      double x_value_range = axis_x.getAdjustedValueRange();
      double x_value_scale = this.plot_box.width / x_value_range;
      double x_value_offset = axis_x.getAdjustedValueMin();

      double y_value_range = axis_y.getAdjustedValueRange();
      double y_value_scale = this.plot_box.width / y_value_range;
      double y_value_offset = axis_y.getAdjustedValueMin();

/*
      System.out.println("painting Values: "
                       + " value offsets: ("
                       + x_value_offset + ", " + y_value_offset + ")\n"
                       + " value to plot scales: ("
                       + x_value_scale + ", " + y_value_scale + ")\n"
                        );
      System.out.println();
      System.out.println("The plot values:");
*/

      Point2D.Double prev_point = null;
      for (int point_index = 0;
           point_index < plot_item.num_points;
           point_index++)
         {
         double x_raw_value = x_values[point_index];
         double y_raw_value = y_values[point_index];

         double x_plot_value = (x_raw_value - x_value_offset) * x_value_scale;
         double y_plot_value = (y_raw_value - y_value_offset) * y_value_scale;

         if (axis_x.isLog())
            {
            /*
            *  The value needs converted to a displayable log of the value.
            */
            x_plot_value = Math.log10(x_raw_value - x_value_offset)
                           * x_value_scale;
            }
         else
            {
            x_plot_value = (x_raw_value - x_value_offset) * x_value_scale;
            }

         if (axis_y.isLog())
            {
            /*
            *  The value needs converted to a displayable log of the value.
            */
            y_plot_value = Math.log10(x_raw_value - x_value_offset)
                           * y_value_scale;
            }
         else
            {
            y_plot_value = (y_raw_value - y_value_offset) * y_value_scale;
            }

         Point2D.Double curr_point = new Point2D.Double(x_plot_value,
                                                        y_plot_value);
         if (prev_point != null)
            {
            Line2D.Double line = new Line2D.Double(curr_point, prev_point);
            graphics_2d.draw(line);
            }

         if (plot_item.radius > 0)
            {
            Point2D ulc = new Point2D.Double(curr_point.x - plot_item.radius,
                                             curr_point.y - plot_item.radius);
            Point2D lrc = new Point2D.Double(curr_point.x + plot_item.radius,
                                             curr_point.y + plot_item.radius);

            Ellipse2D ell = new Ellipse2D.Double();
            ell.setFrameFromDiagonal(ulc, lrc);

            graphics_2d.setColor(background_color);
            graphics_2d.fill(ell);

            graphics_2d.setStroke(new BasicStroke(1.0F));
            graphics_2d.setColor(this.foreground_color);
            graphics_2d.draw(ell);
            }

         prev_point = curr_point;
         }
/*
      System.out.println("PlotPanel::paintPlotItem() on EXIT.\n"
*/
      }


   /**
   *  Paint a scrolled plot (DUMMIED -- under development still, not usable).
   *  @param graphics_2d the graphics area in which to draw the node.
   *  @param plot_item the reference to the plot_item (plot line) to repaint.
   */
   private void paintScrolledPlot(Graphics2D graphics_2d)
      {
/*
      System.out.println("PlotPanel::paintScrolledPlot() on ENTRY:\n"
                       + "   plot_box.x = " + this.plot_box.x + "\n"
                       + "   plot_box.y = " + this.plot_box.y + "\n"
                       + "   plot_box.width = " + this.plot_box.width + "\n"
                       + "   plot_box.height = " + this.plot_box.height + "\n"
                        );
*/

      /*
      *  The plot image is updated as each value is added to the plot,
      *  So here all we do is copy the image into the plot display area.
      */

      if (scrolled_plot.magnitudes != null)
         {
         /*
         *  Get the plot image parts for displaying the values as
         *  columns of boxes per step.
         */
         boolean colors_selected = scrolled_plot.colors_selected;
         int plot_num_cols = scrolled_plot.num_cols_max;
         int num_boxes = scrolled_plot.y_values.size();

         int plot_col_width = (int) Math.floor(this.plot_box.width
                                               / plot_num_cols);
         int plot_col_height = (int) Math.floor(this.plot_box.height);

         int plot_col_offset = (int) (this.plot_box.width -
                                      (plot_col_width * plot_num_cols));

         double box_col_height = (int) Math.floor((double) plot_col_height /
                                                  num_boxes);
         double box_entry_height = box_col_height * 0.90;

         int num_cols = scrolled_plot.x_values.size();

         /*
         *  Get the magnitude extremes for the autoscaling.
         */
         double mag_max = -1.0;
         double mag_min = 100000.0;
         for (int col_index = 0;
              col_index < num_cols;
              col_index++)
            {
            ArrayList<Double> mags =
                           scrolled_plot.magnitudes.get(col_index);
            for (int mag_index = 0;
                 mag_index < mags.size();
                 mag_index++)
               {
               Double magnitude = mags.get(mag_index);
               if (magnitude != null)
                  {
                  if (mag_min > magnitude)
                     {
                     mag_min = magnitude;
                     }
                  if (mag_max < magnitude)
                     {
                     mag_max = magnitude;
                     }
                  }
               }
            }

         scrolled_plot.setValuesRange(mag_min, mag_max);

/*
         System.out.println("Drawing the scrolled plot image:\n"
                          + "   plot_num_cols = " + plot_num_cols + "\n"
                          + "   num_boxes = " + num_boxes + "\n"
                          + "   num_cols = " + num_cols + "\n"
                          + "   plot_col_width = " + plot_col_width + "\n"
                          + "   plot_col_height = " + plot_col_height + "\n"
                          + "   box_col_height = " + box_col_height + "\n"
                          + "   box_entry_height = " + box_entry_height + "\n"
                          + "   mag_min = " + mag_min + "\n"
                          + "   mag_max = " + mag_max + "\n"
                           );
*/

         /*
         *  These indices start at the right end of the plot and progress to
         *  the left.
         */
         for (int col_index = 0;
              col_index < num_cols;
              col_index++)
            {
            int plot_col_index = col_index + plot_num_cols - num_cols;

/*
            System.out.println("Drawing column:\n"
                             + "   plot_col_index = " + plot_col_index + "\n"
                             + "   col_index = " + col_index + "\n"
                              );
*/

            BufferedImage col_image =
                                new BufferedImage(plot_col_width,
                                                  plot_col_height,
                                                  BufferedImage.TYPE_INT_ARGB);
            Graphics2D col_graphics_2d = col_image.createGraphics();

            /*
            *  Get the x-value for this column, and the array of
            *  magnitudes for the column.
            */
            double x_value = scrolled_plot.x_values.get(col_index);
            ArrayList<Double> mags = scrolled_plot.magnitudes.get(col_index);

            for (int mag_index = 0;
                 mag_index < mags.size();
                 mag_index++)
               {
               Double magnitude = mags.get(mag_index);
               Color color = Color.BLACK;
               if (magnitude == null)
                  {
                  color = Color.lightGray;
                  }
               else
                  {
/*
                  color = scrolled_plot.getColorForValue(magnitude,
                                                         mag_min,
                                                         mag_max,
                                                         colors_selected);
*/
                  /*
                  *  Force rainbow colors.
                  */
                  color = ScrolledPlot.getColorForValue(magnitude,
                                                        mag_min,
                                                        mag_max,
                                                        true);
                  }
/*
               System.out.println("Drawing mag box:\n"
                                + "   magnitude = " + magnitude + "\n"
                                + "   color = " + color + "\n"
                                + "   mag_min = " + mag_min + "\n"
                                + "   mag_max = " + mag_max + "\n"
                                 );
*/
               Rectangle2D.Double value_box =
                 new Rectangle2D.Double(0.0,
                                        plot_col_height -
                                              mag_index * box_col_height +
                                              box_col_height * 0.05,
                                        plot_col_width,
                                        box_entry_height);
               col_graphics_2d.setColor(color);
               col_graphics_2d.fill(value_box);
               }
/*
            System.out.println("Drawing the scrolled plot subimage ("
                             + plot_col_index
                             + ", "
                             + col_index
                             + ")");
*/

            /*
            *  Get the offsets into the plot area for this next strip.
            */
            double x_loc = this.plot_box.x +
                           plot_col_index * plot_col_width +
                           plot_col_offset;
            double y_loc = this.plot_box.y;

            /*
            *  Scale the strip to fit at the proper location in the 
            *  plot area.
            */
            double x_scale = col_width / scrolled_plot.plot_height;
            double y_scale = col_height / scrolled_plot.plot_width;


            /*
            *  Transform the strip destination and draw it into the
            *  plot area.
            */
            AffineTransform image_at = new AffineTransform();
            image_at.setToIdentity();
/*
            image_at.scale(x_scale, y_scale);
*/
            image_at.translate(x_loc, y_loc);
            graphics_2d.drawImage(col_image, image_at, null);
            }
         }
      }


   /**
   *  Print the current plot.
   *  @param graphics the environment for the printout.
   *  @param page_format the page format control structure.
   *  @param page_index the index of the current page to be printed.
   */
   public int print(Graphics graphics,
                    PageFormat page_format,
                    int page_index)
      throws PrinterException
      {
      int return_value = Printable.NO_SUCH_PAGE;

      double x_offset = page_format.getImageableX();
      double y_offset = page_format.getImageableY();

      double page_width = page_format.getImageableWidth();
      double page_height = page_format.getImageableHeight();


      int curr_width = (int) page_width;
      int curr_height = (int) page_height;

      /*
      *  Adjust to the displayed aspect ratio, rather than the paper
      *  size.
      */
      if (page_width * last_aspect_ratio > page_height)
         {
         curr_width = (int) Math.floor(page_height / last_aspect_ratio);
         }
      else
         {
         curr_height = (int) Math.floor(page_width * last_aspect_ratio);
         }

/*
      System.out.println("PlotPanel::print() on ENTRY:\n"
                       + "   offsets = ("
                       + x_offset
                       + ", "
                       + y_offset
                       + ")\n"
                       + "   curr size = ("
                       + curr_width
                       + ", "
                       + curr_height
                       + ")\n"
                       + "   page_index = "
                       + page_index
                        );
*/


      if (panel == null)
         {
         /*
         *  There are no contents to be printed.
         */
         }
      else if (page_index == 0)
         {
         if (plot_2d != null || scrolled_plot != null)
            {
            /*
            *  This "super.print" is NOT desirable.  It prints another
            *  copy using the "paintComponent" above, ignoring the page
            *  offsets and size information.
            */
            //super.print(graphics);

            Graphics2D graphics_2d = (Graphics2D) graphics;
            graphics_2d.translate(x_offset, y_offset);

            drawContents(graphics, curr_width, curr_height);
            }

         return_value = Printable.PAGE_EXISTS;
         }

/*
      System.out.println("PlotPanel::print() on EXIT.");
*/
      return(return_value);
      }

   
   /**
   *  Write the plot that is encapsulate in this StoredPicture instance
   *  to a file.
   *  @param out_file_path where to write the "png" formatted image.
   */
   public void writePlotImage(String out_file_path)
      {
      /*
      *  Write the contents of the plot to the file.
      */
      String file_path = out_file_path + ".png";
      int curr_width = this.getWidth();
      int curr_height = this.getHeight();
      try
         {
         File output_file = new File(file_path);

         /*
         *  Get the plot into an image buffer, then write it to the file.
         */
         BufferedImage plot_image = 
                                new BufferedImage(curr_width,
                                                  curr_height,
                                                  BufferedImage.TYPE_INT_ARGB);
         Graphics2D plot_graphics_2d = plot_image.createGraphics();
         drawContents(plot_graphics_2d, curr_width, curr_height);

         ImageIO.write(plot_image, "png", output_file);
         System.out.println("StoredImage::writePlotImage():"
                          + " Plot2D written to: \""
                          + file_path
                          + "\""
                           );
         }
      catch (Exception except)
         {
         System.out.println("StoredImage::writePlotImage(): ERROR openning \""
                          + file_path
                          + "\".\n"
                          + except);
         except.printStackTrace(System.out);
         }
      }

   }

