
/*
*  The definition of a single plot axis.
*/

package util;


import java.util.*;

import java.awt.*;


/**
*  This class defines the basic axis structure that can be used by plot
*  to display the plot data.
*/

public class Axis
   {

   String title = null;

   int num = 0;

   /*
   *  The displayable linear data region parameters, after beautifying
   *  for a plot.
   */
   double lin_data_min;
   double lin_data_max;
   double lin_data_range;

   /*
   *  The various value collections for the linear tics.
   */
   ArrayList<Double> lin_major_tic_values = null;
   ArrayList<String> lin_major_tic_labels = null;
   ArrayList<Double> lin_minor_tic_values = null;

   /*
   *  The displayable data region parameters, after beautifying for a plot.
   *  These are returned to the caller if the axis "is_log".  They are
   *     log_data_min = log10(1.0 * 10^min_exp) = min_exp
   *     log_data_max = log10(1.0 * 10^max_exp) = max_exp
   *     log_data_range = log_data_max - log_data_min
   *  These values can be used directly for autoranging the log10 data
   *  values to the plot region.
   */
   double log_data_min;
   double log_data_max;
   double log_data_range;

   /*
   *  The various value collections for the log tics.
   */
   ArrayList<Double> log_major_tic_values = null;
   ArrayList<String> log_major_tic_labels = null;
   ArrayList<Double> log_minor_tic_values = null;

   /*
   * The largest and smallest positions of the tic data extremes.
   */
   int most_sig_digit_pos;
   int least_sig_digit_pos;

   /*
   *  The raw data range values, may get adjusted to include 0.0
   */
   double raw_data_value_min;
   double raw_data_value_max;
   double raw_data_value_range;

   /*
   *  The actual data range values.
   */
   double actual_data_value_min;
   double actual_data_value_max;
   double actual_data_value_range;

   /*
   * Flags for the display of the different components of the plot.
   */
   boolean display_tics = true;
   boolean display_grid = true;
   boolean display_minor_tics = true;
   boolean is_log = false;

   /*
   *  If it is not a log axis this indicates if zero is to be forced into
   *  the plot range.
   */
   boolean include_zero = false;
   boolean prev_include_zero = false;

   static double [] log_tics =
      {
      0.000, 0.301, 0.477, 0.602, 0.699,
      0.778, 0.845, 0.903, 0.954, 1.000
      };

   /**
   *  Constructor, expects the scale to be set later.
   *  @param axis_title the title to label this axis.
   */
   public Axis(String axis_title)
      {
/*
      System.out.println("Axis::Constructor #1 on ENTRY:\n"
                       + "   axis_title = " + axis_title + "\n"
                        );
*/

      this.title = axis_title;

/*
      System.out.println("Axis::Constructor #1 on EXIT.");
*/
      }

   /**
   * Gets the title string for the axis.
   * @return the title string.
   */
   public String getTitle()
      {
      return(this.title);
      }

   /**
   *  Retrieve the is_log flag value for the axis.
   *  @return the value of the is_log flag.
   */
   public boolean isLog()
      {
      return(this.is_log);
      }

   /**
   *  Set the is_log flag value for the axis.
   *  @param is_log is true to make the axis be a log axis.
   */
   public void setLog(boolean is_log)
      {
      this.is_log = is_log;
      if (this.is_log)
         {
         this.prev_include_zero = this.include_zero;
         this.include_zero = false;
         setupLog();
         }
      else
         {
         this.include_zero = this.prev_include_zero;
         setupLinear();
         }
      }

   /**
   * Checks for the display_tics flag setting.
   * @return true if the display_tics flag is set.
   */
   public boolean getDisplayTics()
      {
      return(this.display_tics);
      }

   /**
   * Sets the display_tics flag.
   * @param display_tics flag for displaying tics.
   */
   public void setDisplayTics(boolean display_tics)
      {
      this.display_tics = display_tics;
      }

   /**
   * Checks for the display_minor_tics flag setting.
   * @return true if the display_minor_tics flag is set.
   */
   public boolean getDisplayMinorTics()
      {
      return(this.display_minor_tics);
      }

   /**
   * Sets the display_minor_tics flag.
   * @param display_minor_tics flag for displaying minor tics.
   */
   public void setDisplayMinorTics(boolean display_minor_tics)
      {
      this.display_minor_tics = display_minor_tics;
      }

   /**
   * Checks for the display_grid flag setting.
   * @return true if the display_grid flag is set.
   */
   public boolean getDisplayGrid()
      {
      return(this.display_grid);
      }

   /**
   * Sets the display_grid flag.
   * @param display_grid flag for displaying the grid.
   */
   public void setDisplayGrid(boolean display_grid)
      {
      this.display_grid = display_grid;
      }

   /**
   *  Updates the scales and tics for an existing plot axis.
   *  @param axis_title the title to label this axis.
   *  @param value_min the minimum value to cover with this axis.
   *  @param value_max the maximum value to cover with this axis.
   *  @param is_log if true the axis is to use a log scale.
   *  @param include_zero if true include zero even if the data doesn't.
   *  @throws Exception if log of non-positive values requested.
   */
   public void updateAxis(String axis_title,
                          double value_min,
                          double value_max,
                          boolean is_log,
                          boolean include_zero)
      throws Exception
      {
/*
      System.out.println("Axis::updateAxis on ENTRY:\n"
                       + "   axis_title = " + axis_title + "\n"
                       + "   value_min = " + value_min + "\n"
                       + "   value_max = " + value_max + "\n"
                       + "   is_log = " + is_log + "\n"
                       + "   d_include_zero = " + d_include_zero + "\n"
                        );
*/
      this.title = axis_title;

      this.prev_include_zero = include_zero;

      this.is_log = is_log;
      if (this.is_log)
         {
         this.include_zero = false;
         if (value_min <= 0.0 || value_max <= 0.0)
            {
            throw(new Exception("Axis.updateAxis() ERROR: "
                                + "Log requested of non-positive numbers."));
            }
         }
      else
         {
         this.include_zero = include_zero;
         }

      this.raw_data_value_min = value_min;
      this.raw_data_value_max = value_max;

      this.actual_data_value_min = value_min;
      this.actual_data_value_max = value_max;

      if (this.raw_data_value_min > 0.0 && this.include_zero)
         {
         this.raw_data_value_min = 0.0;
         }
      if (this.raw_data_value_max < 0.0 && this.include_zero)
         {
         this.raw_data_value_max = 0.0;
         }

      this.raw_data_value_range = this.raw_data_value_max
                                  - this.raw_data_value_min;
      this.actual_data_value_range = this.actual_data_value_max
                                     - this.actual_data_value_min;

      if (this.is_log)
         {
         setupLog();
         }
      else
         {
         setupLinear();
         }

/*
      System.out.println("Axis::updateAxis on EXIT.");
*/
      }


   /**
   *  Sets up the linear axis values for use in a plot.  Gets stats and values
   *  for the plot.
   */
   public void setupLinear()
      {
/*
      System.out.println("Axis::setupLinear on ENTRY."
                       + "   title = \"" + this.title + "\"\n"
                       + "   raw_data_value_max = "
                         + this.raw_data_value_max
                         + "\n"
                       + "   raw_data_value_min = "
                         + this.raw_data_value_min
                         + "\n"
                       + "   raw_data_value_range = "
                         + this.raw_data_value_range
                         + "\n"
                        );
*/
      /*
      *  It is a linear axis.
      */
      int format_inputs = 1;

      /*
      *  Clear and reset the linear tic position arrays,'
      *  All values should be adjusted to the "generic" plot
      *  range [0.0, 1.0]
      */
      lin_major_tic_values = new ArrayList<Double>();
      lin_major_tic_labels = new ArrayList<String>();
      lin_minor_tic_values = new ArrayList<Double>();

      /*
      *  If the max and min are "close" but large this can be very
      *  different from "mag_power" and indicates how many digits
      *  are needed.
      */

      /*
      *  Get the most significant digit position needed for the
      *  formatted tic labels.
      */
      double data_value_max_mag = Math.max(
                                     Math.abs(this.raw_data_value_max),
                                     Math.abs(this.raw_data_value_min));

      double mag_power = Math.log10(data_value_max_mag);

/*
      System.out.println("Axis::setupLinear() STATUS (initial):\n"
                  + "   data_value_max_mag = " + data_value_max_mag + "\n"
                  + "   mag_power = " + mag_power + "\n"
                        );
*/

      /*
      *  This converts to the most significant digit for the
      *  largest magnitude tics.
      */
      this.most_sig_digit_pos = (int) Math.ceil(mag_power);

      /*
      *  Get the first decimal place where the max and min differ.
      */
      double power = Math.log10(this.raw_data_value_range);
      this.least_sig_digit_pos = (int) Math.floor(power);

/*
      System.out.println("Axis::setupLinear() STATUS:\n"
                       + "   title = \"" + this.title + "\"\n"
                       + "   most_sig_digit_pos = "
                         + this.most_sig_digit_pos
                         + "\n"
                       + "   least_sig_digit_pos = "
                         + this.least_sig_digit_pos
                         + "\n"
                       + "   is_log = \"" + this.is_log + "\"\n"
                        );
*/
      double mag  = Math.pow(10.0, this.most_sig_digit_pos);

/*
      System.out.println("   Initial mag = " + mag);
*/

      if (raw_data_value_max <= 0.2 * mag)
         {
         mag *= 0.2;
         }
      else if (raw_data_value_max <= 0.3 * mag)
         {
         mag *= 0.3;
         }
      else if (raw_data_value_max <= 0.4 * mag)
         {
         mag *= 0.4;
         }
      else if (raw_data_value_max <= 0.5 * mag)
         {
         mag *= 0.5;
         }
      else if (raw_data_value_max <= 0.6 * mag)
         {
         mag *= 0.6;
         }
      else if (raw_data_value_max <= 0.7 * mag)
         {
         mag *= 0.7;
         }
      else if (raw_data_value_max <= 0.8 * mag)
         {
         mag *= 0.8;
         }
      else if (raw_data_value_max <= 0.9 * mag)
         {
         mag *= 0.9;
         }

      /*
      *  These represent the displayable value min, max, and range.
      */
      this.lin_data_max = mag * Math.ceil(this.raw_data_value_max / mag);
      this.lin_data_min = mag * Math.floor(this.raw_data_value_min / mag);
      this.lin_data_range = this.lin_data_max - this.lin_data_min;

      int num_tics = (int) Math.floor(this.lin_data_range / mag + 0.5);

/*
      System.out.println("Axis::setupLinear() STATUS"
                       + "(after linear adjust):\n"
                       + "   title = \"" + this.title + "\"\n"
                       + "   (" + this.lin_data_min
                          + ", "
                          + this.lin_data_max
                          + ")"
                       + "  " + this.lin_data_range + "\n"
                       + "   mag = " + mag + "\n"
                       + "   num_tics = " + num_tics + "\n"
                       + "   most_sig_digit_pos = "
                         + this.most_sig_digit_pos
                         + "\n"
                        );
*/

      if (num_tics > 10)
         {
         if ((num_tics & 1) != 0)
            {
            num_tics++;
            this.lin_data_range = num_tics * mag;
            if ((num_tics & 1) != 0)
               {
               this.lin_data_max += this.lin_data_range;
               }
            else
               {
               this.lin_data_min -= this.lin_data_range;
               }
            }
         num_tics /= 2;
         }

/*
      System.out.println("Axis::setupLinear() STATUS"
                       + "(after second linear adjust):\n"
                       + "   title = \"" + title + "\"\n"
                       + "   ("
                          + this.lin_data_min
                          + ", "
                          + this.lin_data_max
                          + ")"
                       + "  " + lin_data_range + "\n"
                       + "   mag = " + mag + "\n"
                       + "   num_tics = " + num_tics + "\n"
                        );
*/
      int num_minor_tics = 2;
      if (num_tics <= 1)
         {
         num_tics = 5;
         }
      else if (num_tics == 2)
         {
         num_tics *= 2;
         num_minor_tics = 5;
         }
      else if (num_tics < 5)
         {
         num_minor_tics = 5;
         }

      double tic_incr = this.lin_data_range / num_tics;
      double minor_tic_incr = tic_incr / num_minor_tics;
/*
      System.out.println("   num_minor_tics = " + num_minor_tics + "\n"
                       + "   tic_incr = " + tic_incr + "\n"
                       + "   minor_tic_incr = " + minor_tic_incr + "\n"
                        );
*/

      /*
      * Get the major and minor tic values and the maximum major tic
      * value magnitude.
      */
      double prev_tic_value = 0.0;
      double max_tic_mag = -1.0;

      for (int tic_index = 0;
           tic_index <= num_tics;
           tic_index++)
         {
         double tic_value = this.lin_data_min + tic_index * tic_incr;
         double tic_value_mag = Math.abs(tic_value);
/*
         System.out.println("Axis::setupLinear() STATUS #1:\n"
                          + "   tic_index = " + tic_index + "\n"
                          + "   tic_value = " + tic_value + "\n"
                           );
*/
         if (max_tic_mag < tic_value_mag)
            {
            max_tic_mag = tic_value_mag;
            }

         /*
         *  Add the next major tic entry.
         */
         this.lin_major_tic_values.add(tic_value);

         if (tic_index != 0)
            {
            for (int minor_tic_index = 1;
                 minor_tic_index < num_minor_tics;
                 minor_tic_index++)
               {
               double minor_tic_value = prev_tic_value +
                                        minor_tic_index * minor_tic_incr;
               this.lin_minor_tic_values.add(minor_tic_value);
               }
            }

         prev_tic_value = tic_value;
         }

      /*
      *  Get the actual format to use for the tic labels,
      *  based on the larges label's value.
      */
      String tic_fmt = null;

      if (max_tic_mag > 10000.0 ||
          max_tic_mag < 0.01)
         {
         tic_fmt = "%7.2g";
         }
      else if (max_tic_mag > 1000.0)
         {
         tic_fmt = "%5.0f";
         }
      else if (max_tic_mag > 100.0)
         {
         tic_fmt = "%5.0f";
         }
      else if (max_tic_mag > 10.0)
         {
         tic_fmt = "%5.1f";
         }
      else if (max_tic_mag > 1.0)
         {
         tic_fmt = "%5.2f";
         }
      else if (max_tic_mag > 0.1)
         {
         tic_fmt = "%5.3f";
         }
      else
         {
         tic_fmt = "%6.3f";
         }

/*
      System.out.println("Axis::setupLinear() STATUS #2:\n"
                       + "   tic_fmt = " + tic_fmt + "\n"
                        );
*/
      for (int tic_index = 0;
           tic_index <= num_tics;
           tic_index++)
         {
         double tic_value = this.lin_major_tic_values.get(tic_index);
         String tic_label = String.format(tic_fmt, tic_value);
         this.lin_major_tic_labels.add(tic_label);

/*
         System.out.println("   " + tic_index + " ("
                          + tic_value + ", \""
                          + tic_label + "\")"
                           );
*/
         }

/*
      System.out.println("Axis::setupLinear on EXIT.");
*/
      }

   /**
   *  Sets up the log axis values for use in a plot.  Gets stats and values
   *  for the plot.
   */
   public void setupLog()
      {
/*
      System.out.println("Axis::setupLog() on ENTRY:\n"
                       + "   title = \"" + this.title + "\"\n"
                       + "   raw_data_value_max = "
                         + this.raw_data_value_max
                         + "\n"
                       + "   raw_data_value_min = "
                         + this.raw_data_value_min
                         + "\n"
                       + "   raw_data_value_range = "
                         + this.raw_data_value_range
                         + "\n"
                       + "   actual_data_value_max = "
                         + this.actual_data_value_max
                         + "\n"
                       + "   actual_data_value_min = "
                         + this.actual_data_value_min
                         + "\n"
                       + "   actual_data_value_range = "
                         + this.actual_data_value_range
                         + "\n"
                        );
*/
      /*
      *  It is a logarithmic axis.
      */
      /*
      *  Clear and reset the tic position arrays,
      *  All values should be adjusted to the "generic" plot
      *  range [0.0, 1.0]
      */
      this.log_major_tic_values = new ArrayList<Double>();
      this.log_major_tic_labels = new ArrayList<String>();
      this.log_minor_tic_values = new ArrayList<Double>();

      double data_value_log_max = Math.log10(this.raw_data_value_max);
      double data_value_log_min = data_value_log_max - 3.0;
      if (this.actual_data_value_min > 0.0)
         {
         data_value_log_min = Math.log10(this.actual_data_value_min);
         }

      /*
      *  These are the actual range of values to be displayed on the
      *  log axis.
      */
      this.log_data_min = Math.floor(data_value_log_min);
      this.log_data_max = Math.ceil(data_value_log_max);
      this.log_data_range = this.log_data_max - this.log_data_min;
      if (this.log_data_range < 1.0)
         {
         this.log_data_range = 1.0;
         this.log_data_min = this.log_data_max - 1.0;
         }

      /*
      *  Get the log cycles needed.
      */
      int log_num_tics = (int) (this.log_data_max - this.log_data_min) + 1;

/*
      System.out.println("Axis::setupLog() STATUS "
                       + "(after log adjust):\n"
                       + "   title = \"" + title + "\"\n"
                       + "   log_num_tics = " + log_num_tics + "\n"
                       + "   actual_data_value_min = "
                         + this.actual_data_value_min
                         + "\n"
                       + "   raw_data_value_max = "
                         + this.raw_data_value_max
                         + "\n"
                       + "   log_data_min = " + this.log_data_min + "\n"
                       + "   log_data_max = " + this.log_data_max + "\n"
                       + "   log_data_range = " + this.log_data_range + "\n"
                       + "   log_num_tics = " + log_num_tics + "\n"
                        );
*/

      int data_exponent = (int) log_data_min;
      double prev_tic_value = 0.0;
      for (int tic_index = 0;
           tic_index <= log_num_tics;
           tic_index++)
         {
         /*
         *  Build the tic label format string.
         */
         String tic_fmt_dec = null;
         String tic_fmt = null;
         if (data_exponent > 3)
            {
            tic_fmt = "%7.1g";
            }
         else if (data_exponent > 0)
            {
            tic_fmt_dec = String.format("%d.0", data_exponent);
            tic_fmt = "%" + tic_fmt_dec + "f";
            }
         else if (data_exponent < -3)
            {
            tic_fmt = "%8.1g";
            }
         else if (data_exponent < 0)
            {
            tic_fmt_dec = String.format("%d.%d",
                                        2 - data_exponent,
                                        -data_exponent);
            tic_fmt = "%" + tic_fmt_dec + "f";
            }
         else
            {
            tic_fmt = "%3.1f";
            }
         /*
         * Compute the power of 10 and store into "string".
         */
         double tic_value = Math.pow(10.0, data_exponent);

         String log_tic_label = String.format(tic_fmt, tic_value);
/*
         System.out.println("tic_fmt_dec = \"" + tic_fmt_dec + "\"");
         System.out.println("tic_fmt = \"" + tic_fmt + "\"");
         System.out.print("   minor tics for "
                          + tic_value
                          + " ["
                          + tic_index
                          + "]: (");
         String delim = "";
*/
         if (tic_index != 0)
            {
            /*
            * Add the logarithmically spaced minor tic marks, starting
            * at the previous tic_value.
            */
            for (int minor_tic_index = 2;
                 minor_tic_index < 10;
                 minor_tic_index++)
               {
               double log_minor_tic_value = Math.log10(prev_tic_value
                                                       * minor_tic_index);
               log_minor_tic_values.add(log_minor_tic_value);
/*
               System.out.print(delim + log_minor_tic_value);
               delim = ", ";
*/
               }
            }
/*
         System.out.println(")");
*/

         /* Save the major tic entries. */
         double log_tic_value = data_exponent;
         log_major_tic_values.add(log_tic_value);
         log_major_tic_labels.add(log_tic_label);

         data_exponent++;
         prev_tic_value = tic_value;
         }

/*
      System.out.println("Axis::setupLog on EXIT.");
*/
      }


   /**
   *  Retrieve the major tic location values for the axis.
   *  @return the location values for each of the major tic marks.
   */
   public ArrayList<Double> getMajorTics()
      {
      ArrayList<Double> values = this.lin_major_tic_values;
      if (this.is_log)
         {
         values = this.log_major_tic_values;
         }
      return(values);
      }


   /**
   *  Retrieve the major tic labels for the axis.
   *  @return the labels for each of the major tic marks.
   */
   public ArrayList<String> getTicLabels()
      {
      ArrayList<String> labels = this.lin_major_tic_labels;
      if (this.is_log)
         {
         labels = this.log_major_tic_labels;
         }
      return(labels);
      }


   /**
   *  Retrieve the minor tic location values for the axis.
   *  @return the location values for each of the minor tic marks.
   */
   public ArrayList<Double> getMinorTics()
      {
      ArrayList<Double> values = this.lin_minor_tic_values;
      if (this.is_log)
         {
         values = this.log_minor_tic_values;
         }
      return(values);
      }


   /**
   *  Retrieve the adjusted minimum (display minimum) value for the axis
   *  data values.  If the axis is_log then this is the log of the smallest
   *  value to include in the display, otherwise it is the value of the
   *  smallest value to include in the display.
   *  @return the adjusted minimum value for the axis.
   */
   public double getAdjustedValueMin()
      {
      double value = this.lin_data_min;
      if (this.is_log)
         {
         value = this.log_data_min;
         }
      return(value);
      }


   /**
   *  Retrieve the adjusted maximum (display maximum) value for the axis
   *  data values.  If the axis is_log then this is the log of the largest
   *  value to include in the display, otherwise it is the value of the
   *  largest value to include in the display.
   *  @return the adjusted maximum value for the axis.
   */
   public double getAdjustedValueMax()
      {
      double value = this.lin_data_max;
      if (this.is_log)
         {
         value = this.log_data_max;
         }
      return(value);
      }


   /**
   *  Retrieve the adjusted range (display range) for the axis
   *  data values.  If the axis is_log then this is the range of the logs
   *  of the extreme values to include in the display, otherwise it is
   *  the range of the extreme values to include in the display.
   *  @return the adjusted range value for the axis.
   */
   public double getAdjustedValueRange()
      {
      double value = this.lin_data_range;
      if (this.is_log)
         {
         value = this.log_data_range;
         }
      return(value);
      }
   }

