/*
* This is a nested set of classes to give a numeric class type for contained
* data.
*/

package util;

import java.lang.*;

import util.geom.*;

public abstract class Numeric
   {
   public abstract String toString();
   public abstract void Print();

   public abstract boolean isInteger();
   public abstract boolean isFloat();
   public abstract boolean isDouble();
   public abstract boolean isBounded();
   public abstract boolean isNonZero();

   public abstract java.lang.Integer intValue();
   public abstract java.lang.Float floatValue();
   public abstract java.lang.Double doubleValue();
   public abstract Vector2D boundedValue();
   public abstract java.lang.Double getLower();
   public abstract java.lang.Double getUpper();

   public abstract Numeric Add(Numeric other);
   public abstract Numeric Subtract(Numeric other);
   public abstract Numeric Multiply(Numeric other);
   public abstract Numeric Divide(Numeric other);

   public static class Integer extends Numeric
      {
      java.lang.Integer value = null;
      public Integer(int value)
         {
         this.value = value;
         }
      public String toString()
         {
         String out_string = "" + this.value;
         return(out_string);
         }
      public void Print()
         {
         System.out.print("" + this.value);
         }

      public boolean isInteger()
         {
         return(true);
         }
      public boolean isFloat()
         {
         return(false);
         }
      public boolean isDouble()
         {
         return(false);
         }
      public boolean isBounded()
         {
         return(false);
         }
      public boolean isNonZero()
         {
         return(this.value != 0);
         }

      public java.lang.Integer intValue()
         {
         return(this.value.intValue());
         }

      public java.lang.Float floatValue()
         {
         return(this.value.floatValue());
         }

      public java.lang.Double doubleValue()
         {
         return(this.value.doubleValue());
         }

      public Vector2D boundedValue()
         {
         return(new Vector2D(this.getLower(), this.getUpper()));
         }

      public java.lang.Double getLower()
         {
         return(this.value.doubleValue() - 0.5);
         }

      public java.lang.Double getUpper()
         {
         return(this.value.doubleValue() + 0.5);
         }


      public Numeric Add(Numeric other)
         {
         Numeric result = null;
         if (other.isInteger())
            {
            result = new Numeric.Integer(this.value.intValue()
                                            + other.intValue());
            }
         else if (other.isFloat())
            {
            result = new Numeric.Float(this.value.floatValue()
                                          + other.floatValue());
            }
          else if (other.isDouble())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           + other.doubleValue());
            }
          else if (other.isBounded())
            {
            result = new Numeric.Bounded(this.getLower() + other.getLower(),
                                         this.getUpper() + other.getUpper());
            }
         return(result);
         }

      public Numeric Subtract(Numeric other)
         {
         Numeric result = null;
         if (other.isInteger())
            {
            result = new Numeric.Integer(this.value.intValue()
                                            - other.intValue());
            }
         else if (other.isFloat())
            {
            result = new Numeric.Float(this.value.floatValue()
                                          - other.floatValue());
            }
          else if (other.isDouble())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           - other.doubleValue());
            }
          else if (other.isBounded())
            {
            /*
            *  The other's bounds are reversed by negation.
            */
            result = new Numeric.Bounded(this.getLower() - other.getUpper(),
                                         this.getUpper() - other.getLower());
            }
         return(result);
         }

      public Numeric Multiply(Numeric other)
         {
         Numeric result = null;
         if (other.isInteger())
            {
            result = new Numeric.Integer(this.value.intValue()
                                            * other.intValue());
            }
         else if (other.isFloat())
            {
            result = new Numeric.Float(this.value.floatValue()
                                          * other.floatValue());
            }
          else if (other.isDouble())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           * other.doubleValue());
            }
          else if (other.isBounded())
            {
            /*
            *  The bounds combinations must be considered.
            *    [++][++] ==> no problem.
            *    [++][-+] ==> no problem.
            *    [++][--] ==> reverse order.
            *    [-+][++] ==> reverse order.
            *    [-+][-+] ==> reverse order.
            *    [-+][--] ==> reverse order.
            *    [--][++] ==> reverse order.
            *    [--][-+] ==> reverse order.
            *    [--][--] ==> no problem.
            */
            double this_lower = this.getLower();
            double this_upper = this.getUpper();

            double other_lower = other.getLower();
            double other_upper = other.getUpper();

            double lower_lower = this_lower * other_lower;
            double lower_upper = this_lower * other_upper;
            double upper_lower = this_upper * other_lower;
            double upper_upper = this_upper * other_upper;

            double new_lower = Math.min(lower_lower, lower_upper);
            new_lower = Math.min(new_lower, upper_lower);
            new_lower = Math.min(new_lower, upper_upper);

            double new_upper = Math.max(lower_lower, lower_upper);
            new_upper = Math.max(new_upper, upper_lower);
            new_upper = Math.max(new_upper, upper_upper);

            result = new Numeric.Bounded(new_lower, new_upper);
            }
         return(result);
         }

      public Numeric Divide(Numeric other)
         {
         Numeric result = null;
         /*
         *  If the dividend is zero or is an interval that includes zero then
         *  the result will be infinite, so null will be returned.
         */
         if (other.isNonZero())
            {
            if (other.isInteger())
               {
               result = new Numeric.Integer(this.value.intValue()
                                               / other.intValue());
               }
            else if (other.isFloat())
               {
               result = new Numeric.Float(this.value.floatValue()
                                             / other.floatValue());
               }
             else if (other.isDouble())
               {
               result = new Numeric.Double(this.value.doubleValue()
                                              / other.doubleValue());
               }
             else if (other.isBounded())
               {
               /*
               *  The bounds combinations must be considered.  Any interval
               *  that includes zero for the denominator must be excluded.
               */
               double this_lower = this.getLower();
               double this_upper = this.getUpper();

               double other_lower = other.getLower();
               double other_upper = other.getUpper();

               double lower_lower = this_lower / other_lower;
               double lower_upper = this_lower / other_upper;
               double upper_lower = this_upper / other_lower;
               double upper_upper = this_upper / other_upper;

               double new_lower = Math.min(lower_lower, lower_upper);
               new_lower = Math.min(new_lower, upper_lower);
               new_lower = Math.min(new_lower, upper_upper);

               double new_upper = Math.max(lower_lower, lower_upper);
               new_upper = Math.max(new_upper, upper_lower);
               new_upper = Math.max(new_upper, upper_upper);

               result = new Numeric.Bounded(new_lower, new_upper);
               }
            }
         return(result);
         }
      }

   public static class Float extends Numeric
      {
      java.lang.Float value = null;
      public Float(float value)
         {
         this.value = value;
         }
      public String toString()
         {
         String out_string = "" + this.value;
         return(out_string);
         }
      public void Print()
         {
         System.out.print("" + this.value);
         }

      public boolean isInteger()
         {
         return(false);
         }
      public boolean isFloat()
         {
         return(true);
         }
      public boolean isDouble()
         {
         return(false);
         }
      public boolean isBounded()
         {
         return(false);
         }
      public boolean isNonZero()
         {
         return(this.value != 0.0F);
         }

      public java.lang.Integer intValue()
         {
         return(this.value.intValue());
         }

      public java.lang.Float floatValue()
         {
         return(this.value.floatValue());
         }

      public java.lang.Double doubleValue()
         {
         return(this.value.doubleValue());
         }

      public Vector2D boundedValue()
         {
         return(new Vector2D(this.getLower(), this.getUpper()));
         }

      public java.lang.Double getLower()
         {
         return(this.value.doubleValue());
         }

      public java.lang.Double getUpper()
         {
         return(this.value.doubleValue());
         }


      public Numeric Add(Numeric other)
         {
         Numeric result = null;
         if (other.isInteger())
            {
            result = new Numeric.Float(this.value.floatValue()
                                          + other.floatValue());
            }
         else if (other.isFloat())
            {
            result = new Numeric.Float(this.value.floatValue()
                                          + other.floatValue());
            }
          else if (other.isDouble())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           + other.doubleValue());
            }
          else if (other.isBounded())
            {
            result = new Numeric.Bounded(this.getLower() + other.getLower(),
                                         this.getUpper() + other.getUpper());
            }
         return(result);
         }

      public Numeric Subtract(Numeric other)
         {
         Numeric result = null;
         if (other.isInteger())
            {
            result = new Numeric.Float(this.value.floatValue()
                                          - other.floatValue());
            }
         else if (other.isFloat())
            {
            result = new Numeric.Float(this.value.floatValue()
                                          - other.floatValue());
            }
          else if (other.isDouble())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           - other.doubleValue());
            }
          else if (other.isBounded())
            {
            /*
            *  The other's bounds are reversed.
            */
            result = new Numeric.Bounded(this.getLower() - other.getUpper(),
                                         this.getUpper() - other.getLower());
            }
         return(result);
         }

      public Numeric Multiply(Numeric other)
         {
         Numeric result = null;
         if (other.isInteger())
            {
            result = new Numeric.Float(this.value.floatValue()
                                          * other.floatValue());
            }
         else if (other.isFloat())
            {
            result = new Numeric.Float(this.value.floatValue()
                                          * other.floatValue());
            }
          else if (other.isDouble())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           * other.doubleValue());
            }
          else if (other.isBounded())
            {
            /*
            *  The bounds combinations must be considered.
            *    [++][++] ==> no problem.
            *    [++][-+] ==> no problem.
            *    [++][--] ==> reverse order.
            *    [-+][++] ==> reverse order.
            *    [-+][-+] ==> reverse order.
            *    [-+][--] ==> reverse order.
            *    [--][++] ==> reverse order.
            *    [--][-+] ==> reverse order.
            *    [--][--] ==> no problem.
            */
            double this_lower = this.getLower();
            double this_upper = this.getUpper();

            double other_lower = other.getLower();
            double other_upper = other.getUpper();

            double lower_lower = this_lower * other_lower;
            double lower_upper = this_lower * other_upper;
            double upper_lower = this_upper * other_lower;
            double upper_upper = this_upper * other_upper;

            double new_lower = Math.min(lower_lower, lower_upper);
            new_lower = Math.min(new_lower, upper_lower);
            new_lower = Math.min(new_lower, upper_upper);

            double new_upper = Math.max(lower_lower, lower_upper);
            new_upper = Math.max(new_upper, upper_lower);
            new_upper = Math.max(new_upper, upper_upper);

            result = new Numeric.Bounded(new_lower, new_upper);
            }
         return(result);
         }

      public Numeric Divide(Numeric other)
         {
         Numeric result = null;
         /*
         *  If the dividend is zero or is an interval that includes zero then
         *  the result will be infinite, so null will be returned.
         */
         if (other.isNonZero())
            {
            if (other.isInteger())
               {
               result = new Numeric.Float(this.value.floatValue()
                                             / other.floatValue());
               }
            else if (other.isFloat())
               {
               result = new Numeric.Float(this.value.floatValue()
                                             / other.floatValue());
               }
             else if (other.isDouble())
               {
               result = new Numeric.Double(this.value.doubleValue()
                                              / other.doubleValue());
               }
             else if (other.isBounded())
               {
               /*
               *  The bounds combinations must be considered.  Any interval
               *  that includes zero for the denominator must be excluded.
               */
               double this_lower = this.getLower();
               double this_upper = this.getUpper();

               double other_lower = other.getLower();
               double other_upper = other.getUpper();

               double lower_lower = this_lower / other_lower;
               double lower_upper = this_lower / other_upper;
               double upper_lower = this_upper / other_lower;
               double upper_upper = this_upper / other_upper;

               double new_lower = Math.min(lower_lower, lower_upper);
               new_lower = Math.min(new_lower, upper_lower);
               new_lower = Math.min(new_lower, upper_upper);

               double new_upper = Math.max(lower_lower, lower_upper);
               new_upper = Math.max(new_upper, upper_lower);
               new_upper = Math.max(new_upper, upper_upper);

               result = new Numeric.Bounded(new_lower, new_upper);
               }
            }
         return(result);
         }
      }

   public static class Double extends Numeric
      {
      java.lang.Double value = null;
      public Double(double value)
         {
         this.value = value;
         }
      public String toString()
         {
         String out_string = "" + this.value;
         return(out_string);
         }
      public void Print()
         {
         System.out.print("" + this.value);
         }

      public boolean isInteger()
         {
         return(false);
         }
      public boolean isFloat()
         {
         return(false);
         }
      public boolean isDouble()
         {
         return(true);
         }
      public boolean isBounded()
         {
         return(false);
         }
      public boolean isNonZero()
         {
         return(this.value != 0.0);
         }

      public java.lang.Integer intValue()
         {
         return(this.value.intValue());
         }

      public java.lang.Float floatValue()
         {
         return(this.value.floatValue());
         }

      public java.lang.Double doubleValue()
         {
         return(this.value.doubleValue());
         }

      public Vector2D boundedValue()
         {
         return(new Vector2D(this.getLower(), this.getUpper()));
         }

      public java.lang.Double getLower()
         {
         return(this.value.doubleValue());
         }

      public java.lang.Double getUpper()
         {
         return(this.value.doubleValue());
         }


      public Numeric Add(Numeric other)
         {
         Numeric result = null;
         if (other.isInteger())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           + other.doubleValue());
            }
         else if (other.isFloat())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           + other.doubleValue());
            }
          else if (other.isDouble())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           + other.doubleValue());
            }
          else if (other.isBounded())
            {
            result = new Numeric.Bounded(this.getLower() + other.getLower(),
                                         this.getUpper() + other.getUpper());
            }
         return(result);
         }

      public Numeric Subtract(Numeric other)
         {
         Numeric result = null;
         if (other.isInteger())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           - other.doubleValue());
            }
         else if (other.isFloat())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           - other.doubleValue());
            }
          else if (other.isDouble())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           - other.doubleValue());
            }
          else if (other.isBounded())
            {
            /*
            *  The other's bounds are reversed.
            */
            result = new Numeric.Bounded(this.getLower() - other.getUpper(),
                                         this.getUpper() - other.getLower());
            }
         return(result);
         }

      public Numeric Multiply(Numeric other)
         {
         Numeric result = null;
         if (other.isInteger())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           * other.doubleValue());
            }
         else if (other.isFloat())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           * other.doubleValue());
            }
          else if (other.isDouble())
            {
            result = new Numeric.Double(this.value.doubleValue()
                                           * other.doubleValue());
            }
          else if (other.isBounded())
            {
            /*
            *  The bounds combinations must be considered.
            *    [++][++] ==> no problem.
            *    [++][-+] ==> no problem.
            *    [++][--] ==> reverse order.
            *    [-+][++] ==> reverse order.
            *    [-+][-+] ==> reverse order.
            *    [-+][--] ==> reverse order.
            *    [--][++] ==> reverse order.
            *    [--][-+] ==> reverse order.
            *    [--][--] ==> no problem.
            */
            double this_lower = this.getLower();
            double this_upper = this.getUpper();

            double other_lower = other.getLower();
            double other_upper = other.getUpper();

            double lower_lower = this_lower * other_lower;
            double lower_upper = this_lower * other_upper;
            double upper_lower = this_upper * other_lower;
            double upper_upper = this_upper * other_upper;

            double new_lower = Math.min(lower_lower, lower_upper);
            new_lower = Math.min(new_lower, upper_lower);
            new_lower = Math.min(new_lower, upper_upper);

            double new_upper = Math.max(lower_lower, lower_upper);
            new_upper = Math.max(new_upper, upper_lower);
            new_upper = Math.max(new_upper, upper_upper);

            result = new Numeric.Bounded(new_lower, new_upper);
            }
         return(result);
         }

      public Numeric Divide(Numeric other)
         {
         Numeric result = null;
         /*
         *  If the dividend is zero or is an interval that includes zero then
         *  the result will be infinite, so null will be returned.
         */
         if (other.isNonZero())
            {
            if (other.isInteger())
               {
               result = new Numeric.Double(this.value.doubleValue()
                                              / other.doubleValue());
               }
            else if (other.isFloat())
               {
               result = new Numeric.Double(this.value.doubleValue()
                                              / other.doubleValue());
               }
             else if (other.isDouble())
               {
               result = new Numeric.Double(this.value.doubleValue()
                                              / other.doubleValue());
               }
             else if (other.isBounded())
               {
               /*
               *  The bounds combinations must be considered.  Any interval
               *  that includes zero for the denominator must be excluded.
               */
               double this_lower = this.getLower();
               double this_upper = this.getUpper();

               double other_lower = other.getLower();
               double other_upper = other.getUpper();

               double lower_lower = this_lower / other_lower;
               double lower_upper = this_lower / other_upper;
               double upper_lower = this_upper / other_lower;
               double upper_upper = this_upper / other_upper;

               double new_lower = Math.min(lower_lower, lower_upper);
               new_lower = Math.min(new_lower, upper_lower);
               new_lower = Math.min(new_lower, upper_upper);

               double new_upper = Math.max(lower_lower, lower_upper);
               new_upper = Math.max(new_upper, upper_lower);
               new_upper = Math.max(new_upper, upper_upper);

               result = new Numeric.Bounded(new_lower, new_upper);
               }
            }
         return(result);
         }
      }

   public static class Bounded extends Numeric
      {
      java.lang.Double lower = null;
      java.lang.Double upper = null;
      public Bounded(double lower, double upper)
         {
         this.lower = lower;
         this.upper = upper;
         }
      public String toString()
         {
         String out_string = "[" + this.lower + ", " + this.upper + "]";
         return(out_string);
         }
      public void Print()
         {
         System.out.print("[" + this.lower + ", " + this.upper + "]");
         }

      public boolean isInteger()
         {
         return(false);
         }
      public boolean isFloat()
         {
         return(false);
         }
      public boolean isDouble()
         {
         return(false);
         }
      public boolean isBounded()
         {
         return(true);
         }
      public boolean isNonZero()
         {
         /*
         *  This indicates it is not null and the interval does not contain
         *  zero.
         */
         return(this.lower * this.upper > 0.0);
         }

      public java.lang.Integer intValue()
         {
         return(null);
         }

      public java.lang.Float floatValue()
         {
         return(null);
         }

      public java.lang.Double doubleValue()
         {
         return(null);
         }

      public Vector2D boundedValue()
         {
         return(new Vector2D(this.lower, this.upper));
         }

      public java.lang.Double getLower()
         {
         return(this.lower);
         }

      public java.lang.Double getUpper()
         {
         return(this.upper);
         }


      public Numeric Add(Numeric other)
         {
         Numeric result = null;
         /*
         *  The other's bounds are reversed.
         */
         result = new Numeric.Bounded(this.getLower() + other.getLower(),
                                      this.getUpper() + other.getUpper());
         return(result);
         }

      public Numeric Subtract(Numeric other)
         {
         Numeric result = null;

         /*
         *  The other's bounds are reversed.
         */
         result = new Numeric.Bounded(this.getLower() - other.getUpper(),
                                      this.getUpper() - other.getLower());
         return(result);
         }

      public Numeric Multiply(Numeric other)
         {
         Numeric result = null;

         /*
         *  The bounds combinations must be considered.
         *    [++][++] ==> no problem.
         *    [++][-+] ==> no problem.
         *    [++][--] ==> reverse order.
         *    [-+][++] ==> reverse order.
         *    [-+][-+] ==> reverse order.
         *    [-+][--] ==> reverse order.
         *    [--][++] ==> reverse order.
         *    [--][-+] ==> reverse order.
         *    [--][--] ==> no problem.
         */
         double this_lower = this.getLower();
         double this_upper = this.getUpper();

         double other_lower = other.getLower();
         double other_upper = other.getUpper();

         double lower_lower = this_lower * other_lower;
         double lower_upper = this_lower * other_upper;
         double upper_lower = this_upper * other_lower;
         double upper_upper = this_upper * other_upper;

         double new_lower = Math.min(lower_lower, lower_upper);
         new_lower = Math.min(new_lower, upper_lower);
         new_lower = Math.min(new_lower, upper_upper);

         double new_upper = Math.max(lower_lower, lower_upper);
         new_upper = Math.max(new_upper, upper_lower);
         new_upper = Math.max(new_upper, upper_upper);

         result = new Numeric.Bounded(new_lower, new_upper);

         return(result);
         }

      public Numeric Divide(Numeric other)
         {
         Numeric result = null;
         /*
         *  If the dividend is zero or is an interval that includes zero then
         *  the result will be infinite, so null will be returned.
         */
         if (other.isNonZero())
            {
            /*
            *  The bounds combinations must be considered.  Any interval
            *  that includes zero for the denominator must be excluded.
            */
            double this_lower = this.getLower();
            double this_upper = this.getUpper();

            double other_lower = other.getLower();
            double other_upper = other.getUpper();

            double lower_lower = this_lower / other_lower;
            double lower_upper = this_lower / other_upper;
            double upper_lower = this_upper / other_lower;
            double upper_upper = this_upper / other_upper;

            double new_lower = Math.min(lower_lower, lower_upper);
            new_lower = Math.min(new_lower, upper_lower);
            new_lower = Math.min(new_lower, upper_upper);

            double new_upper = Math.max(lower_lower, lower_upper);
            new_upper = Math.max(new_upper, upper_lower);
            new_upper = Math.max(new_upper, upper_upper);

            result = new Numeric.Bounded(new_lower, new_upper);
            }
         return(result);
         }
      }

   public static void main(String [] args)
      {
      Numeric.Integer i1 = new Numeric.Integer(5);
      Numeric.Integer i2 = new Numeric.Integer(2);
      System.out.println("Integer test:");
      System.out.println("   i1 = " + i1);
      System.out.println("   i2 = " + i2);
      System.out.println("   i1 + i2 = " + i1.Add(i2));
      System.out.println("   i1 - i2 = " + i1.Subtract(i2));
      System.out.println("   i1 * i2 = " + i1.Multiply(i2));
      System.out.println("   i1 / i2 = " + i1.Divide(i2));
      System.out.println();

      Numeric.Float f1 = new Numeric.Float(7.265F);
      Numeric.Float f2 = new Numeric.Float(5.735F);
      System.out.println("Float test:");
      System.out.println("   f1 = " + f1);
      System.out.println("   f2 = " + f2);
      System.out.println("   f1 + f2 = " + f1.Add(f2));
      System.out.println("   f1 - f2 = " + f1.Subtract(f2));
      System.out.println("   f1 * f2 = " + f1.Multiply(f2));
      System.out.println("   f1 / f2 = " + f1.Divide(f2));
      System.out.println();

      Numeric.Double d1 = new Numeric.Double(2.735);
      Numeric.Double d2 = new Numeric.Double(3.278);
      System.out.println("Double test:");
      System.out.println("   d1 = " + d1);
      System.out.println("   d2 = " + d2);
      System.out.println("   d1 + d2 = " + d1.Add(d2));
      System.out.println("   d1 - d2 = " + d1.Subtract(d2));
      System.out.println("   d1 * d2 = " + d1.Multiply(d2));
      System.out.println("   d1 / d2 = " + d1.Divide(d2));
      System.out.println();

      Numeric.Bounded b1 = new Numeric.Bounded(1.07 - 0.005, 1.07 + 0.005);
      Numeric.Bounded b2 = new Numeric.Bounded(1.07 - 0.005, 1.07 + 0.005);
      System.out.println("Bounded test:\n");
      System.out.println("   b1 = " + b1);
      System.out.println("   b2 = " + b2);
      System.out.println("   b1 + b2 = " + b1.Add(b2));
      System.out.println("   b1 - b2 = " + b1.Subtract(b2));
      System.out.println("   b1 * b2 = " + b1.Multiply(b2));
      System.out.println("   b1 / b2 = " + b1.Divide(b2));
      System.out.println();
      }
   }
