
package util.geom;

import java.util.*;
import java.io.*;

/**
*  This class allows storing 3-D box descriptions, and all edges are
*  parallel to the x, y, and z axes of the coordinate system.  The face,
*  edge, and node indexing in the box follow the conventions commonly used
*  in finite elements implementations.
*  <pre>
*             3        2
*              *--------*
*             /|       /|
*         0  / |    1 / |
*           *--------*  |
*           |  *-----|--*
*           | / 7    | / 6
*           |/       |/
*           *--------*
*         4          5
*
*     Node position        Node index
*     upper left front         0
*     upper right front        1
*     upper right back         2
*     upper left back          3
*     lower left front         4
*     lower right front        5
*     lower right back         6
*     lower left back          7
*
*     Edge position        Edge index
*      upper front             0
*      upper right             1
*      upper back              2
*      upper left              3
*      middle right front      4
*      middle left front       5
*      middle left back        6
*      middle right back       7
*      lower front             8
*      lower right             9
*      lower back              10
*      lower left              11
*
*     Face position        Face index
*        Front                 0
*        Right                 1
*        Back                  2
*        Left                  3
*        Top                   4
*        Bottom                5
*
*  </pre>
*/
public class Box3D
   {

   /**
   *  The coordinates of the upper left corner of the box.
   */
   public Vector3D ulc = null;

   /**
   *  The width (x-size) of the box.
   */
   double width = 1.0;

   /**
   *  The height (y-size) of the box.
   */
   double height = 1.0;

   /**
   *  The depth (z-size) of the box.
   */
   double depth = 1.0;

   /**
   * The corner nodes as coordinate vectors, only created when requested.
   */
   Vector3D [] nodes = null;

   /**
   *  Creates a unit Box3D object with (0, 0, 0) as the front upper left
   *  corner coordinate values, and (1, 1, 1) as the width, height, and depth.
   */
   public Box3D()
      {
      this.ulc = new Vector3D();
      }

   /**
   *  Creates a Box3D object with (x, y, and z) coordinate values and
   *  the specified (width, height, depth) sizes.
   *  @param x the x-coordinate.
   *  @param y the y-coordinate.
   *  @param z the z-coordinate.
   *  @param width the width.
   *  @param height the height.
   *  @param depth the depth.
   */
   public Box3D(double x,
                double y,
                double z,
                double width,
                double height,
                double depth)
      {
      this.ulc = new Vector3D(x, y, z);

      this.width = width;
      this.height = height;
      this.depth = depth;
      }

   /**
   *  Creates a Box3D object from a Box3D argument's coordinate
   *  values and sizes.
   *  @param box the other box.
   */
   public Box3D(Box3D box)
      {
      this.ulc = new Vector3D(box.ulc);

      this.width = box.width;
      this.height = box.height;
      this.depth = box.depth;
      }

   /**
   * Creates a new Box3D object from the Box3D's coordinate values.
   * @return the new Box3D object reference.
   */
   public Box3D clone()
      {
      Box3D box = new Box3D(this);
      return(box);
      }

   /**
   * Checks to see if the boxes are equal.
   * @return true if it is the same or an equivalent box.
   */
   public boolean equals(Box3D box)
      {
      boolean is_equal = false;
      if (this == box)
         {
         is_equal = true;
         }
      else if (this.ulc.equals(box.ulc) &&
               this.width == box.width &&
               this.height == box.height &&
               this.depth == box.depth)
         {
         is_equal = true;
         }
      return(is_equal);
      }

   /**
   * Checks to see if the boxes are the same size in each dimension.
   * @return true if it is they are the same size.
   */
   public boolean sameSize(Box3D box)
      {
      boolean is_same = false;
      if (this == box)
         {
         is_same = true;
         }
      else if (this.width == box.width &&
               this.height == box.height &&
               this.depth == box.depth)
         {
         is_same = true;
         }
      return(is_same);
      }

   /**
   *  Gets the Box3D upper left corner's x coordinate.
   *  @return the x-coordinate for the upper left corner.
   */
   public Double getX()
      {
      return(this.ulc.x);
      }

   /**
   *  Gets the Box3D upper left corner's y coordinate.
   *  @return the y-coordinate for the upper left corner.
   */
   public Double getY()
      {
      return(this.ulc.y);
      }

   /**
   *  Gets the Box3D upper left corner's z coordinate.
   *  @return the z-coordinate for the upper left corner.
   */
   public Double getZ()
      {
      return(this.ulc.z);
      }

   /**
   *  Gets the Box3D extent along the x-axis (width).
   *  @return the width of the box.
   */
   public Double getWidth()
      {
      return(this.width);
      }

   /**
   *  Gets the Box3D extent along the y-axis (height).
   *  @return the height of the box.
   */
   public Double getHeight()
      {
      return(this.height);
      }

   /**
   *  Gets the Box3D extent along the z-axis (depth).
   *  @return the depth of the box.
   */
   public Double getDepth()
      {
      return(this.depth);
      }

   /**
   *  Gets the Box3D upper left corner as a Vector3D.
   *  @return the Vector3D for the upper left corner.
   */
   public Vector3D getULC()
      {
      return(this.ulc);
      }

   /**
   *  Gets the Box3D extents from the uppper left corner as a Vector3D.
   *  @return the Vector3D for the extents from the upper left corner.
   */
   public Vector3D getExtents()
      {
      Vector3D extents = new Vector3D( this.width, this.height, this.depth);
      return(extents);
      }

   /**
   *  Sets the Box3D upper left corner to the given x, y, and z coordinate
   *  values.
   *  @param x the x-coordinate.
   *  @param y the y-coordinate.
   *  @param z the z-coordinate.
   */
   public void setULC(double x, double y, double z)
      {
      this.ulc.x = x;
      this.ulc.y = y;
      this.ulc.z = z;

      this.nodes = null;
      }

   /**
   *  Sets the Box3D upper left corner to the x, y, and Z coordinate
   *  values of the coords.
   *  @param coords the coords.
   */
   public void setLocation(Vector3D coords)
      {
      this.ulc.x = coords.x;
      this.ulc.y = coords.y;
      this.ulc.z = coords.z;

      this.nodes = null;
      }

   /**
   *  Sets the Box3D size to the width, height, and depth values.
   *  @param width the width.
   *  @param height the height.
   *  @param depth the depth.
   */
   public void setSize(double width, double height, double depth)
      {
      this.width = width;
      this.height = height;
      this.depth = depth;

      this.nodes = null;
      }

   /**
   *  Creates a Face3D object by getting the face by index from the box,
   *  and then using the face center and normal for the point and orientation.
   *     <pre>
   *  Node index designations:
   *             3        2
   *              *--------*
   *             /|       /|
   *         0  / |    1 / |
   *           *--------*  |
   *           |  *-----|--*
   *           | / 7    | / 6
   *           |/       |/
   *           *--------*
   *         4          5
   *
   *
   *     Face position        Face index   Node indices
   *        Front                 0         (0, 1, 5, 4)
   *        Right                 1         (1, 2, 6, 5)
   *        Back                  2         (2, 3, 7, 6)
   *        Left                  3         (3, 0, 4, 7)
   *        Top                   4         (3, 2, 1, 0)
   *        Bottom                5         (4, 5, 6, 7)
   *
   *  </pre>
   *  @param index the index.
   *  @return the array of node points for the face.
   */
   public Vector3D [] getFace(int index)
      {
      Vector3D [] face = null;
      if (this.nodes == null)
         {
         this.deriveNodes();
         }
      switch (index)
         {
         case 0:
            /* Front */
            face = new Vector3D[4];
            face[0] = this.nodes[0];
            face[1] = this.nodes[1];
            face[2] = this.nodes[5];
            face[3] = this.nodes[4];
            break;
         case 1:
            /* Right */
            face = new Vector3D[4];
            face[0] = this.nodes[1];
            face[1] = this.nodes[2];
            face[2] = this.nodes[6];
            face[3] = this.nodes[5];
            break;
         case 2:
            /* Back */
            face = new Vector3D[4];
            face[0] = this.nodes[2];
            face[1] = this.nodes[3];
            face[2] = this.nodes[7];
            face[3] = this.nodes[6];
            break;
         case 3:
            /* Left */
            face = new Vector3D[4];
            face[0] = this.nodes[3];
            face[1] = this.nodes[0];
            face[2] = this.nodes[4];
            face[3] = this.nodes[7];
            break;
         case 4:
            /* Top */
            face = new Vector3D[4];
            face[0] = this.nodes[3];
            face[1] = this.nodes[2];
            face[2] = this.nodes[1];
            face[3] = this.nodes[0];
            break;
         case 5:
            /* Bottom */
            face = new Vector3D[4];
            face[0] = this.nodes[4];
            face[1] = this.nodes[5];
            face[2] = this.nodes[6];
            face[3] = this.nodes[7];
            break;
         default:
            break;
         }
      return(face);
      }

   /**
   *  Creates a Edge3D object by getting the edge by index from the box,
   *  and then returning the edge endpoints.
   * <pre>
   *  Node index designations:
   *             3        2
   *              *--------*
   *             /|       /|
   *         0  / |    1 / |
   *           *--------*  |
   *           |  *-----|--*
   *           | / 7    | / 6
   *           |/       |/
   *           *--------*
   *         4          5
   *
   *
   *     Edge position        Edge index      Node indices
   *      upper front             0             (0, 1)
   *      upper right             1             (1, 2)
   *      upper back              2             (2, 3)
   *      upper left              3             (3, 0)
   *      middle right front      4             (0, 4)
   *      middle left front       5             (1, 5)
   *      middle left back        6             (2, 6)
   *      middle right back       7             (3, 7)
   *      lower front             8             (4, 5)
   *      lower right             9             (5, 6)
   *      lower back              10            (6, 7)
   *      lower left              11            (7, 0)
   *
   * </pre>
   *  @param index the index.
   *  @return the array of node points for the edge.
   */
   public Vector3D [] getEdge(int index)
      {
      Vector3D [] edge = null;
      if (this.nodes == null)
         {
         this.deriveNodes();
         }
      switch (index)
         {
         case 0:
            edge = new Vector3D[2];
            edge[0] = this.nodes[0];
            edge[1] = this.nodes[1];
            break;
         case 1:
            edge = new Vector3D[2];
            edge[0] = this.nodes[1];
            edge[1] = this.nodes[2];
            break;
         case 2:
            edge = new Vector3D[2];
            edge[0] = this.nodes[2];
            edge[1] = this.nodes[3];
            break;
         case 3:
            edge = new Vector3D[2];
            edge[0] = this.nodes[3];
            edge[1] = this.nodes[0];
            break;
         case 4:
            edge = new Vector3D[2];
            edge[0] = this.nodes[0];
            edge[1] = this.nodes[4];
            break;
         case 5:
            edge = new Vector3D[2];
            edge[0] = this.nodes[1];
            edge[1] = this.nodes[5];
            break;
         case 6:
            edge = new Vector3D[2];
            edge[0] = this.nodes[2];
            edge[1] = this.nodes[6];
            break;
         case 7:
            edge = new Vector3D[2];
            edge[0] = this.nodes[3];
            edge[1] = this.nodes[7];
            break;
         case 8:
            edge = new Vector3D[2];
            edge[0] = this.nodes[4];
            edge[1] = this.nodes[5];
            break;
         case 9:
            edge = new Vector3D[2];
            edge[0] = this.nodes[5];
            edge[1] = this.nodes[6];
            break;
         case 10:
            edge = new Vector3D[2];
            edge[0] = this.nodes[6];
            edge[1] = this.nodes[7];
            break;
         case 11:
            edge = new Vector3D[2];
            edge[0] = this.nodes[7];
            edge[1] = this.nodes[0];
            break;
         default:
            break;
         }
      return(edge);
      }

   /**
   *  Gets a Vector3D object for the node point from the node's index in the
   *  box,
   * <pre>
   *  Node index designations:
   *             3        2
   *              *--------*
   *             /|       /|
   *         0  / |    1 / |
   *           *--------*  |
   *           |  *-----|--*
   *           | / 7    | / 6
   *           |/       |/
   *           *--------*
   *         4          5
   *
   *
   *     Node position        Node index
   *     upper left front         0
   *     upper right front        1
   *     upper right back         2
   *     upper left back          3
   *     lower left front         4
   *     lower right front        5
   *     lower right back         6
   *     lower left back          7
   *
   * </pre>
   *  @param index the index.
   */
   public Vector3D getNode(int index)
      {
      if (this.nodes == null)
         {
         this.deriveNodes();
         }
      return(this.nodes[index]);
      }

   /**
   *  Creates a Vector3D object by getting the node by index from the box,
   *  and then returning the node coordinates.
   * <pre>
   *  Node index designations:
   *             3        2
   *              *--------*
   *             /|       /|
   *         0  / |    1 / |
   *           *--------*  |
   *           |  *-----|--*
   *           | / 7    | / 6
   *           |/       |/
   *           *--------*
   *         4          5
   *
   *
   *     Node position        Node index
   *     upper left front         0
   *     upper right front        1
   *     upper right back         2
   *     upper left back          3
   *     lower left front         4
   *     lower right front        5
   *     lower right back         6
   *     lower left back          7
   *
   * </pre>
   *  @param index the index.
   */
   public Vector3D deriveNode(int index)
      {
      Vector3D node = null;
      switch (index)
         {
         case 0:
            node = new Vector3D(this.ulc.x,
                               this.ulc.y,
                               this.ulc.z);
            break;
         case 1:
            node = new Vector3D(this.ulc.x + this.width,
                               this.ulc.y,
                               this.ulc.z);
            break;
         case 2:
            node = new Vector3D(this.ulc.x + this.width,
                               this.ulc.y,
                               this.ulc.z + this.depth);
            break;
         case 3:
            node = new Vector3D(this.ulc.x,
                               this.ulc.y,
                               this.ulc.z + this.depth);
            break;
         case 4:
            node = new Vector3D(this.ulc.x,
                               this.ulc.y + this.height,
                               this.ulc.z);
            break;
         case 5:
            node = new Vector3D(this.ulc.x + this.width,
                               this.ulc.y + this.height,
                               this.ulc.z);
            break;
         case 6:
            node = new Vector3D(this.ulc.x + this.width,
                               this.ulc.y + this.height,
                               this.ulc.z + this.depth);
            break;
         case 7:
            node = new Vector3D(this.ulc.x,
                               this.ulc.y + this.height,
                               this.ulc.z + this.depth);
            break;
         default:
            break;
         }
      return(node);
      }

   /**
   *  Creates the nodes points array so the points don't have to be
   *  recreated for each node, edge, and face request.
   * <pre>
   *  Node index designations:
   *             3        2
   *              *--------*
   *             /|       /|
   *         0  / |    1 / |
   *           *--------*  |
   *           |  *-----|--*
   *           | / 7    | / 6
   *           |/       |/
   *           *--------*
   *         4          5
   *
   *
   *     Node position        Node index
   *     upper left front         0
   *     upper right front        1
   *     upper right back         2
   *     upper left back          3
   *     lower left front         4
   *     lower right front        5
   *     lower right back         6
   *     lower left back          7
   *
   * </pre>
   *
   */
   public void deriveNodes()
      {
      this.nodes = new Vector3D[8];
      for (int index = 0;
           index < 8;
           index++)
         {
         nodes[index] = deriveNode(index);
         }
      }

   /**
   *  Returns true if the Box3D contains the coords.
   *  @param x the x-coordinate.
   *  @param y the y-coordinate.
   *  @param z the z-coordinate.
   */
   public boolean contains(double x, double y, double z)
      {
      boolean contains_coords = true;

      if (x < this.ulc.x || y < this.ulc.y || z < this.ulc.z)
         {
         contains_coords = false;
         }
      else if (x >= this.ulc.x + this.width ||
               y >= this.ulc.y + this.height ||
               z >= this.ulc.z + this.depth)
         {
         contains_coords = false;
         }

      return(contains_coords);
      }

   /**
   *  Returns true if the Box3D contains the coords.
   *  @param coords the coords.
   */
   public boolean contains(Vector3D coords)
      {
      boolean contains_coords = true;

      if (coords.x < this.ulc.x ||
          coords.y < this.ulc.y ||
          coords.z < this.ulc.z)
         {
         contains_coords = false;
         }
      else if (coords.x >= this.ulc.x + this.width ||
               coords.y >= this.ulc.y + this.height ||
               coords.z >= this.ulc.z + this.depth)
         {
         contains_coords = false;
         }

      return(contains_coords);
      }

   /**
   *  Returns the intersection box or null.
   *  @param box the other box.
   *  @return the intersection or null if there is no intersection.
   */
   public Box3D intersect(Box3D box)
      {
      Box3D result = null;

      if (box.ulc.x + box.width <= this.ulc.x ||
          box.ulc.y + box.height <= this.ulc.y ||
          box.ulc.z + box.depth <= this.ulc.z)
         {
         /*
         *  They do not intersect.
         */
         }
      else if (this.ulc.x + this.width <= box.ulc.x ||
               this.ulc.y + this.height <= box.ulc.y ||
               this.ulc.z + this.depth <= box.ulc.z)
         {
         /*
         *  They do not intersect.
         */
         }
      else
         {
         /*
         *  They intersect, so get the parameters needed.
         */
         double x = Math.max(this.ulc.x, box.ulc.x);
         double y = Math.max(this.ulc.y, box.ulc.y);
         double z = Math.max(this.ulc.z, box.ulc.z);

         double width = Math.min(this.ulc.x + this.width,
                                 box.ulc.x + box.width) - x;
         double height = Math.min(this.ulc.y + this.height,
                                  box.ulc.y + box.height) - y;
         double depth = Math.min(this.ulc.z + this.depth,
                                 box.ulc.z + box.depth) - z;

         if (width > 0.0 &&
             height > 0.0 &&
             depth > 0.0)
            {
            result = new Box3D(x, y, z, width, height, depth);
            }
         }

      return(result);
      }

   /*
   *  Convert the information about the box into a string.
   */
   public String toString()
      {
      String out_string = "("
                        + this.ulc.x
                        + ", "
                        + this.ulc.y
                        + ", "
                        + this.ulc.z
                        + "),("
                        + this.width
                        + ", "
                        + this.height
                        + ", "
                        + this.depth
                        + ")"
                         ;
      return(out_string);
      }

   public void Print()
      {
      System.out.print(this);
      }
   }

