
package util.geom;

import java.util.*;
import java.io.*;

import util.*;

/**
*  This class allows storing and manipulating 3x3 matrices especially for 
*  support of computer graphics as in OpenGL.
*/
public class Matrix_3x3
   {

   public double [] matrix = new double [9];

   /**
   *  Creates an identity Matrix_3x3.
   */
   public static Matrix_3x3 GetIdentity()
      {
      Matrix_3x3 ident = new Matrix_3x3();
      ident.matrix[0] = 1.0;
      ident.matrix[4] = 1.0;
      ident.matrix[8] = 1.0;

      return(ident);
      }


   /**
   *  For support of graphics this generates a graphics rotation matrix
   *  for rotating an augmented 2-D vector about the origin in a
   *  counterclockwise (looking from the origin) direction around the x axis
   *  by the given angle (in radians).
   *  <pre>
   *         rot(V, a) = V * M(a)
   *  </pre>
   *  @param angle the angle of rotation in radians.
   *  @return the rotation transform, M(a).
   */
   public static Matrix_3x3 getXRotated(double angle)
      {
      Matrix_3x3 out = Matrix_3x3.GetIdentity();
      /*
      *  x' = x
      *  y' = y cos(a) + z sin(a)
      *  z' = -y sin(a) + z cos(a)
      */
      out.matrix[4] = Math.cos(angle);
      out.matrix[7] = Math.sin(angle);

      out.matrix[5] = -Math.sin(angle);
      out.matrix[8] = Math.cos(angle);

      return(out);
      }


   /**
   *  For support of graphics this generates a graphics rotation matrix
   *  for rotating an augmented 2-D vector about the origin in a
   *  counterclockwise (looking from the origin) direction around the y axis
   *  by the given angle (in radians).
   *  <pre>
   *         rot(V, a) = V * M(a)
   *  </pre>
   *  @param angle the angle of rotation in radians.
   *  @return the rotation transform, M(a).
   */
   public static Matrix_3x3 getYRotated(double angle)
      {
      Matrix_3x3 out = Matrix_3x3.GetIdentity();
      /*
      *  x' = x cos(a) - z sin(a)
      *  y' = y
      *  z' = x sin(a) + z cos(a)
      */
      out.matrix[0] = Math.cos(angle);
      out.matrix[6] = -Math.sin(angle);

      out.matrix[2] = Math.sin(angle);
      out.matrix[8] = Math.cos(angle);

      return(out);
      }


   /**
   *  For support of graphics this generates a graphics rotation matrix
   *  for rotating an augmented 2-D vector about the origin in a
   *  counterclockwise (looking from the origin) direction around the z axis
   *  by the given angle (in radians).
   *  <pre>
   *         rot(V, a) = V * M(a)
   *  </pre>
   *  @param angle the angle of rotation in radians.
   *  @return the rotation transform, M(a).
   */
   public static Matrix_3x3 getZRotated(double angle)
      {
      Matrix_3x3 out = Matrix_3x3.GetIdentity();
      /*
      *  x' = x cos(a) + y sin(a)
      *  y' = -x sin(a) + y cos(a)
      *  z' = z
      */
      out.matrix[0] = Math.cos(angle);
      out.matrix[3] = Math.sin(angle);

      out.matrix[1] = -Math.sin(angle);
      out.matrix[4] = Math.cos(angle);

      return(out);
      }


   /**
   *  Creates a Matrix_3x3 object with 0.0 as the entry values.
   */
   public Matrix_3x3()
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = 0.0;
         }
      }


   /**
   *  Creates a Matrix_3x3 object from a Matrix_3x3 argument's component values.
   *  @param in the other matrix.
   */
   public Matrix_3x3(Matrix_3x3 in)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = in.matrix[entry_index];
         }
      }


   /**
   *  Creates a Matrix_3x3 object from the array of values in row-major
   *  form.
   *  @param in_array the array of values.
   */
   public Matrix_3x3(double [] in_array)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = in_array[entry_index];
         }
      }


   /**
   *  Creates a Matrix_3x3 object with the given entry values.
   *  @param m00 the (0, 0) entry value;
   *  @param m01 the (0, 1) entry value;
   *  @param m02 the (0, 2) entry value;
   *  @param m10 the (1, 0) entry value;
   *  @param m11 the (1, 1) entry value;
   *  @param m12 the (1, 2) entry value;
   *  @param m20 the (2, 0) entry value;
   *  @param m21 the (2, 1) entry value;
   *  @param m22 the (2, 2) entry value;
   */
   public Matrix_3x3(double m00, double m01, double m02,
                     double m10, double m11, double m12,
                     double m20, double m21, double m22)
      {
      this.matrix[0] = m00;
      this.matrix[1] = m01;
      this.matrix[2] = m02;
      this.matrix[3] = m10;
      this.matrix[4] = m11;
      this.matrix[5] = m12;
      this.matrix[6] = m20;
      this.matrix[7] = m21;
      this.matrix[8] = m22;
      }


   /**
   * Creates a new Matrix_3x3 object from the Matrix_3x3's component values.
   * @return the new Matrix_3x3 object reference.
   */
   public Matrix_3x3 clone()
      {
      Matrix_3x3 out_matrix = new Matrix_3x3(this);
      return(out_matrix);
      }

   /**
   * Gets the designated entry of this Matrix_3x3.
   * @param row the matrix row for the entry.
   * @param column the matrix column for the entry.
   * @return the designated entry component value.
   */
   public double getEntry(int row, int column)
      {
      return(this.matrix[row * 3 + column]);
      }

   /**
   * Sets the designated entry of this Matrix_3x3.
   * @param row the matrix row for the entry.
   * @param column the matrix column for the entry.
   * @param value the value to store.
   */
   public void setEntry(int row, int column, double value)
      {
      this.matrix[row * 3 + column] = value;
      }


   /**
   *  Sets the Matrix_3x3 object to the values of another matrix.
   *  @param in the other matrix.
   */
   public void set(Matrix_3x3 in)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = in.matrix[entry_index];
         }
      }


   /**
   *  Sets the Matrix_3x3 object to the values of an input array in row-major
   *  form.
   *  @param in_array the array of values.
   */
   public void set(double [] in_array)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = in_array[entry_index];
         }
      }


   /**
   *  Sets the Matrix_4x entry values to the given values.
   *  @param m00 the (0, 0) entry value;
   *  @param m01 the (0, 1) entry value;
   *  @param m02 the (0, 2) entry value;
   *  @param m10 the (1, 0) entry value;
   *  @param m11 the (1, 1) entry value;
   *  @param m12 the (1, 2) entry value;
   *  @param m20 the (2, 0) entry value;
   *  @param m21 the (2, 1) entry value;
   *  @param m22 the (2, 2) entry value;
   */
   public void set(double m00, double m01, double m02,
                   double m10, double m11, double m12,
                   double m20, double m21, double m22)
      {
      this.matrix[0] = m00;
      this.matrix[1] = m01;
      this.matrix[2] = m02;
      this.matrix[3] = m10;
      this.matrix[4] = m11;
      this.matrix[5] = m12;
      this.matrix[6] = m20;
      this.matrix[7] = m21;
      this.matrix[8] = m22;
      }


   /**
   *  Scales the entries in the matrix.
   *  @param scale the scale multiplier.
   */
   public void scale(double scale)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] *= scale;
         }
      }


   /**
   *  Add the input matrix to the current matrix.
   *  @param in the matrix to add.
   */
   public void add(Matrix_3x3 in)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] += in.matrix[entry_index];
         }
      }


   /**
   *  Subtract the input matrix from the current matrix.
   *  @param in the matrix to subtract.
   */
   public void subtract(Matrix_3x3 in)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] -= in.matrix[entry_index];
         }
      }


   /**
   *  Multiply this matrix by the given (right side) matrix.
   *  @param in the matrix to post-multiply.
   *  @return the product of the matrices.
   */
   public Matrix_3x3 multiply(Matrix_3x3 in)
      {
      Matrix_3x3 out_matrix = new
            Matrix_3x3(
                      matrix[0] * in.matrix[0] +
                         matrix[1] * in.matrix[3] +
                         matrix[2] * in.matrix[6],
                      matrix[0] * in.matrix[1] +
                         matrix[1] * in.matrix[4] +
                         matrix[2] * in.matrix[7],
                      matrix[0] * in.matrix[2] +
                         matrix[1] * in.matrix[5] +
                         matrix[2] * in.matrix[8],

                      matrix[3] * in.matrix[0] +
                         matrix[4] * in.matrix[3] +
                         matrix[5] * in.matrix[6],
                      matrix[3] * in.matrix[1] +
                         matrix[4] * in.matrix[4] +
                         matrix[5] * in.matrix[7],
                      matrix[3] * in.matrix[2] +
                         matrix[4] * in.matrix[5] +
                         matrix[5] * in.matrix[8],

                      matrix[6] * in.matrix[0] +
                         matrix[7] * in.matrix[3] +
                         matrix[8] * in.matrix[6],
                      matrix[6] * in.matrix[1] +
                         matrix[7] * in.matrix[4] +
                         matrix[8] * in.matrix[7],
                      matrix[6] * in.matrix[2] +
                         matrix[7] * in.matrix[5] +
                         matrix[8] * in.matrix[8]
                      );
      
      return(out_matrix);
      }


   /**
   *  For support of graphics this multiplies a Vector3D (augmented with a 1.0
   *  entry) times a Matrix_3x3 (v' X M), performing the 4-dimensional
   *  transformation to enable scaling, translation, and rotations.
   *  @param in_vector the vector (v).
   *  @param in the matrix (M).
   *  @return the resulting transformed vector.
   */
   public static Vector3D Multiply(Vector3D in_vector, Matrix_3x3 in)
      {
      Vector3D out_vector = new
           Vector3D(
                    in_vector.x * in.matrix[0] +
                    in_vector.y * in.matrix[3] +
                    in_vector.z * in.matrix[6],

                    in_vector.x * in.matrix[1] +
                    in_vector.y * in.matrix[4] +
                    in_vector.z * in.matrix[7],

                    in_vector.x * in.matrix[2] +
                    in_vector.y * in.matrix[5] +
                    in_vector.z * in.matrix[8]
                   );
      return(out_vector);
      }


   /**
   *  For support of graphics this multiplies (M x v) a Matrix_3x3 times a
   *  Vector3D (augmented with a 1.0 entry), performing the 4-dimensional
   *  transformation to enable scaling, translation, and rotations.
   *  @param in the matrix (M).
   *  @param in_vector the vector (v).
   *  @return the resulting transformed vector.
   */
   public static Vector3D Multiply(Matrix_3x3 in, Vector3D in_vector)
      {
      Vector3D out_vector = new
           Vector3D(
                    in.matrix[0] * in_vector.x +
                    in.matrix[1] * in_vector.y +
                    in.matrix[2] * in_vector.z,

                    in.matrix[3] * in_vector.x +
                    in.matrix[4] * in_vector.y +
                    in.matrix[5] * in_vector.z,

                    in.matrix[6] * in_vector.x +
                    in.matrix[7] * in_vector.y +
                    in.matrix[8] * in_vector.z
                   );

      return(out_vector);
      }

   /**
   * Returns a String that represents the value of this Matrix_3x3.
   * @return the string representation of the matrix.
   */
   public String toString()
      {
      String out_string = new String("");

      out_string = String.format("(%9.3g %9.3g %9.3g)\n"
                               + "(%9.3g %9.3g %9.3g)\n"
                               + "(%9.3g %9.3g %9.3g)\n",
                                 this.matrix[0],
                                 this.matrix[1],
                                 this.matrix[2],
                                 this.matrix[3],
                                 this.matrix[4],
                                 this.matrix[5],
                                 this.matrix[6],
                                 this.matrix[7],
                                 this.matrix[8]
                                );
      return(out_string);
      }

   /**
   * Prints the representation the value of this Matrix_3x3.
   */
   public void Print()
      {
      System.out.print(this);
      }
   }

