
package util.geom;

import java.util.*;
import java.io.*;

import util.*;

/**
*  This class allows storing and manipulating 4x4 matrices especially for 
*  support of computer graphics as in OpenGL.
*/
public class Matrix_4x4
   {

   public double [] matrix = new double [16];

   /**
   *  Creates an identity Matrix_4x4.
   */
   public static Matrix_4x4 GetIdentity()
      {
      Matrix_4x4 ident = new Matrix_4x4();
      ident.matrix[0] = 1.0;
      ident.matrix[5] = 1.0;
      ident.matrix[10] = 1.0;
      ident.matrix[15] = 1.0;

      return(ident);
      }


   /**
   *  For support of graphics this generates a graphics translation matrix.
   *  @param x_offset the x offset for the transformation.
   *  @param y_offset the y offset for the transformation.
   *  @param z_offset the z offset for the transformation.
   *  @return the translation matrix.
   */
   public static Matrix_4x4 GetTranslated(double x_offset,
                                          double y_offset,
                                          double z_offset)
      {
      Matrix_4x4 out = Matrix_4x4.GetIdentity();
      out.matrix[12] = x_offset;
      out.matrix[13] = y_offset;
      out.matrix[14] = z_offset;

      return(out);
      }


   /**
   *  For support of graphics this generates a graphics scaling matrix.
   *  @param x_scale the x scale for the transformation.
   *  @param y_scale the y scale for the transformation.
   *  @param z_scale the z scale for the transformation.
   */
   public static Matrix_4x4 GetScaled(double x_scale,
                                      double y_scale,
                                      double z_scale)
      {
      Matrix_4x4 out = Matrix_4x4.GetIdentity();
      out.matrix[0] = x_scale;
      out.matrix[5] = y_scale;
      out.matrix[10] = z_scale;

      return(out);
      }


   /**
   *  For support of graphics this generates a graphics rotation matrix
   *  for rotating about the x-axis by the given angle (in radians).
   *  @param angle the angle of rotation in radians.
   */
   public static Matrix_4x4 GetXRotated(double angle)
      {
      Matrix_4x4 out = Matrix_4x4.GetIdentity();
      out.matrix[5] = Math.cos(angle);
      out.matrix[6] = Math.sin(angle);
      out.matrix[9] = -Math.sin(angle);
      out.matrix[10] = Math.cos(angle);

      return(out);
      }


   /**
   *  For support of graphics this generates a graphics rotation matrix
   *  for rotating about the y-axis by the given angle (in radians).
   *  @param angle the angle of rotation in radians.
   */
   public static Matrix_4x4 GetYRotated(double angle)
      {
      Matrix_4x4 out = Matrix_4x4.GetIdentity();
      out.matrix[0] = Math.cos(angle);
      out.matrix[2] = -Math.sin(angle);
      out.matrix[8] = Math.sin(angle);
      out.matrix[10] = Math.cos(angle);

      return(out);
      }


   /**
   *  For support of graphics this generates a graphics rotation matrix
   *  for rotating about the z-axis by the given angle (in radians).
   *  @param angle the angle of rotation in radians.
   */
   public static Matrix_4x4 GetZRotated(double angle)
      {
      Matrix_4x4 out = Matrix_4x4.GetIdentity();
      out.matrix[0] = Math.cos(angle);
      out.matrix[1] = Math.sin(angle);
      out.matrix[4] = -Math.sin(angle);
      out.matrix[5] = Math.cos(angle);

      return(out);
      }


   /**
   *  For support of graphics this generates a graphics rotation matrix
   *  for rotating about the x-axis by the given angle (in radians).
   *  @param angle the angle of rotation in radians.
   *  @param axis the rotation center axis vector.
   */
   public static Matrix_4x4 GetAxisRotated(double angle,
                                           Vector3D axis)
      {
      Matrix_4x4 matrix = null;
      if (axis != null)
         {
         matrix = new Matrix_4x4();

         double sin_angle = Math.sin(angle);
         double cos_angle = Math.cos(angle);

         Vector3D norm_axis = new Vector3D(axis);
         double length = norm_axis.normalizeL2();

/*
         Matrix_3x3 norm_ext_prod_mat = norm_axis.exteriorProduct(norm_axis);
         Matrix_3x3 sum_mat = Matrix_3x3.GetIdentity();

         sum_mat = sum_mat.subtract(norm_ext_proc_mat);
         sum_mat.Scale(cos_angle);

         sum_mat = sum_mat.add(norm_ext_prod_mat);

         Matrix_3x3 sym_axis_mat = new Matrix_3x3(0.0,
                                                  -norm_axis.z,
                                                  norm_axis.y,
                                                  norm_axis.z,
                                                  0.0,
                                                  -norm_axis.x,
                                                  -norm_axis.y,
                                                  norm_axis.x,
                                                  0.0);
         sym_axis_mat.scale(cos_angle);
         sum_mat.add(sym_axis_mat);
*/

         double omc = 1.0 - cos_angle;
         matrix.set(norm_axis.x * norm_axis.x * omc + cos_angle,
                    norm_axis.x * norm_axis.y * omc - norm_axis.z * sin_angle,
                    norm_axis.x * norm_axis.z * omc + norm_axis.y * sin_angle,
                    0.0,
                    norm_axis.y * norm_axis.x * omc + norm_axis.z * sin_angle,
                    norm_axis.y * norm_axis.y * omc + cos_angle,
                    norm_axis.y * norm_axis.z * omc - norm_axis.x * sin_angle,
                    0.0,
                    norm_axis.z * norm_axis.x * omc - norm_axis.y * sin_angle,
                    norm_axis.z * norm_axis.y * omc + norm_axis.x * sin_angle,
                    norm_axis.z * norm_axis.z * omc + cos_angle,
                    0.0,
                    0.0,
                    0.0,
                    0.0,
                    1.0
                   );
         }

      return(matrix);
      }


   /**
   *  Creates a Matrix_4x4 object with 0.0 as the entry values.
   */
   public Matrix_4x4()
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = 0.0;
         }
      }


   /**
   *  Creates a Matrix_4x4 object from a Matrix_4x4 argument's component values.
   *  @param in the other matrix.
   */
   public Matrix_4x4(Matrix_4x4 in)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = in.matrix[entry_index];
         }
      }


   /**
   *  Creates a Matrix_4x4 object from the array of values in row-major
   *  form.
   *  @param in_array the array of values.
   */
   public Matrix_4x4(double [] in_array)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = in_array[entry_index];
         }
      }


   /**
   *  Creates a Matrix_4x4 object with the given entry values.
   *  @param m00 the (0, 0) entry value;
   *  @param m01 the (0, 1) entry value;
   *  @param m02 the (0, 2) entry value;
   *  @param m03 the (0, 3) entry value;
   *  @param m10 the (1, 0) entry value;
   *  @param m11 the (1, 1) entry value;
   *  @param m12 the (1, 2) entry value;
   *  @param m13 the (1, 3) entry value;
   *  @param m20 the (2, 0) entry value;
   *  @param m21 the (2, 1) entry value;
   *  @param m22 the (2, 2) entry value;
   *  @param m23 the (2, 3) entry value;
   *  @param m30 the (3, 0) entry value;
   *  @param m31 the (3, 1) entry value;
   *  @param m32 the (3, 2) entry value;
   *  @param m33 the (3, 3) entry value;
   */
   public Matrix_4x4(double m00, double m01, double m02, double m03,
                     double m10, double m11, double m12, double m13,
                     double m20, double m21, double m22, double m23,
                     double m30, double m31, double m32, double m33)
      {
      this.matrix[0] = m00;
      this.matrix[1] = m01;
      this.matrix[2] = m02;
      this.matrix[3] = m03;
      this.matrix[4] = m10;
      this.matrix[5] = m11;
      this.matrix[6] = m12;
      this.matrix[7] = m13;
      this.matrix[8] = m20;
      this.matrix[9] = m21;
      this.matrix[10] = m22;
      this.matrix[11] = m23;
      this.matrix[12] = m30;
      this.matrix[13] = m31;
      this.matrix[14] = m32;
      this.matrix[15] = m33;
      }


   /**
   * Creates a new Matrix_4x4 object from the Matrix_4x4's component values.
   * @return the new Matrix_4x4 object reference.
   */
   public Matrix_4x4 clone()
      {
      Matrix_4x4 out_matrix = new Matrix_4x4(this);
      return(out_matrix);
      }

   /**
   * Gets the designated entry of this Matrix_4x4.
   * @param row the matrix row for the entry.
   * @param column the matrix column for the entry.
   * @return the designated entry component value.
   */
   public double getEntry(int row, int column)
      {
      return(this.matrix[row * 4 + column]);
      }

   /**
   * Sets the designated entry of this Matrix_4x4.
   * @param row the matrix row for the entry.
   * @param column the matrix column for the entry.
   * @param value the value to store.
   */
   public void setEntry(int row, int column, double value)
      {
      this.matrix[row * 4 + column] = value;
      }


   /**
   *  Sets the Matrix_4x4 object to the values of another matrix.
   *  @param in the other matrix.
   */
   public void set(Matrix_4x4 in)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = in.matrix[entry_index];
         }
      }


   /**
   *  Sets the Matrix_4x4 object to the values of an input array in row-major
   *  form.
   *  @param in_array the array of values.
   */
   public void set(double [] in_array)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = in_array[entry_index];
         }
      }


   /**
   *  Sets the Matrix_4x entry values to the given values.
   *  @param m00 the (0, 0) entry value;
   *  @param m01 the (0, 1) entry value;
   *  @param m02 the (0, 2) entry value;
   *  @param m03 the (0, 3) entry value;
   *  @param m10 the (1, 0) entry value;
   *  @param m11 the (1, 1) entry value;
   *  @param m12 the (1, 2) entry value;
   *  @param m13 the (1, 3) entry value;
   *  @param m20 the (2, 0) entry value;
   *  @param m21 the (2, 1) entry value;
   *  @param m22 the (2, 2) entry value;
   *  @param m23 the (2, 3) entry value;
   *  @param m30 the (3, 0) entry value;
   *  @param m31 the (3, 1) entry value;
   *  @param m32 the (3, 2) entry value;
   *  @param m33 the (3, 3) entry value;
   */
   public void set(double m00, double m01, double m02, double m03,
                   double m10, double m11, double m12, double m13,
                   double m20, double m21, double m22, double m23,
                   double m30, double m31, double m32, double m33)
      {
      this.matrix[0] = m00;
      this.matrix[1] = m01;
      this.matrix[2] = m02;
      this.matrix[3] = m03;
      this.matrix[4] = m10;
      this.matrix[5] = m11;
      this.matrix[6] = m12;
      this.matrix[7] = m13;
      this.matrix[8] = m20;
      this.matrix[9] = m21;
      this.matrix[10] = m22;
      this.matrix[11] = m23;
      this.matrix[12] = m30;
      this.matrix[13] = m31;
      this.matrix[14] = m32;
      this.matrix[15] = m33;
      }


   /**
   *  Scales the entries in the matrix.
   *  @param scale the scale multiplier.
   */
   public void scale(double scale)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] *= scale;
         }
      }


   /**
   *  Add the input matrix to the current matrix.
   *  @param in the matrix to add.
   */
   public void add(Matrix_4x4 in)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] += in.matrix[entry_index];
         }
      }


   /**
   *  Subtract the input matrix from the current matrix.
   *  @param in the matrix to subtract.
   */
   public void subtract(Matrix_4x4 in)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] -= in.matrix[entry_index];
         }
      }


   /**
   *  Multiply this matrix by the given (right side) matrix.
   *  @param in the matrix to post-multiply.
   *  @return the product of the matrices.
   */
   public Matrix_4x4 multiply(Matrix_4x4 in)
      {
      Matrix_4x4 out_matrix = new
            Matrix_4x4(
                      matrix[0] * in.matrix[0] +
                         matrix[1] * in.matrix[4] +
                         matrix[2] * in.matrix[8] +
                         matrix[3] * in.matrix[12],
                      matrix[0] * in.matrix[1] +
                         matrix[1] * in.matrix[5] +
                         matrix[2] * in.matrix[9] +
                         matrix[3] * in.matrix[13],
                      matrix[0] * in.matrix[2] +
                         matrix[1] * in.matrix[6] +
                         matrix[2] * in.matrix[10] +
                         matrix[3] * in.matrix[14],
                      matrix[0] * in.matrix[3] +
                         matrix[1] * in.matrix[7] +
                         matrix[2] * in.matrix[11] +
                         matrix[3] * in.matrix[15],
                      matrix[4] * in.matrix[0] +
                         matrix[5] * in.matrix[4] +
                         matrix[6] * in.matrix[8] +
                         matrix[7] * in.matrix[12],
                      matrix[4] * in.matrix[1] +
                         matrix[5] * in.matrix[5] +
                         matrix[6] * in.matrix[9] +
                         matrix[7] * in.matrix[13],
                      matrix[4] * in.matrix[2] +
                         matrix[5] * in.matrix[6] +
                         matrix[6] * in.matrix[10] +
                         matrix[7] * in.matrix[14],
                      matrix[4] * in.matrix[3] +
                         matrix[5] * in.matrix[7] +
                         matrix[6] * in.matrix[11] +
                         matrix[7] * in.matrix[15],
                      matrix[8] * in.matrix[0] +
                         matrix[9] * in.matrix[4] +
                         matrix[10] * in.matrix[8] +
                         matrix[11] * in.matrix[12],
                      matrix[8] * in.matrix[1] +
                         matrix[1] * in.matrix[5] +
                         matrix[2] * in.matrix[9] +
                         matrix[3] * in.matrix[13],
                      matrix[8] * in.matrix[2] +
                         matrix[9] * in.matrix[6] +
                         matrix[10] * in.matrix[10] +
                         matrix[11] * in.matrix[14],
                      matrix[8] * in.matrix[3] +
                         matrix[9] * in.matrix[7] +
                         matrix[10] * in.matrix[11] +
                         matrix[11] * in.matrix[15],
                      matrix[12] * in.matrix[0] +
                         matrix[13] * in.matrix[4] +
                         matrix[14] * in.matrix[8] +
                         matrix[15] * in.matrix[12],
                      matrix[12] * in.matrix[1] +
                         matrix[13] * in.matrix[5] +
                         matrix[14] * in.matrix[9] +
                         matrix[15] * in.matrix[13],
                      matrix[12] * in.matrix[2] +
                         matrix[13] * in.matrix[6] +
                         matrix[14] * in.matrix[10] +
                         matrix[15] * in.matrix[14],
                      matrix[12] * in.matrix[3] +
                         matrix[13] * in.matrix[7] +
                         matrix[14] * in.matrix[11] +
                         matrix[15] * in.matrix[15]
                      );
      
      return(out_matrix);
      }


   /**
   *  For support of graphics this multiplies a Vector3D (augmented with a 1.0
   *  entry) times a Matrix_4x4 (v' X M), performing the 4-dimensional
   *  transformation to enable scaling, translation, and rotations.
   *  @param in_vector the vector (v).
   *  @param in the matrix (M).
   *  @return the resulting transformed vector.
   */
   public static Vector3D Multiply(Vector3D in_vector, Matrix_4x4 in)
      {
      Vector3D out_vector = new
           Vector3D(
                    in_vector.x * in.matrix[0] +
                    in_vector.y * in.matrix[4] +
                    in_vector.z * in.matrix[8] +
                    1.0 * in.matrix[12],

                    in_vector.x * in.matrix[1] +
                    in_vector.y * in.matrix[5] +
                    in_vector.z * in.matrix[9] +
                    1.0 * in.matrix[13],

                    in_vector.x * in.matrix[2] +
                    in_vector.y * in.matrix[6] +
                    in_vector.z * in.matrix[10] +
                    1.0 * in.matrix[14]
                   );
      return(out_vector);
      }


   /**
   *  For support of graphics this multiplies (M x v) a Matrix_4x4 times a
   *  Vector3D (augmented with a 1.0 entry), performing the 4-dimensional
   *  transformation to enable scaling, translation, and rotations.
   *  @param in the matrix (M).
   *  @param in_vector the vector (v).
   *  @return the resulting transformed vector.
   */
   public static Vector3D Multiply(Matrix_4x4 in, Vector3D in_vector)
      {
      Vector3D out_vector = new
           Vector3D(
                    in.matrix[0] * in_vector.x +
                    in.matrix[1] * in_vector.y +
                    in.matrix[2] * in_vector.z +
                    in.matrix[3] * 1.0,

                    in.matrix[4] * in_vector.x +
                    in.matrix[5] * in_vector.y +
                    in.matrix[6] * in_vector.z +
                    in.matrix[7] * 1.0,

                    in.matrix[8] * in_vector.x +
                    in.matrix[9] * in_vector.y +
                    in.matrix[10] * in_vector.z +
                    in.matrix[11] * 1.0
                   );

      return(out_vector);
      }

   /**
   * Returns a String that represents the value of this Matrix_4x4.
   * @return the string representation of the matrix.
   */
   public String toString()
      {
      String out_string = new String("");

      out_string = String.format("(%9.3g %9.3g %9.3g %9.3g)\n"
                               + "(%9.3g %9.3g %9.3g %9.3g)\n"
                               + "(%9.3g %9.3g %9.3g %9.3g)\n"
                               + "(%9.3g %9.3g %9.3g %9.3g)\n",
                                 this.matrix[0],
                                 this.matrix[1],
                                 this.matrix[2],
                                 this.matrix[3],
                                 this.matrix[4],
                                 this.matrix[5],
                                 this.matrix[6],
                                 this.matrix[7],
                                 this.matrix[8],
                                 this.matrix[9],
                                 this.matrix[10],
                                 this.matrix[11],
                                 this.matrix[12],
                                 this.matrix[13],
                                 this.matrix[14],
                                 this.matrix[15]
                                );
      return(out_string);
      }

   /**
   * Prints the representation the value of this Matrix_4x4.
   */
   public void Print()
      {
      System.out.print(this);
      }
   }

