
package util.geom;

import java.util.*;
import java.io.*;

import util.*;

/**
*  This class allows storing and manipulating NxM matrices.
*/
public class Matrix_NxM
   {

   /**
   * The rank of the matrix, once it is calculated.
   */
   Integer rank = null;

   /**
   * The number of rows in the matrix.
   */
   public int rows = 0;

   /**
   * The number of columns in the matrix.
   */
   public int cols = 0;


   /**
   * The default format for the matrix elements.
   */
   public static String default_format = "%9.3g";

   /**
   * The chosen format for the matrix elements, initially set to the
   * default.
   */
   public String format = default_format;

   /**
   * The values for the matrix, stored as a linear array.
   */
   public double [] matrix = null;

   /**
   *  Creates as much of an identity Matrix_NxM as possible.
   *  @param rows the number of rows.
   *  @param cols the number of columns.
   *  @return the NxM matrix with 1.0 in all diagonals and 0.0 elsewhere.
   */
   public static Matrix_NxM getIdentity(int rows, int cols)
      {
      Matrix_NxM ident = new Matrix_NxM(rows, cols);
      ident.makeIdentity();
      return(ident);
      }


   /**
   *  Creates a Matrix_NxM object with 0.0 as the entry values.
   *  @param rows the number of rows.
   *  @param cols the number of columns.
   */
   public Matrix_NxM(int rows, int cols)
      {
      this.rows = rows;
      this.cols = cols;
      this.matrix = new double[rows * cols];
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = 0.0;
         }
      if (this.rows < this.cols)
         {
         this.rank = this.rows;
         }
      else
         {
         this.rank = this.cols;
         }
      }


   /**
   *  Creates a Matrix_NxM object from a Matrix_NxM argument's component values.
   *  @param in the other matrix.
   */
   public Matrix_NxM(Matrix_NxM in)
      {
      this.rows = in.rows;
      this.cols = in.cols;
      this.matrix = new double[in.matrix.length];
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = in.matrix[entry_index];
         }
      }


   /**
   *  Creates a Matrix_NxM object from the array of values in row-major
   *  form.
   *  @param in_array the array of values.
   */
   public Matrix_NxM(int rows, double [] in_array)
      {
      this.rows = rows;
      this.cols = in_array.length / this.rows;
      this.matrix = new double[this.rows * this.cols];
      for (int entry_index = 0;
           entry_index < in_array.length;
           entry_index++)
         {
         this.matrix[entry_index] = in_array[entry_index];
         }
      }


   /**
   * Creates a new Matrix_NxM object from the Matrix_NxM's component values.
   * @return the new Matrix_NxM object reference.
   */
   public Matrix_NxM clone()
      {
      Matrix_NxM out_matrix = new Matrix_NxM(this);
      return(out_matrix);
      }

   /**
   *  Make the rectangular array into an identity as much as possible
   *  by placing 1.0 on the diagonals and 0.0 everywhere else.
   */
   public void makeIdentity()
      {
      int entry_index = 0;
      for (int row_index = 0;
           row_index < this.rows;
           row_index++)
         {
         for (int col_index = 0;
              col_index < this.cols;
              col_index++)
            {
            if (row_index == col_index)
               {
               this.matrix[entry_index] = 1.0;
               }
            else
               {
               this.matrix[entry_index] = 0.0;
               }
            entry_index++;
            }
         }
      if (this.rows < this.cols)
         {
         this.rank = this.rows;
         }
      else
         {
         this.rank = this.cols;
         }
      }

   /**
   *  Make the rectangular array into an all-zero by placing 0.0 on
   *  everywhere.
   */
   public void makeZero()
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = 0.0;
         }
      this.rank = 0;
      }


   /**
   * Gets the number of rows in the matrix.
   * @return the number of rows in the matrix.
   */
   public int getNumRows()
      {
      return(this.rows);
      }


   /**
   * Gets the number of columns in the matrix.
   * @return the number of columns in the matrix.
   */
   public int getNumColumns()
      {
      return(this.cols);
      }


   /**
   * Gets the rank of the matrix.
   * @return the rank of the matrix.
   */
   public int getRank()
      {
      if (this.rank == null)
         {
         Matrix_NxM lower = new Matrix_NxM(this.rows, this.rows);
         Matrix_NxM upper = new Matrix_NxM(this.cols, this.cols);
         Matrix_NxM clone = new Matrix_NxM(this);
         clone.getGeneralizedInverse(upper, lower);
         }
      return(this.rank);
      }


   /**
   * Gets the designated entry of this Matrix_NxM.
   * @param row_index the matrix row for the entry.
   * @param col_index the matrix column for the entry.
   * @return the designated entry component value.
   */
   public double getEntry(int row_index, int col_index)
      {
      return(this.matrix[row_index * this.cols + col_index]);
      }

   /**
   * Sets the designated entry of this Matrix_NxM.
   * @param row_index the matrix row for the entry.
   * @param col_index the matrix column for the entry.
   * @param value the value to store.
   */
   public void setEntry(int row_index, int col_index, double value)
      {
      this.matrix[row_index * this.cols + col_index] = value;
      this.rank = null;
      }


   /**
   * Gets the designated row of this Matrix_NxM.
   * @param row_index the matrix row for the entry.
   * @return the designated row vector value.
   */
   public VectorND getRow(int row_index)
      {
      VectorND result_vect = null;
      if (row_index < 0 || row_index >= this.rows)
         {
         System.out.println("Matrix_NxM.getRow(row_index) ERROR:\n"
                          + "   The row index ("
                          + row_index
                          + " is outside the matrix rows range [0, "
                          + (this.rows - 1)
                          + "]");
         }
      else
         {
         result_vect = new VectorND(this.cols);
         for (int col_index = 0;
              col_index < this.cols;
              col_index++)
            {
            int entry_index = row_index * this.cols + col_index;
            result_vect.values[col_index] = this.matrix[entry_index];
            }
         }
      return(result_vect);
      }

   /**
   * Sets the designated row of this Matrix_NxM.
   * @param row_index the matrix row for the entry.
   * @param vector the vector of values to store.
   * @return true if there is an error detected.
   */
   public boolean setRow(int row_index, VectorND vector)
      {
      boolean error_flag = true;
      if (row_index < 0 || row_index >= this.rows)
         {
         System.out.println("Matrix_NxM.setRow(row_index, vector) ERROR:\n"
                          + "   The row index ("
                          + row_index
                          + " is outside the matrix rows range [0, "
                          + (this.rows - 1)
                          + "]");
         }
      else if (vector == null)
         {
         System.out.println("Matrix_NxM.setRow(row_index, vector) ERROR:\n"
                          + "   The values vector is null.");
         }
      else if (vector.values.length != this.cols)
         {
         System.out.println("Matrix_NxM.setRow(row_index, vector) ERROR:\n"
                          + "   The values vector length ("
                          + vector.values.length
                          + ") does not equal the number of columns ("
                          + this.cols
                          + ")."
                           );
         }
      else
         {
         error_flag = false;
         for (int col_index = 0;
              col_index < this.cols;
              col_index++)
            {
            int entry_index = row_index * this.cols + col_index;
            this.matrix[entry_index] = vector.values[col_index];
            }
         }
      return(error_flag);
      }


   /**
   * Gets the designated column of this Matrix_NxM.
   * @param col_index the matrix column for the entry.
   * @return the designated column vector value.
   */
   public VectorND getColumn(int col_index)
      {
      VectorND result_vect = null;
      if (col_index < 0 || col_index >= this.cols)
         {
         System.out.println("Matrix_NxM.getColumn(col_index) ERROR:\n"
                          + "   The column index ("
                          + col_index
                          + " is outside the matrix columns range [0, "
                          + (this.cols - 1)
                          + "]");
         }
      else
         {
         result_vect = new VectorND(this.cols);
         for (int row_index = 0;
              row_index < this.cols;
              row_index++)
            {
            int entry_index = row_index * this.cols + col_index;
            result_vect.values[row_index] = this.matrix[entry_index];
            }
         }
      return(result_vect);
      }

   /**
   * Sets the designated column of this Matrix_NxM.
   * @param col_index the matrix column for the entry.
   * @param vector the vector of values to store.
   * @return true if there is an error detected.
   */
   public boolean setColumn(int col_index, VectorND vector)
      {
      boolean error_flag = true;
      if (col_index < 0 || col_index >= this.cols)
         {
         System.out.println("Matrix_NxM.setColumn(col_index, vector) ERROR:\n"
                          + "   The column index ("
                          + col_index
                          + " is outside the matrix columns range [0, "
                          + (this.cols - 1)
                          + "]");
         }
      else if (vector == null)
         {
         System.out.println("Matrix_NxM.setColumn(col_index, vector) ERROR:\n"
                          + "   The values vector is null.");
         }
      else if (vector.values.length != this.cols)
         {
         System.out.println("Matrix_NxM.setColumn(col_index, vector) ERROR:\n"
                          + "   The values vector length ("
                          + vector.values.length
                          + ") does not equal the number of columns ("
                          + this.cols
                          + ")."
                           );
         }
      else
         {
         error_flag = false;
         for (int row_index = 0;
              row_index < this.cols;
              row_index++)
            {
            int entry_index = row_index * this.cols + col_index;
            this.matrix[entry_index] = vector.values[row_index];
            }
         }
      return(error_flag);
      }


   /**
   *  Sets the Matrix_NxM object to the values of another matrix.  It assumes
   *  the matrices are the same dimensionality.
   *  @param in the other matrix.
   */
   public void set(Matrix_NxM in)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] = in.matrix[entry_index];
         }
      this.rank = null;
      }


   /**
   *  Sets the Matrix_NxM object to the values of an input array in row-major
   *  form.
   *  @param in_array the array of values.
   */
   public void set(double [] in_array)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length && entry_index < in_array.length;
           entry_index++)
         {
         this.matrix[entry_index] = in_array[entry_index];
         }
      this.rank = null;
      }


   /**
   *  Scales the entries in the matrix.
   *  @param scale the scale multiplier.
   */
   public void scale(double scale)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] *= scale;
         }
      if (scale == 0.0)
         {
         this.rank = null;
         }
      }


   /**
   *  Add the input matrix to the current matrix.
   *  @param in_matrix the matrix to add.
   */
   public void addTo(Matrix_NxM in_matrix)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] += in_matrix.matrix[entry_index];
         }
      this.rank = null;
      }


   /**
   *  Subtract the input matrix from the current matrix.
   *  @param in_matrix the matrix to subtract.
   */
   public void subtractFrom(Matrix_NxM in_matrix)
      {
      for (int entry_index = 0;
           entry_index < this.matrix.length;
           entry_index++)
         {
         this.matrix[entry_index] -= in_matrix.matrix[entry_index];
         }
      this.rank = null;
      }


   /**
   *  Add the input matrix and the current matrix to get a result matrix.
   *  @param in_matrix the matrix to add.
   *  @return the result matrix.
   */
   public Matrix_NxM add(Matrix_NxM in_matrix)
      {
      Matrix_NxM out_matrix = new Matrix_NxM(this);
      out_matrix.addTo(in_matrix);
      return(out_matrix);
      }


   /**
   *  Subtract the input matrix from the current matrix to get a result matrix.
   *  @param in_matrix the matrix to subtract.
   *  @return the result matrix.
   */
   public Matrix_NxM subtract(Matrix_NxM in_matrix)
      {
      Matrix_NxM out_matrix = new Matrix_NxM(this);
      out_matrix.subtractFrom(in_matrix);
      return(out_matrix);
      }


   /**
   *  Multiply this matrix by the given (right side) matrix to get a result
   *  matrix.
   *  @param in_matrix the matrix to post-multiply.
   *  @return the product of the matrices.
   */
   public Matrix_NxM Multiply(Matrix_NxM in_matrix)
      {
      Matrix_NxM out_matrix = null;

/*
      System.out.println("Matrix.Multiply(in_matrix)"
                       + " on ENTRY.\n"
                       + "this:\n"
                       + this
                       + "in_matrix:\n"
                       + in_matrix
                        );
*/
      if (this.cols != in_matrix.rows)
         {
         System.out.println("Matrix_NxM.Multiply(in_matrix) ERROR:\n"
                          + "  This matrix is "
                          + this.rows
                          + " x "
                          + this.cols
                          + " and in_matrix is "
                          + in_matrix.rows
                          + " x "
                          + in_matrix.cols
                          + " which is incompatible for multiply."
                           );
         }
      else
         {
         out_matrix = new Matrix_NxM(this.rows, in_matrix.cols);
         for (int row_index = 0;
              row_index < this.rows;
              row_index++)
            {
            for (int col_index = 0;
                 col_index < in_matrix.cols;
                 col_index++)
               {
               int entry_index = row_index * in_matrix.cols + col_index;
               double value = 0.0;
               out_matrix.matrix[entry_index] = 0.0;
               for (int col_row_index = 0;
                    col_row_index < this.cols;
                    col_row_index++)
                  {
                  int entry_index_1 = row_index * this.cols + col_row_index;
                  int entry_index_2 = col_row_index * in_matrix.cols +
                                      col_index;

                  value += this.matrix[entry_index_1] *
                           in_matrix.matrix[entry_index_2];
      
                  }

/*
               System.out.println("   -- " + entry_index + "  "
                                + value
                                 );
*/
               out_matrix.matrix[entry_index] = value;
               }
            }
         }
      out_matrix.rank = null;
      return(out_matrix);
      }


   /**
   *  Multiplies the vector times the matrix (VM).
   *  @param in_vector the vector (v).
   *  @return the resulting transformed vector.
   */
   public VectorND preMultiply(VectorND in_vector)
      {
      VectorND out_vector = new VectorND(this.cols);
      for (int col_index = 0;
           col_index < this.cols;
           col_index++)
         {
         int entry_index = col_index;
         out_vector.values[col_index] = 0.0;
         for (int row_index = 0;
              row_index < this.rows;
              row_index++)
            {
            out_vector.values[col_index] += in_vector.values[row_index] *
                                            this.matrix[entry_index];
            entry_index += this.cols;
            }
         }
      return(out_vector);
      }


   /**
   *  Multiplies the matrix times the vector (MV).
   *  @param in_vector the vector (v).
   *  @return the resulting transformed vector.
   */
   public VectorND postMultiply(VectorND in_vector)
      {
      VectorND out_vector = new VectorND(this.rows);
      int entry_index = 0;
      for (int row_index = 0;
           row_index < this.rows;
           row_index++)
         {
         out_vector.values[row_index] = 0.0;
         for (int col_index = 0;
              col_index < this.cols;
              col_index++)
            {
            out_vector.values[row_index] += this.matrix[entry_index] *
                                            in_vector.values[col_index];
            entry_index++;
            }
         }
      return(out_vector);
      }

   /**
   *  Perform the Gaussian elimination on a matrix and its augmented matrices,
   *  giving the generalized form for the two resulting row and column
   *  operation matrices. This gives a form of decomposition that
   *  may be used on singular and non-square matrices to identify
   *  generalized inverses and their null spaces.  The row and column
   *  operation matrices are transformed into the generalize decomposition.
   *  The base matrix is converted to as close to an identity as is possible.
   *  <pre>
   *          <u>A</u> | <u>I</u>    =====&gt;    <u>I</u> | <u>R</u>
   *          I |                S |
   *  where
   *     A is the base matrix.
   *     R is the right result matrix, or the row operations matrix.
   *     S is the bottom result matrix, or the column operations matrix.
   *  </pre>
   *  @param right a square identify which is # rows x # rows.
   *  @param bottom a square identify which is # columns x # columns.
   *  @return true if an error was detected.
   */
   public boolean getGeneralizedInverse(Matrix_NxM right, Matrix_NxM bottom)
      {
      boolean error_flag = false;

/*
      System.out.println("Matrix_NxM.getGeneralizedInverse(right, bottom)"
                       + " on ENTRY.");
*/

      if (right.rows != right.cols)
         {
         System.out.println("Matrix_NxM.GenInverse(right, bottom) ERROR:\n"
                          + "  The right matrix is not square.\n");
         error_flag = true;
         }
      else if (bottom.rows != bottom.cols)
         {
         System.out.println("Matrix_NxM.GenInverse(right, bottom) ERROR:\n"
                          + "  The bottom matrix is not square.\n");
         error_flag = true;
         }
      else if (this.rows != right.rows)
         {
         System.out.println("Matrix_NxM.GenInverse(right, bottom) ERROR:\n"
                          + "  The right matrix is not # rows x # rows.\n");
         error_flag = true;
         }
      else if (this.cols != bottom.cols)
         {
         System.out.println("Matrix_NxM.GenInverse(right, bottom) ERROR:\n"
                          + "  The bottom matrix is not # cols x # cols.\n");
         error_flag = true;
         }
      else
         {
         right.makeIdentity();
         bottom.makeIdentity();

         /*
         *  This is set large to get into the loop.  Its value changes
         *  every pass through the loop.
         */
         double largest_value = Double.MAX_VALUE;

         /*
         *  The decomposition is done when the largest value is 0.0 or
         *  it runs out of rows and columns.
         */
         this.rank = 0;
         for (int curr_row_index = 0;
              curr_row_index < this.rows && largest_value != 0.0;
              curr_row_index++)
            {
            int curr_col_index = curr_row_index;
            /*
            * Get the largest magnitude value in the rest of the matrix.
            */
            int diag_entry_index = curr_row_index * this.cols +
                                   curr_col_index;
/*
            System.out.println("Largest location search start = ("
                             + curr_row_index
                             + ", "
                             + curr_col_index
                             + ")"
                              );
*/
            int [] largest_loc = this.getLargestLoc(curr_row_index,
                                                    curr_col_index);
            int largest_entry_index = largest_loc[0] * this.cols +
                                      largest_loc[1];
            largest_value = this.matrix[largest_entry_index];
/*
            System.out.println("Largest entry location = ("
                             + largest_loc[0]
                             + ", "
                             + largest_loc[1]
                             + "): "
                             + largest_value
                              );
*/
            if (largest_value != 0.0)
               {
               /*
               *  There is still work to be done on the decomposition.
               */
               if (largest_loc[0] > curr_row_index)
                  {
/*
                  System.out.println("   Swapping rows "
                                   + largest_loc[0]
                                   + " and "
                                   + curr_row_index
                                    );
*/
                  this.swapRows(curr_row_index, largest_loc[0]);
                  right.swapRows(curr_row_index, largest_loc[0]);
                  }
               /*
               *  Since this is on the diagonal the row index is also the
               *  current column index of interest.
               */
               if (largest_loc[1] > curr_col_index)
                  {
/*
                  System.out.println("   Swapping columns "
                                   + largest_loc[1]
                                   + " and "
                                   + curr_col_index
                                    );
*/
                  this.swapColumns(curr_col_index, largest_loc[1]);
                  bottom.swapColumns(curr_col_index, largest_loc[1]);
                  }
/*
               System.out.println("After any swaps --\n"
                                + "matrix:\n"
                                + this
                                + "right:\n"
                                + right
                                + "bottom:\n"
                                + bottom
                                + "largest_value = "
                                + largest_value
                                 );
*/
               double diag_value = largest_value;
               if (diag_value != 1.0)
                  {
                  /*
                  * There are variations possible here, but this is the
                  * simplest decomposition.
                  *
                  * For other uses both the row and column could be
                  * scaled by the square root of the inverse of the diagonal
                  * term value.
                  */
                  double scale = 1.0 / diag_value;

                  this.scaleRow(scale, curr_row_index, curr_col_index);
                  right.scaleRow(scale, curr_row_index, null);
/*
                  System.out.println("Row "
                                   + curr_row_index
                                   + " is scaled --\n"
                                   + "matrix:\n"
                                   + this
                                   + "right:\n"
                                   + right
                                    );
*/
                  }
               /*
               * The current row can now be used to empty the rest of the
               * column via linear row combinations.
               */
               for (int row_index = curr_row_index + 1;
                    row_index < this.rows;
                    row_index++)
                  {
                  int entry_index = row_index * this.cols + curr_col_index;
                  double entry_value = this.matrix[entry_index];
                  if (entry_value != 0.0)
                     {
                     Integer start_col = curr_col_index + 1;
                     this.combineRows(curr_row_index,
                                      entry_value,
                                      row_index,
                                      start_col);
                     right.combineRows(curr_row_index,
                                       entry_value,
                                       row_index,
                                       null);
                     }
                  this.matrix[entry_index] = 0.0;
                  }
/*
               System.out.println("After combining rows to empty column "
                                + curr_col_index
                                + " --\n"
                                + "matrix:\n"
                                + this
                                + "right:\n"
                                + right
                                 );
*/
               /*
               * The current column can be used to empty the rest of the
               * row via linear column combinations.
               */
               for (int col_index = curr_col_index + 1;
                    col_index < this.cols;
                    col_index++)
                  {
                  int entry_index = curr_row_index * this.cols + col_index;
                  double entry_value = this.matrix[entry_index];
                  if (entry_value != 0.0)
                     {
                     Integer start_row = curr_row_index + 1;
                     this.combineColumns(curr_col_index,
                                         entry_value,
                                         col_index,
                                         start_row);
                     bottom.combineColumns(curr_col_index,
                                           entry_value,
                                           col_index,
                                           null);
                     }
                  this.matrix[entry_index] = 0.0;
                  }
/*
               System.out.println("After combining columns to empty row "
                                + curr_row_index
                                + " --\n"
                                + "matrix:\n"
                                + this
                                + "bottom:\n"
                                + bottom
                                 );
*/
               this.rank++;
               }
            }
         }
/*
      System.out.println("Matrix_NxM.getGeneralizedInverse(right, bottom)"
                       + " on EXIT.");
*/

      return(error_flag);
      }

   /**
   * Perform LU decomposition on the matrix in-place, replacing the matrix
   * with the conventional representation of the LU decomposition.
   * @param permutations returns the array of permuted row indices for the
   *     rows swaps performed
   * @return the parity indicator for the row swaps,
   *     &gt; 0 for even,
   *     &lt; 0 for odd.
   */
   public Integer LUDecomp(Integer [] permutations)
      {
      Integer parity_indicator = null;

/*
      System.out.println("Matrix.LUDecomp(permutations)"
                       + " on ENTRY.\n"
                       + "this:\n"
                       + this
                        );
*/

      /*
      *  This is needed for later operations with the matrices.
      */
      double [] row_scale_factor = null;

      boolean error_flag = false;

      if (this.rows != this.cols)
         {
         System.out.println("Matrix_NxM.LUDecomp(permutations) ERROR:\n"
                          + "  The matrix is not square.\n");
         error_flag = true;
         }
      else if (permutations == null ||
               permutations.length != this.rows)
         {
         System.out.println("Matrix_NxM.LUDecomp(permutations) ERROR:\n"
                          + "  The permutations is null or not is not the"
                          + " same size as the matrix row count.\n");
         error_flag = true;
         }
      else
         {
         row_scale_factor = new double[this.rows];
         int entry_index = 0;
         for (int row_index = 0;
              row_index < this.rows;
              row_index++)
            {
            /*
            * Initialize the array of row indices while we are looping.
            */
            permutations[row_index] = row_index;

            /*
            * Get the largest value in the row, and create the divisor for that
            * row.
            */
            double largest_mag_in_row = 0.0;
            for (int col_index = 0;
                 col_index < this.cols;
                 col_index++)
               {
               double abs_value = Math.abs(this.matrix[entry_index]);
               if (largest_mag_in_row < abs_value)
                  {
                  largest_mag_in_row = abs_value;
                  }
               entry_index++;
               }
            if (largest_mag_in_row == 0.0)
               {
               System.out.println("Matrix_NxM.LUDecomp(permutations) ERROR:\n"
                                + "  The matrix is singular."
                                 );
               error_flag = true;
               }
            else
               {
               /*
               * Get the scale factor for the row for later use.
               */
               row_scale_factor[row_index] = 1.0 / largest_mag_in_row;
/*
               System.out.println("   largest_mag_in_row "
                                + row_index
                                + " is "
                                + largest_mag_in_row
                                + " giving a scale of "
                                + row_scale_factor[row_index]
                                 );
*/
               }
            }
/*
         System.out.println();
*/
         }
         
      if (!error_flag)
         {
/*
         System.out.println("Matrix_NxM.LUDecomp(permutations) STATUS:\n"
                          + "   Calculating the decomposition."
                           );
*/
         /*
         *  This is used to keep track of the row swapping parity.
         */
         parity_indicator = 1;

         /*
         * The main part of the algorithm is implemented here, calculating
         * the upper values for a column and all rows, then calculating the
         * lower values for the column and all rows, then repeating for the
         * next column until done.
         */
         int entry_index = 0;
         for (int diag_col_row_index = 0;
              diag_col_row_index < this.cols;
              diag_col_row_index++)
            {
            /*
            * This is the diagonal, so the column index is the row index.
            */
            int diag_entry_index = diag_col_row_index * this.cols +
                                   diag_col_row_index;

            /*
            * Calculate the "upper" entries for this column.
            */
            for (int row_index = 0;
                 row_index < diag_col_row_index;
                 row_index++)
               {
               /*
               * Start at the left in the lower matrix and the top on the
               * upper matrix and go toward the diagonal.
               */
               entry_index = row_index * this.cols + diag_col_row_index;
               double sum = this.matrix[entry_index];
               for (int col_row_index = 0;
                    col_row_index < row_index;
                    col_row_index++)
                  {
                  int entry_index1 = row_index * this.cols + col_row_index;
                  int entry_index2 = col_row_index * this.cols +
                                     diag_col_row_index;
                  sum -= this.matrix[entry_index1] *
                         this.matrix[entry_index2];
                  }
               this.matrix[entry_index] = sum;
               }
/*
            System.out.println("After calculating upper for column "
                             + diag_col_row_index
                             + " -- \n"
                             + "matrix:\n"
                             + this
                              );
*/

            /*
            * The above calculation is repeated for each later row to find
            * the row with the largest FOM (or column entry!).  This allows
            * swapping rows to use the largest values first to minimize
            * the error in the final results.
            */

            /*
            * This also calculates the unscaled values for the rest of
            * the "lower" entries for this column.
            */
            double largest_entry = 0.0;
            int largest_row_index = diag_col_row_index;
            /*
            * Start at the diagonal and go through the rows.
            */
            for (int row_index = diag_col_row_index;
                 row_index < this.rows;
                 row_index++)
               {
               /*
               * Start at the left in the lower matrix and the top on the
               * upper matrix and go to the diagonal.
               */
               entry_index = row_index * this.cols + diag_col_row_index;
               double sum = this.matrix[entry_index];
               for (int col_row_index = 0;
                    col_row_index < diag_col_row_index;
                    col_row_index++)
                  {
                  int entry_index1 = row_index * this.cols + col_row_index;
                  int entry_index2 = col_row_index * this.cols +
                                     diag_col_row_index;
                  sum -= this.matrix[entry_index1] *
                         this.matrix[entry_index2];
                  }
               /*
               *  The diagonal goes into the upper, the rest go into the lower.
               */
               this.matrix[entry_index] = sum;
/*
               System.out.println("   value stored for location ("
                                + row_index
                                + ", "
                                + diag_col_row_index
                                + ") = "
                                + sum
                                + "\n"
                                + "matrix:\n"
                                + this
                                 );
*/

               /*
               * Is this figure of merit better?  If so save it and its
               * location.
               */
               double fom = row_scale_factor[row_index] * Math.abs(sum);
               if (largest_entry < fom)
                  {
                  largest_entry = fom;
                  largest_row_index = row_index;
                  }
/*
               System.out.println(" FOM for row "
                                + row_index
                                + " is "
                                + fom
                                 );
*/

               /*
               * Some algorithms say just use the largest row entry.
               * Is this better?  It's location will override the fom above.
               */
               if (largest_entry < this.matrix[entry_index])
                  {
                  largest_entry = this.matrix[entry_index];
                  largest_row_index = row_index;
                  }
/*
               System.out.println(" Row entry for row "
                                + row_index
                                + " is "
                                + this.matrix[entry_index]
                                 );
*/
               }
/*
            System.out.println(" Largest row entry is for row "
                             + largest_row_index
                             + " and is "
                             + largest_entry
                              );
*/

            if (largest_row_index != diag_col_row_index)
               {
               /*
               *  It is not on the current diagonal, so swap it.
               */

/*
               System.out.println("   Swapping rows "
                                + diag_col_row_index
                                + " and "
                                + largest_row_index
                                 );
*/
               parity_indicator *= -1;

               /*
               * Swap the row-associated scale factor magnitudes.
               */
               double value = row_scale_factor[diag_col_row_index];
               row_scale_factor[diag_col_row_index] =
                                       row_scale_factor[largest_row_index];
               row_scale_factor[largest_row_index] = value;

               /*
               * The largest is not on this row, so swap rows.
               */
               this.swapRows(diag_col_row_index, largest_row_index);
               }
            permutations[diag_col_row_index] = largest_row_index;

            /*
            *  Check to see if the current diagonal is zero, and if it is give
            *  it a small number instead.
            */
            if (this.matrix[diag_entry_index] == 0.0)
               {
               /*
               *  From Press. et. al., "Numerical Recipies in C++". Second
               *  Edition, 2005 reprint, page 50, put in a very small value
               *  so it doesn't appear to be singular.
               */
               System.out.println("Matrix_NxM.LUDecomp(permutations) ERROR:\n"
                                + "  The matrix is too close to singular.\n"
                                + "  A tiny value was substituted for a zero"
                                + " divisor at ("
                                + diag_col_row_index
                                + ", "
                                + diag_col_row_index
                                + ")."
                                 );
               this.matrix[diag_entry_index] = 1.0e-20;
               }
/*
            System.out.println("   Before scaling all the entries for col "
                             + diag_col_row_index
                             + " -- \n"
                             + "matrix:\n"
                             + this
                              );
*/
            if (diag_col_row_index != this.cols - 1)
               {
               double scale = 1.0 / this.matrix[diag_entry_index];
               for (int row_index = diag_col_row_index + 1;
                    row_index < this.rows;
                    row_index++)
                  {
                  int entry_index1 = row_index * this.cols + diag_col_row_index;
                  this.matrix[entry_index1] *= scale;
                  }
               }
/*
            System.out.println("   After scaling all the entries for col "
                             + diag_col_row_index
                             + " -- \n"
                             + "matrix:\n"
                             + this
                              );
*/
            }
         }

/*
      System.out.println("Matrix.LUDecomp(permutations)"
                       + " on EXIT.\n"
                       + "this:\n"
                       + this
                        );
*/

      return(parity_indicator);
      }

   /**
   * Perform LU decomposition on the matrix using seperate lower and upper
   * diagonal matrices.  The results are returned in the pre-allocated matrices
   * lower and upper.
   * @param lower the matrix to receive the lower triangular result.
   * @param upper the matrix to receive the upper triangular result.
   * @param permutations returns the array of permuted row indices for the
   *     row swaps performed
   * @return the parity indicator for the row swaps,
   *     &gt; 0 for even,
   *     &lt; 0 for odd.
   */
   public Integer LUDecomp(Matrix_NxM lower,
                           Matrix_NxM upper,
                           Integer [] permutations)
      {
      Integer parity_indicator = null;

/*
      System.out.println("Matrix.LUDecomp(lower, upper, permutations)"
                       + " on ENTRY.\n"
                       + "this:\n"
                       + this
                       + "lower:\n"
                       + lower
                       + "upper:\n"
                       + upper
                        );
*/

      /*
      *  This is needed for later operations with the matrices.
      */
      double [] row_scale_factor = null;

      boolean error_flag = false;
      if (this.rows != this.cols)
         {
         System.out.println("Matrix_NxM.LUDecomp(lower, upper,"
                          + " permutations) ERROR:\n"
                          + "  The matrix is not square.\n");
         error_flag = true;
         }
      else if (this.rows != lower.rows ||
               this.cols != lower.cols)
         {
         System.out.println("Matrix_NxM.LUDecomp(lower, upper,"
                          + " permutations) ERROR:\n"
                          + "  The lower matrix is not the same size.\n");
         error_flag = true;
         }
      else if (this.rows != upper.rows ||
               this.cols != upper.cols)
         {
         System.out.println("Matrix_NxM.LUDecomp(lower, upper,"
                          + " permutations) ERROR:\n"
                          + "  The upper matrix is not the same size.\n");
         error_flag = true;
         }
      else if (permutations == null ||
               permutations.length != this.rows)
         {
         System.out.println("Matrix_NxM.LUDecomp(lower, upper,"
                          + " permutations) ERROR:\n"
                          + "  The permutations is null or not is not the"
                          + " same size as the matrix row count.\n");
         error_flag = true;
         }
      else
         {
         row_scale_factor = new double[this.rows];
         int entry_index = 0;
         for (int row_index = 0;
              row_index < this.rows;
              row_index++)
            {
            /*
            * Initialize the array of row indices while we are looping.
            */
            permutations[row_index] = row_index;

            /*
            * Get the largest value in the row, and create the divisor for that
            * row.
            */
            double largest_mag_in_row = 0.0;
            for (int col_index = 0;
                 col_index < this.cols;
                 col_index++)
               {
               double abs_value = Math.abs(this.matrix[entry_index]);
               if (largest_mag_in_row < abs_value)
                  {
                  largest_mag_in_row = abs_value;
                  }
               entry_index++;
               }
            if (largest_mag_in_row == 0.0)
               {
               System.out.println("Matrix_NxM.LUDecomp(lower, upper,"
                                + " permutations) ERROR:\n"
                                + "  The matrix is singular."
                                 );
               error_flag = true;
               }
            else
               {
               /*
               * Get the scale factor for the row for later use.
               */
               row_scale_factor[row_index] = 1.0 / largest_mag_in_row;

/*
               System.out.println("   largest_mag_in_row "
                                + row_index
                                + " is "
                                + largest_mag_in_row
                                + " giving a scale of "
                                + row_scale_factor[row_index]
                                 );
*/
               }
            }
/*
         System.out.println();
*/
         }
         
      if (!error_flag)
         {
/*
         System.out.println("Matrix_NxM.LUDecomp(lower, upper,"
                          + " permutations) STATUS:\n"
                          + "   Calculating the decomposition."
                           );
*/
         /*
         *  This is used to keep track of the row swapping parity.
         */
         parity_indicator = 1;

         /*
         * If they are not the same matrix, set the lower's diagonal to all
         * ones.  This is not done in the space-saving approach of using
         * the same matrix, as the lower diagonals are just assumed to be all
         * 1.0 and the upper diagonals are stored as the result diagonals.
         */
         lower.makeIdentity();
         upper.makeZero();

         /*
         * The main part of the algorithm is implemented here, calculating
         * the upper values for a column and all rows, then calculating the
         * lower values for the column and all rows, then repeating for the
         * next column until done.
         */
         int entry_index = 0;
         for (int diag_col_row_index = 0;
              diag_col_row_index < this.cols;
              diag_col_row_index++)
            {
            /*
            * This is the diagonal, so the column index is the row index.
            */
            int diag_entry_index = diag_col_row_index * this.cols +
                                   diag_col_row_index;

            /*
            * Calculate the "upper" entries for this column.
            */
            for (int row_index = 0;
                 row_index <= diag_col_row_index;
                 row_index++)
               {
               /*
               * Start at the left in the lower matrix and the top on the
               * upper matrix and go toward the diagonal.
               */
               entry_index = row_index * this.cols + diag_col_row_index;
               double sum = this.matrix[entry_index];
               for (int col_row_index = 0;
                    col_row_index < row_index;
                    col_row_index++)
                  {
                  int entry_index1 = row_index * this.cols + col_row_index;
                  int entry_index2 = col_row_index * this.cols +
                                     diag_col_row_index;
                  sum -= lower.matrix[entry_index1] *
                         upper.matrix[entry_index2];
                  }
               upper.matrix[entry_index] = sum;
               }

/*
            System.out.println("After calculating upper for column "
                             + diag_col_row_index
                             + " -- \n"
                             + "upper:\n"
                             + upper
                              );
*/

            /*
            * The above calculation is repeated for each later row to find
            * the row with the largest FOM (or column entry!).  This allows
            * swapping rows to use the largest values first to minimize
            * the error in the final results.
            */

            /*
            * This also calculates the unscaled values for the rest of
            * the "lower" entries for this column.
            */
            double largest_entry = 0.0;
            int largest_row_index = diag_col_row_index;
            /*
            * Start at the diagonal and go through the rows.
            */
            for (int row_index = diag_col_row_index;
                 row_index < this.rows;
                 row_index++)
               {
               /*
               * Start at the left in the lower matrix and the top on the
               * upper matrix and go to the diagonal.
               */
               entry_index = row_index * this.cols + diag_col_row_index;
               double sum = this.matrix[entry_index];
               for (int col_row_index = 0;
                    col_row_index < diag_col_row_index;
                    col_row_index++)
                  {
                  int entry_index1 = row_index * this.cols + col_row_index;
                  int entry_index2 = col_row_index * this.cols +
                                     diag_col_row_index;
                  sum -= lower.matrix[entry_index1] *
                         upper.matrix[entry_index2];
                  }
               /*
               *  The diagonal goes into the upper, the rest go into the lower.
               */
               if (row_index == diag_col_row_index)
                  {
                  upper.matrix[entry_index] = sum;
                  }
               else
                  {
                  lower.matrix[entry_index] = sum;
                  }

/*
               System.out.println("   value stored for location ("
                                + row_index
                                + ", "
                                + diag_col_row_index
                                + ") = "
                                + sum
                                + "\n"
                                + "matrix:\n"
                                + this
                                + "upper:\n"
                                + upper
                                + "lower:\n"
                                + lower
                                 );
*/

               /*
               * Is this figure of merit better?  If so save it and its
               * location.
               */
               double fom = row_scale_factor[row_index] * Math.abs(sum);
               if (largest_entry < fom)
                  {
                  largest_entry = fom;
                  largest_row_index = row_index;
                  }

/*
               System.out.println(" FOM for row "
                                + row_index
                                + " is "
                                + fom
                                 );
*/

               /*
               * Some algorithms say just use the largest row entry.
               * Is this better?  It's location will override the fom above.
               */
/*
               if (largest_entry < this.matrix[entry_index])
                  {
                  largest_entry = this.matrix[entry_index];
                  largest_row_index = row_index;
                  }
               System.out.println(" Row entry for row "
                                + row_index
                                + " is "
                                + this.matrix[entry_index]
                                 );
*/
               }
/*
            System.out.println(" Largest row entry is for row "
                             + largest_row_index
                             + " and is "
                             + largest_entry
                              );
*/

            if (largest_row_index != diag_col_row_index)
               {
               /*
               *  It is not on the current diagonal, so swap it.
               */

/*
               System.out.println("   Swapping rows "
                                + diag_col_row_index
                                + " and "
                                + largest_row_index
                                 );
*/

               parity_indicator *= -1;

               /*
               * Swap the row-associated scale factor magnitudes.
               */
               double value = row_scale_factor[diag_col_row_index];
               row_scale_factor[diag_col_row_index] =
                                       row_scale_factor[largest_row_index];
               row_scale_factor[largest_row_index] = value;

               /*
               * The largest is not on this row, so swap rows.
               */
               this.swapRows(diag_col_row_index, largest_row_index);

               if (upper != this)
                  {
                  /*
                  *  Only the diagonal element is valid for the upper
                  *  matrix.
                  */
                  int entry_index1 = largest_row_index * this.cols +
                                     diag_col_row_index;
                  double entry_value = upper.matrix[diag_entry_index];
                  upper.matrix[diag_entry_index] = lower.matrix[entry_index1];
                  lower.matrix[entry_index1] = entry_value;;
                  }
               if (lower != this)
                  {
                  /*
                  *  Only the first few columns are valid for the lower
                  *  matrix at this time.
                  */
                  for (int col_index = 0;
                       col_index < diag_col_row_index;
                       col_index++)
                     {
                     int entry_index1 = diag_col_row_index * this.cols +
                                        col_index;
                     int entry_index2 = largest_row_index * this.cols +
                                        col_index;
                     double entry_value = lower.matrix[entry_index1];
                     lower.matrix[entry_index1] = lower.matrix[entry_index2];
                     lower.matrix[entry_index2] = entry_value;
                     }
                  }
               }
            permutations[diag_col_row_index] = largest_row_index;

            /*
            *  Check to see if the current diagonal is zero, and if it is give
            *  it a small number instead.
            */
            if (upper.matrix[diag_entry_index] == 0.0)
               {
               /*
               *  From Press. et. al., "Numerical Recipies in C++". Second
               *  Edition, 2005 reprint, page 50, put in a very small value
               *  so it doesn't appear to be singular.
               */
               System.out.println("Matrix_NxM.LUDecomp(lower, upper,"
                                + " permutations) ERROR:\n"
                                + "  The matrix is too close to singular.\n"
                                + "  A tiny value was substituted for a zero"
                                + " divisor at ("
                                + diag_col_row_index
                                + ", "
                                + diag_col_row_index
                                + ")."
                                 );
               upper.matrix[diag_entry_index] = 1.0e-20;
               }
/*
            System.out.println("   Before scaling all the entries for col "
                             + diag_col_row_index
                             + " -- \n"
                             + "upper:\n"
                             + upper
                              );
*/
            if (diag_col_row_index != this.cols - 1)
               {
               double scale = 1.0 / upper.matrix[diag_entry_index];
               for (int row_index = diag_col_row_index + 1;
                    row_index < this.rows;
                    row_index++)
                  {
                  int entry_index1 = row_index * this.cols + diag_col_row_index;
                  lower.matrix[entry_index1] *= scale;
                  }
               }
/*
            System.out.println("   After scaling all the entries for col "
                             + diag_col_row_index
                             + " -- \n"
                             + "matrix:\n"
                             + this
                             + "upper:\n"
                             + upper
                             + "lower:\n"
                             + lower
                              );
*/
            }
         }

      return(parity_indicator);
      }

   /**
   * Use the LU decomposition of the matrix to solve the simultaneous
   * equations.  The derivation is as follows:
   * <pre>
   *
   *    Ax = y
   *
   * Replace A with the LU decomposition
   *
   *    LUx = y
   *
   *    Substitute an easier system of equations for Ux
   *
   *    Ux = z
   *
   *    Calculate vector for the easier system of equations from
   *
   *    Lz = y
   *
   * where
   *    A is this matrix,
   *    L is the lower matrix,
   *    U is the upper matrix,
   *    y is the vector of knowns,
   *    x is the vector of solutions to be found, and
   *    z is the intermediate solution vector.
   * </pre>
   * The matrix and its lower and upper decomposition matrices
   * are created as in LUDecomp() as in-place matrices (a common convention
   * to save space) above and are both in the same Matrix_NxM object.
   * This algorithm assumes the lower's (L's) diagonal elements are all 1.0.
   * @param permutations the permutation of the row indices.
   * @param vector the y vector from above, and the result is returned
   *     in it.
   * @return true if there was an error.
   */
   public boolean LUSolve(Integer [] permutations,
                          VectorND vector)
      {
      boolean error_flag = false;

/*
      System.out.println("Matrix.LUSolve(permutations, vector)"
                       + " on ENTRY:\n"
                       + "   vector: " + vector
                        );
*/

      if (this.rows != this.cols)
         {
         System.out.println("Matrix_NxM.LUSolve(permutations, vector)"
                          + " ERROR:\n"
                          + "  The matrix is not square.\n");
         error_flag = true;
         }
      else if (this.cols != vector.values.length)
         {
         System.out.println("Matrix_NxM.LUSolve(permutations, vector)"
                          + " ERROR:\n"
                          + "  The vector and matrix are not the same size.\n");
         error_flag = true;
         }
      else
         {
         /*
         * First do the forward substitution to get the intermediate vector
         * solution from the lower matrix and the original vector.
         */
         int diag_index = 0;
         for (int row_index = 0;
              row_index < this.rows;
              row_index++)
            {
            int permuted_index = permutations[row_index];
            double sum = vector.values[permuted_index];
            vector.values[permuted_index] = vector.values[row_index];
            if (diag_index > 0)
               {
               for (int row_col_index = diag_index - 1;
                    row_col_index < row_index;
                    row_col_index++)
                  {
                  int entry_index = row_index * this.cols +
                                    row_col_index;
                  Double entry_value = this.matrix[entry_index];

/*
                  System.out.println("   ("
                                   + row_index
                                   + ", "
                                   + row_col_index
                                   + ") = "
                                   + entry_value
                                    );
*/
                  sum -= entry_value * vector.values[row_col_index];
                  }
               }
            else if (sum != 0.0)
               {
               diag_index = row_index + 1;
               }
            vector.values[row_index] = sum;
            }

/*
         System.out.println("Intermediate vector:\n"
                          + vector
                           );
*/
         /*
         * Now do the backward substitution to get the result vector
         * solution from the upper matrix and the intermediate vector.
         */
         for (int row_index = this.rows - 1;
              row_index >= 0;
              row_index--)
            {
            double sum = vector.values[row_index];
            for (int row_col_index = row_index + 1;
                 row_col_index < this.cols;
                 row_col_index++)
               {
               int entry_index = row_index * this.cols + row_col_index;
               Double entry_value = this.matrix[entry_index];

/*
               System.out.println("   ("
                                + row_index
                                + ", "
                                + row_col_index
                                + ") = "
                                + entry_value
                                 );
*/
               sum -= entry_value * vector.values[row_col_index];
               }

            diag_index = row_index * this.cols + row_index;
            Double diag_value = this.matrix[diag_index];

/*
            System.out.println("   d ("
                             + row_index
                             + ", "
                             + row_index
                             + ") = "
                             + diag_value
                              );
*/

            vector.values[row_index] = sum / diag_value;

/*
            System.out.println("back propagation row "
                             + row_index
                             + "\n"
                             + "   vector = "
                             + vector
                              );
*/
            }
         }
      return(error_flag);
      }

   /**
   * Use the LU decomposition of the matrix to solve the simultaneous
   * equations.  The derivation is as follows:
   * <pre>
   *
   *    Ax = y
   *
   * Replace A with the LU decomposition
   *
   *    LUx = y
   *
   *    Substitute an easier system of equations for Ux
   *
   *    Ux = z
   *
   *    Calculate vector for the easier system of equations from
   *
   *    Lz = y
   *
   * where
   *    A is this matrix,
   *    L is the lower matrix,
   *    U is the upper matrix,
   *    y is the vector of knowns,
   *    x is the vector of solutions to be found, and
   *    z is the intermediate solution vector.
   * </pre>
   * The matrix and its lower and upper decomposition matrices
   * are created as in LUDecomp() above and may be the same Matrix_NxM object,
   * or may be separate Matrices.  This algorithm assumes the lower's
   * diagonal elements are all 1.0.
   * @param upper the upper triangular matrix.
   * @param lower the lower triangular matrix.
   * @param permutations the permutation of the row indices.
   * @param base_vector the y vector from above.
   * @return true if there was an error.
   */
   public VectorND LUSolve(Matrix_NxM upper,
                           Matrix_NxM lower,
                           Integer [] permutations,
                           VectorND base_vector)
      {
      VectorND vector = null;
      boolean error_flag = false;

/*
      System.out.println("Matrix."
                       + "LUSolve(upper, lower, permutations, base_vector)"
                       + " on ENTRY:\n"
                       + "upper:\n"
                       + upper
                       + "lower:\n"
                       + lower
                       + "   base_vector: " + base_vector
                        );
*/
      if (this.rows != this.cols)
         {
         System.out.println("Matrix_NxM.LUSolve(lower, upper,"
                          + " permutations, base_vector) ERROR:\n"
                          + "  The matrix is not square.\n");
         error_flag = true;
         }
      else if (this.rows != lower.rows ||
               this.cols != lower.cols)
         {
         System.out.println("Matrix_NxM.LUSolve(lower, upper,"
                          + " permutations, base_vector) ERROR:\n"
                          + "  The lower matrix is not the same size.\n");
         error_flag = true;
         }
      else if (this.rows != upper.rows ||
               this.cols != upper.cols)
         {
         System.out.println("Matrix_NxM.LUSolve(lower, upper,"
                          + " permutations, base_vector) ERROR:\n"
                          + "  The upper matrix is not the same size.\n");
         error_flag = true;
         }
      else if (this.cols != base_vector.values.length)
         {
         System.out.println("Matrix_NxM.LUSolve(lower, upper,"
                          + " permutations, base_vector) ERROR:\n"
                          + "  The vector and matrix are not the same size.\n");
         error_flag = true;
         }
      else
         {
         vector = new VectorND(base_vector);
         /*
         * First do the forward substitution to get the intermediate vector
         * solution from the lower matrix and the original vector.
         */
         int diag_index = 0;
         for (int row_index = 0;
              row_index < lower.rows;
              row_index++)
            {
            int permuted_index = permutations[row_index];
            double sum = vector.values[permuted_index];
            vector.values[permuted_index] = vector.values[row_index];
            if (diag_index > 0)
               {
               for (int row_col_index = diag_index -1;
                    row_col_index < row_index;
                    row_col_index++)
                  {
                  int entry_index = row_index * upper.cols + row_col_index;
                  sum -= lower.matrix[entry_index] *
                         vector.values[row_col_index];
                  }
               }
            else if (sum != 0.0)
               {
               diag_index = row_index + 1;
               }
            vector.values[row_index] = sum;
            }
/*
         System.out.println("Intermediate vector:\n"
                          + vector
                           );
*/
         /*
         * Now do the backward substitution to get the result vector
         * solution from the upper matrix and the intermediate vector.
         */
         for (int row_index = upper.rows - 1;
              row_index >= 0;
              row_index--)
            {
            double sum = vector.values[row_index];
            for (int row_col_index = row_index + 1;
                 row_col_index < upper.cols;
                 row_col_index++)
               {
               int entry_index = row_index * upper.cols + row_col_index;
               sum -= upper.matrix[entry_index] *
                      vector.values[row_col_index];
               }
            diag_index = row_index * upper.cols + row_index;
            Double diag_value = upper.matrix[diag_index];
            if (diag_value != null &&
                diag_value != 0.0)
               {
               vector.values[row_index] = sum / diag_value;
               }
            else if (sum > 0)
               {
               vector.values[row_index] = Double.POSITIVE_INFINITY;
               }
            else if (sum < 0)
               {
               vector.values[row_index] = Double.NEGATIVE_INFINITY;
               }
            else
               {
               vector.values[row_index] = Double.NaN;
               }

/*
            System.out.println("back propagation row "
                             + row_index
                             + "\n"
                             + "   diag_value = "
                             + diag_value
                             + "\n"
                             + "   vector = "
                             + vector
                              );
*/
            }
         }
      return(vector);
      }


   /**
   * Get the location of the largest value in the matrix beyond the start
   * row and column indices.
   * @param start_row_index the first row to check.
   * @param start_col_index the first column to check.
   * @return the array containing the row and column of the largest value.
   */
   public int [] getLargestLoc(int start_row_index, int start_col_index)
      {
      int [] largest_location = new int[2];

      largest_location[0] = 0;
      largest_location[1] = 0;

      double largest_mag = -1.0;

      for (int row_index = start_row_index;
           row_index < this.rows;
           row_index++)
         {
         for (int col_index = start_col_index;
              col_index < this.cols;
              col_index++)
            {
            int entry_index = row_index * this.cols + col_index;
            double entry_value_mag = Math.abs(this.matrix[entry_index]);
            if (largest_mag < entry_value_mag)
               {
               largest_mag = entry_value_mag;
               largest_location[0] = row_index;
               largest_location[1] = col_index;
               }
            }
         }
      return(largest_location);
      }

   /**
   * Swaps two rows in the matrix.
   * @param row_index1 the index of the first of the rows to swap.
   * @param row_index2 the index of the second of the rows to swap.
   */
   public void swapRows(int row_index1, int row_index2)
      {
      int entry_index1 = row_index1 * this.cols;
      int entry_index2 = row_index2 * this.cols;
      for (int col_index = 0;
           col_index < this.cols;
           col_index++)
         {
         double value = this.matrix[entry_index1];
         this.matrix[entry_index1] = this.matrix[entry_index2];
         this.matrix[entry_index2] = value;
         entry_index1++;
         entry_index2++;
         }
      }

   /**
   * Permutes the rows of the matrix using the permutation ordering of the
   * integer indices vector.
   * @param permutation_indices the vector of permuted indices.
   */
   public void permuteRows(Integer [] permutation_indices)
      {
      int [] clone_indices = new int[permutation_indices.length];
      /*
      * Get the inverse indices mapping.
      */
/*
      String prefix = "   permutation_indices = (";
*/
      for (int row_index = 0;
           row_index < clone_indices.length;
           row_index++)
         {
         clone_indices[row_index] = permutation_indices[row_index];

/*
         System.out.print(prefix + clone_indices[row_index]);
         prefix = ", ";
*/
         }
/*
      System.out.println(")");
*/

      for (int row_index = 0;
           row_index < clone_indices.length;
           row_index++)
         {
         int row_index1 = clone_indices[row_index];
         if (row_index1 != row_index)
            {
/*
            System.out.println("  swapping rows "
                             + row_index
                             + " and "
                             + row_index1
                              );
*/
            this.swapRows(row_index, row_index1);

            clone_indices[row_index1] = row_index1;
            clone_indices[row_index] = row_index;
            }
         }
/*
      prefix = "   reordered clone_indices = (";
      for (int row_index = 0;
           row_index < clone_indices.length;
           row_index++)
         {
         System.out.print(prefix + clone_indices[row_index]);
         prefix = ", ";
         }
      System.out.println(")");
*/
      }

   /**
   * Scale a single row.
   * @param scale the scale factor to apply.
   * @param row_index the index of the row to scale.
   * @param start_col the column to start on
   */
   public void scaleRow(double scale, int row_index, Integer start_col)
      {
      int start_col_index = 0;
      if (start_col != null)
         {
         start_col_index = start_col;
         }
      for (int col_index = start_col_index;
           col_index < this.cols;
           col_index++)
         {
         int entry_index = row_index * this.cols + col_index;
         this.matrix[entry_index] *= scale;
         }
      }

   /**
   * Linearly combine the first row into the second of the matrix.
   * <pre>
   *    row[2] = row[2] - row[1] * scale
   * </pre>
   * @param row_index1 the index of the row to scale and subtract.
   * @param scale the scale factor to apply.
   * @param row_index2 the index of the row to subtract from.
   * @param start_col the index to start with or 0 if this is null.
   */
   public void combineRows(int row_index1,
                           double scale,
                           int row_index2,
                           Integer start_col)
      {
/*
      System.out.println("Matrix."
                       + "combineRows(row_index1, scale, row_index2, start_col)"
                       + " on ENTRY:\n"
                       + "this:"
                       + this
                       + "   row_index1 = " + row_index1 + "\n"
                       + "   scale = " + scale + "\n"
                       + "   row_index2 = " + row_index2 + "\n"
                       + "   start_col = " + start_col + "\n"
                        );
*/
      int start_col_index = 0;
      if (start_col != null)
         {
         start_col_index = start_col;
         }
      for (int col_index = start_col_index;
           col_index < this.cols;
           col_index++)
         {
         int entry_index1 = row_index1 * this.cols + col_index;
         int entry_index2 = row_index2 * this.cols + col_index;
         this.matrix[entry_index2] -= this.matrix[entry_index1] * scale;
         }
/*
      System.out.println("Matrix."
                       + "combineRows(row_index1, scale, row_index2, start_col)"
                       + " on EXIT:\n"
                       + "this:"
                       + this
                        );
*/
      }

   /**
   * Swaps two columns in the matrix.
   * @param col_index1 the index of the first of the columns to swap.
   * @param col_index2 the index of the second of the columns to swap.
   */
   public void swapColumns(int col_index1, int col_index2)
      {
      int entry_index1 = col_index1;
      int entry_index2 = col_index2;
      for (int row_index = 0;
           row_index < this.rows;
           row_index++)
         {
         double value = this.matrix[entry_index1];
         this.matrix[entry_index1] = this.matrix[entry_index2];
         this.matrix[entry_index2] = value;
         entry_index1 += this.cols;
         entry_index2 += this.cols;
         }
      }

   /**
   * Scale the given column.
   * @param scale the scale factor to apply.
   * @param col_index the index of the column to scale.
   */
   public void scaleColumn(double scale, int col_index, Integer start_row)
      {
      int start_row_index = 0;
      if (start_row != null)
         {
         start_row_index = start_row;
         }
      for (int row_index = start_row_index;
           row_index < this.rows;
           row_index++)
         {
         int entry_index = row_index * this.cols + col_index;
         this.matrix[entry_index] *= scale;
         }
      }

   /**
   * Linearly combine the first column into the second of the matrix.
   * <pre>
   *    col[2] = col[2] - col[1] * scale
   * </pre>
   * @param col_index1 the index of the column to scale and subtract.
   * @param scale the scale factor to apply.
   * @param col_index2 the index of the column to subtract from.
   * @param start_row the index to start with or 0 if this is null.
   */
   public void combineColumns(int col_index1,
                              double scale,
                              int col_index2,
                              Integer start_row)
      {
/*
      System.out.println("Matrix."
                       + "combineColumns(col_index1, scale, col_index2,"
                       + " start_row)"
                       + " on ENTRY:\n"
                       + "this:"
                       + this
                       + "   col_index1 = " + col_index1 + "\n"
                       + "   scale = " + scale + "\n"
                       + "   col_index2 = " + col_index2 + "\n"
                       + "   start_row = " + start_row + "\n"
                        );
*/
      int start_row_index = 0;
      if (start_row != null)
         {
         start_row_index = start_row;
         }
      for (int row_index = start_row_index;
           row_index < this.rows;
           row_index++)
         {
         int entry_index1 = row_index * this.cols + col_index1;
         int entry_index2 = row_index * this.cols + col_index2;
         this.matrix[entry_index2] -= this.matrix[entry_index1] * scale;
         }
/*
      System.out.println("Matrix."
                       + "combineColumns(col_index1, scale, col_index2,"
                       + " start_row)"
                       + " on EXIT:\n"
                       + "this:"
                       + this
                        );
*/
      }

   /**
   * Returns a String that represents the value of this Matrix_NxM.
   * @return the string representation of the matrix.
   */
   public String toString()
      {
      String out_string = "(\n";
      int entry_index = 0;
      for (int row_index = 0;
           row_index < this.rows;
           row_index++)
         {
         String prefix = "   (";
         for (int col_index = 0;
              col_index < this.cols;
              col_index++)
            {
            out_string += String.format("%s" + format,
                                        prefix,
                                        this.matrix[entry_index]);
            prefix = " ";
            entry_index++;
            }
         out_string += ")\n";
         }
      out_string += ")\n";
      return(out_string);
      }

   /**
   * Prints the representation the value of this Matrix_NxM.
   */
   public void Print()
      {
      System.out.print(this);
      }

   /**
   * Transpose the matrix.
   * @return the result matrix.
   */
   public Matrix_NxM Transpose()
      {
      Matrix_NxM result = null;

/*
      System.out.println("Matrix_NxM.Transpose()"
                       + " on ENTRY:\n");
*/

      result = new Matrix_NxM(this.cols, this.rows);
      for (int row_index = 0;
           row_index < this.rows;
           row_index++)
         {
         for (int col_index = 0;
              col_index < this.cols;
              col_index++)
            {
            int entry_index1 = row_index * this.cols + col_index;
            int entry_index2 = col_index * result.cols + row_index;

            result.matrix[entry_index2] = this.matrix[entry_index1];
            }
         }
      return(result);
      }
   }

