
package util.geom;

import java.util.*;
import java.util.regex.*;
import java.io.*;
import java.awt.geom.*;

import util.*;

/**
*  This class allows storing 2-D vector components.
*/
public class Vector2D extends Parsable
   {

   /**
   * Set true to print out debug information.
   */
   public static boolean debug_flag = false;

   /**
   * Vector2D basic (not detailed) value pattern
   */
   static String basic_pattern_regex = "\\s*"
                                     + "("
                                     + "\\("
                                     + "[^\\)]*"
                                     + "\\)"
                                     + ")"
                                     ;


   static Pattern basic_pattern = null;

   static String value_pattern_regex = "\\s*"
                                     + "\\("
                                     + "\\s*"
                                     + "([^,\\s]*)"
                                     + "\\s*"
                                     + ","
                                     + "\\s*"
                                     + "([^\\)\\s]*)"
                                     + "\\s*"
                                     + "\\)"
                                     ;

   static Pattern value_pattern = null;


   public String complex_format = "(%9.3f %+9.3f i)";
   public static String default_format = "(%9.3f, %9.3f)";
   public String format = default_format;

   public double x;
   public double y;

   /**
   *  Creates a Vector2D object with default values of 0.0.
   */
   public Vector2D()
      {
      this.x = 0.0;
      this.y = 0.0;
      }

   /**
   *  Creates a Vector2D object with (x, y) component values.
   *  @param x the x-component.
   *  @param y the y-component.
   */
   public Vector2D(double x, double y)
      {
      this.x = x;
      this.y = y;
      }

   /**
   *  Creates a Vector2D object from a Vector2D argument's component values.
   *  @param vector the other vector.
   */
   public Vector2D(Vector2D vector)
      {
      this.x = vector.x;
      this.y = vector.y;
      }

   /**
   *  Creates a Vector2D object from a Point2D argument's component values.
   *  @param point the point to use.
   */
   public Vector2D(Point2D point)
      {
      this.x = point.getX();
      this.y = point.getY();
      }

   /**
   * Creates a new Vector2D object from the Vector2D's component values.
   * @return the new Vector2D object reference.
   */
   public Vector2D clone()
      {
      Vector2D vector = new Vector2D(this);
      return(vector);
      }

   /**
   * Returns the X component of this Vector2D.
   * @return the X component value.
   */
   public double getX()
      {
      return(this.x);
      }

   /**
   * Returns the Y component of this Vector2D.
   * @return the Y component value.
   */
   public double getY()
      {
      return(this.y);
      }

   /**
   * Returns the component values of this Vector2D in an array.
   * @return the Y component value.
   */
   public double [] getArray()
      {
      double [] array =
         {
         this.x,
         this.y
         };
      return(array);
      }

   /**
   * Returns the a Point2D.Double with the coordinates.
   * @return the Y component value.
   */
   public Point2D getPoint2D()
      {
      return(new Point2D.Double(this.x, this.y));
      }

   /**
   *  Sets the Vector2D component values to the x and y component values.
   *  @param x the x-component.
   *  @param y the y-component.
   */
   public void set(double x, double y)
      {
      this.x = x;
      this.y = y;
      }

   /**
   *  Sets the Vector2D object to the x and y component values of
   *  another vector.
   *  @param vector the other vector.
   */
   public void set(Vector2D vector)
      {
      this.x = vector.x;
      this.y = vector.y;
      }

   /**
   *  Compares the vector and returns true if they are equal.
   *  @param vector the other vector.
   *  @return true if the vectors are the same vector or equal.
   */
   public boolean equals(Vector2D vector)
      {
      boolean are_equal = false;
      if (this == vector ||
          (this.x == vector.x &&
           this.y == vector.y))
         {
         are_equal = true;
         }
      return(are_equal);
      }

   /**
   *  Compares the vector and returns true if they are close.
   *  @param vector the other vector.
   *  @param threshold the threshold (it absolute value is used.)
   *  @return true if the vectors are as close as or closer than the
   *      threshold distance.
   */
   public boolean close(Vector2D vector, double threshold)
      {
      boolean are_close = false;
      if (this.equals(vector))
         {
         are_close = true;
         }
      else
         {
         Vector2D diff_vector = this.subtract(vector);
         if (diff_vector.magnitude() <= Math.abs(threshold))
            {
            are_close = true;
            }
         }
      return(are_close);
      }

   /**
   *  Returns the Euclidean magnitude of these vector components
   *  @param x the X component
   *  @param y the Y component
   */
   public static double magnitude(double x, double y)
      {
      double sumsq = x * x + y * y;
      double mag = Math.sqrt(sumsq);
      
      return(mag);
      }

   /**
   *  Returns the Euclidian magnitude of the Vector2D.
   */
   public double magnitude()
      {
      double sumsq = this.x * this.x + this.y * this.y;
      double mag = Math.sqrt(sumsq);
      
      return(mag);
      }

   /**
   *  Returns the square of the  magnitude of the Vector2D.
   */
   public double magnitudeSQ()
      {
      double sumsq = this.x * this.x + this.y * this.y;
      return(sumsq);
      }

   /**
   *  Returns the Euclidean angle of these vector components in radians.
   *  @param x the X component
   *  @param y the Y component
   */
   public static double angle(double x, double y)
      {
      double angle = Math.atan2(y, x);
      
      return(angle);
      }

   /**
   *  Returns the Euclidian angle of the Vector2D in radians.
   */
   public double angle()
      {
      double angle = Math.atan2(this.y, this.x);
      
      return(angle);
      }

   /**
   *  Returns the polar representation of the vector with 0 degrees (in
   *  radians) to the right and positive angles in the CCW (widdershins)
   *  direction, assuming it was in rectangular coordinates initially.
   */
   public Vector2D getPolar()
      {
      Vector2D result = new Vector2D(this.magnitude(), this.angle());
      return(result);
      }

   /**
   *  Returns the rectangular representation of the vector, assuming it was
   *  in polar coordinates, with 0 degrees (in radians) to the right and
   *  positive angles in the CCW (widdershins) direction.
   */
   public Vector2D getRect()
      {
      Vector2D result = new Vector2D(this.x * Math.cos(this.y),
                                     this.x * Math.sin(this.y));
      return(result);
      }


   /**
   *  Scales the vector in-place.
   *  @param value the scale multiplier.
   */
   public void scale(double value)
      {
      this.x *= value;
      this.y *= value;
      }


   /**
   *  Returns the scaled vector.
   *  @param scale the scale multiplier.
   *  @return a scaled vector.
   */
   public Vector2D getScaled(double scale)
      {
      double x_val = scale * this.x;
      double y_val = scale * this.y;

      Vector2D vector = new Vector2D(x_val, y_val);
      
      return(vector);
      }


   /**
   *  Returns the difference between the two Vector2D vectors.
   *  @param vector1 the first vector
   *  @param vector2 the second vector
   */
   public static Vector2D diff(Vector2D vector1, Vector2D vector2)
      {
      double diff_x = vector1.x - vector2.x;
      double diff_y = vector1.y - vector2.y;

      Vector2D vector = new Vector2D(diff_x, diff_y);
      
      return(vector);
      }

   /**
   *  Returns the difference between the two component vectors.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @return the difference Vector2D.
   */
   public static Vector2D diff(double x1, double y1, double x2, double y2)
      {
      Vector2D diff_vector = new Vector2D(x1 - x2, y1 - y2);
      
      return(diff_vector);
      }

   /**
   *  Returns the difference between the two Vector2D vectors.
   *  @param vector the other vector
   *  @return the difference Vector2D.
   */
   public Vector2D subtract(Vector2D vector)
      {
      double diff_x = this.x - vector.x;
      double diff_y = this.y - vector.y;

      Vector2D diff_vector = new Vector2D(diff_x, diff_y);
      
      return(diff_vector);
      }

   /**
   *  Returns the difference between the two component vectors.
   *  @param x the vector's X component
   *  @param y the vector's Y component
   *  @return the difference Vector2D.
   */
   public Vector2D subtract(double x, double y)
      {
      Vector2D diff_vector = new Vector2D(this.x - x, this.y - y);
      
      return(diff_vector);
      }


   /**
   *  Add the input vector to the current vector.
   *  @param vector the vector to add.
   */
   public void addInto(Vector2D vector)
      {
      this.x += vector.x;
      this.y += vector.y;
      }


   /**
   *  Add the input vector to the current vector.
   *  @param x the x-value to add.
   *  @param y the y-value to add.
   */
   public void addInto(double x, double y)
      {
      this.x += x;
      this.y += y;
      }


   /**
   *  Subtract the input vector from the current vector.
   *  @param vector the vector to subtract.
   */
   public void subtractOut(Vector2D vector)
      {
      this.x -= vector.x;
      this.y -= vector.y;
      }


   /**
   *  Subtract the input point from the current vector.
   *  @param point the point to subtract.
   */
   public void subtractOut(Point2D point)
      {
      this.x -= point.getX();
      this.y -= point.getY();
      }


   /**
   *  Subtract the input vector from the current vector.
   *  @param x the x-value to subtract.
   *  @param y the y-value to subtract.
   */
   public void subtractOut(double x, double y)
      {
      this.x -= x;
      this.y -= y;
      }


   /**
   *  Returns the sum of the two Vector2D vectors.
   *  @param vector1 the first vector
   *  @param vector2 the second vector
   *  @return the sum Vector2D.
   */
   public static Vector2D sum(Vector2D vector1, Vector2D vector2)
      {
      double sum_x = vector1.x + vector2.x;
      double sum_y = vector1.y + vector2.y;

      Vector2D diff_vector = new Vector2D(sum_x, sum_y);
      
      return(diff_vector);
      }


   /**
   *  Returns the sum of the two component vectors.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @return the sum Vector2D.
   */
   public static Vector2D sum(double x1, double y1, double x2, double y2)
      {
      Vector2D sum_vector = new Vector2D(x1 + x2, y1 + y2);
      
      return(sum_vector);
      }

   /**
   *  Returns the sum of the two Vector2D vectors.
   *  @param vector the other vector
   *  @return the sum Vector2D.
   */
   public Vector2D add(Vector2D vector)
      {
      double sum_x = this.x + vector.x;
      double sum_y = this.y + vector.y;

      Vector2D sum_vector = new Vector2D(sum_x, sum_y);
      
      return(sum_vector);
      }

   /**
   *  Returns the sum of the Vector2D and Point2D vectors.
   *  @param point the point
   *  @return the sum Vector2D.
   */
   public Vector2D add(Point2D point)
      {
      double sum_x = this.x + point.getX();
      double sum_y = this.y + point.getY();

      Vector2D sum_vector = new Vector2D(sum_x, sum_y);
      
      return(sum_vector);
      }

   /**
   *  Returns the sum of the two component vectors.
   *  @param x the vector's X component
   *  @param y the vector's Y component
   *  @return the sum Vector2D.
   */
   public Vector2D add(double x, double y)
      {
      Vector2D sum_vector = new Vector2D(this.x + x, this.y + y);
      
      return(sum_vector);
      }

   /**
   *  Adds the Vector2D to the Vector2D.
   *  @param vector the other vector
   *  @return the updated original Vector2D object.
   */
   public Vector2D addTo(Vector2D vector)
      {
      this.x += vector.x;
      this.y += vector.y;

      return(this);
      }

   /**
   *  Adds the Point2D to the Vector2D.
   *  @param point the point
   *  @return the updated original Vector2D object.
   */
   public Vector2D addTo(Point2D point)
      {
      this.x += point.getX();
      this.y += point.getY();

      return(this);
      }

   /**
   *  Adds the vector coordinates to the Vector2D.
   *  @param x the vector's X component
   *  @param y the vector's Y component
   *  @return the updated original Vector2D object.
   */
   public Vector2D addTo(double x, double y)
      {
      this.x += x;
      this.y += y;

      return(this);
      }


   /**
   *  Returns the vector dot product between the given Vector2D vectors.
   *  @param vector1 the first vector.
   *  @param vector2 the second vector.
   *  @return the dot product scalar.
   */
   public static double dotProd(Vector2D vector1, Vector2D vector2)
      {
      double dot_prod = vector1.x * vector2.x + vector1.y * vector2.y;
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given
   *  vector components.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @return the dot product scalar.
   */
   public static double dotProd(double x1, double y1, double x2, double y2)
      {
      double dot_prod = x1 * x2 + y1 * y2;
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given Vector2D.
   *  @param vector the other vector.
   *  @return the dot product scalar.
   */
   public double dotProd(Vector2D vector)
      {
      double dot_prod = this.x * vector.x + this.y * vector.y;
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given
   *  vector components.
   *  @param x the X component
   *  @param y the Y component
   *  @return the dot product scalar.
   */
   public double dotProd(double x, double y)
      {
      double dot_prod = this.x * x + this.y * y;
      
      return(dot_prod);
      }


   /**
   *  Get the negative of the vector
   *  @return the negated vector
   */
   public Vector2D negative()
      {
      Vector2D vector = new Vector2D(-this.x, -this.y);
      return(vector);
      }


   /**
   *  Negate the vector.
   */
   public void negate()
      {
      this.x = - this.x;
      this.y = - this.y;
      }


   /**
   *  Normalizes the vector.
   *  @return the magnitude before normalizing.
   */
   public double normalize()
      {
      double sumsq = this.x * this.x + this.y * this.y;
      double mag = Math.sqrt(sumsq);
      if (mag != 0.0)
         {
         double mag_inv = 1.0 / mag;

         this.x *= mag_inv;
         this.y *= mag_inv;
         }
      
      return(mag);
      }


   /**
   *  Returns the distance from this Vector2D point to the specified coordinate.
   *  @param x the X coordinate
   *  @param y the Y coordinate
   */
   public double distance(double x, double y)
      {
      double distance = 0.0;
      double x_offset = this.x - x;
      double y_offset = this.y - y;

      double sumsq = x_offset * x_offset + y_offset * y_offset;
      
      distance = Math.sqrt(sumsq);
      
      return(distance);
      }


   /**
   *  Returns the distance squared from this Vector2D to the specified
   *  coordinate.
   *  @param x the X coordinate
   *  @param y the Y coordinate
   */
   public double distanceSQ(double x, double y)
      {
      double x_offset = this.x - x;
      double y_offset = this.y - y;

      double sumsq = x_offset * x_offset + y_offset * y_offset;
      
      return(sumsq);
      }


   /**
   *  Returns the distance squared between the two coordinate locations.
   *  @param x1 the first location's X coordinate
   *  @param y1 the first location's Y coordinate
   *  @param x2 the second location's X coordinate
   *  @param y2 the second location's Y coordinate
   */
   public static double distanceSQ(double x1, double y1, double x2, double y2)
      {
      double x_offset = x1 - x2;
      double y_offset = y1 - y2;

      double sumsq = x_offset * x_offset + y_offset * y_offset;
      
      return(sumsq);
      }

   /**
   * Returns a String that represents the value of this Vector2D.
   * @return the string representation of the vector.
   */
   public String toString()
      {
      String out_string = String.format(this.format, this.x, this.y);
      return(out_string);
      }

   /**
   * Prints the representation the value of this Vector2D.
   */
   public void Print()
      {
      System.out.print(this);
      }


   /**
   * The Parsable routines.
   */


   /**
   *  Get the basic "locatable" pattern for scanning text for an occurrance
   *  of a possible match..
   *  @return the pattern for locating candidate substrings.
   */
   public static Pattern getBasicPattern()
      {
      if (basic_pattern == null)
         {
         basic_pattern = Pattern.compile(basic_pattern_regex);
         }
      return(basic_pattern);
      }

   /**
   *  Determines whether the basic pattern is in the line.
   *  @param line the string of text to search for the pattern.
   *  @return the matcher for the occrrance of the object's pattern or null
   *     if it is not found.
   */
   public static Matcher findMatchInLine(String line)
      {
      Matcher matcher = null;
      if (basic_pattern == null)
         {
         basic_pattern = Pattern.compile(basic_pattern_regex);
         }
      if (line != null)
         {
         matcher = basic_pattern.matcher(line);
         }
      return(matcher);
      }

   /**
   *  Extracts the string where the basic pattern occurs in the line.
   *  @param line the string of text to search for the pattern.
   *  @return the string that was matched, or null if not found.
   */
   public static String getMatchingString(String line)
      {
      String match = null;
      return(match);
      }


   /**
   *  Parses the given string and returns the object value.
   *  @param val_string the input string from which to extract the values.
   *  @return the reference to the new object with the included value
   *     or null if it failed.
   */
   public static Vector2D getVector2DValue(String val_string)
      {
      Vector2D value = null;

      if (debug_flag)
         {
         System.out.println("Vector2D.getVector2DValue(string) on ENTRY:\n"
                          + "   val_string = " + val_string + "\n"
                          + "   value_pattern = \""
                          + value_pattern
                          + "\""
                           );
         }
      Matcher value_matcher = null;

      if (value_pattern == null)
         {
         value_pattern = Pattern.compile(value_pattern_regex);
         }

      if (val_string != null)
         {
         value_matcher = value_pattern.matcher(val_string);
         }

      if (value_matcher != null &&
          value_matcher.find())
         {
         String val1_string = value_matcher.group(1).trim();
         String val2_string = value_matcher.group(2).trim();

         if (debug_flag)
            {
            System.out.println("val1_string: " + val1_string);
            System.out.println("val2_string: " + val2_string);
            }

         Double value1 = parseDoubleValue(val1_string);
         Double value2 = parseDoubleValue(val2_string);

         if (debug_flag)
            {
            System.out.println("   value = ("
                             + value1
                             + ", "
                             + value2
                             + ")"
                              );
            }
         value = new Vector2D(value1, value2);
         }
      return(value);
      }


   /**
   *  Parses the given string and returns the object value.
   *  @param val_string the input string from which to extract the values.
   *  @return the reference to the new object with the included value
   *     or null if it failed.
   */
   public static Object getValueObject(String val_string)
      {
      Object value_obj = getVector2DValue(val_string);

      return(value_obj);
      }
   }

