
package plf;

import java.util.*;

/**
*  This class allows the user to define a piecewise linear function.
*/
public class Function_1D
   {
   /**
   *  The name of the function represented.
   */
   String name;

   /**
   *  The extents of the x-value domain.
   */
   Double min_x_value;
   Double max_x_value;
   Double range_x_value;

   /**
   *  The array of y_values in the function representation.
   */
   double[] y_values;

   /**
   *  The extents of the y-value range.
   */
   Double min_y_value;
   Double max_y_value;
   Double range_y_value;

   /**
   *  The index of the last point in the representation.
   */
   int last_point_index;

   /**
   *  The increment between indices for the x_values.
   */
   Double x_increment;

   /**
   *  The scale factor from the x_values to the indices.
   */
   Double x_scale_factor;

   /**
   *  The constructor stores the function value defintions.  The
   *  min_x-value and max_x_value represent the domain range for the
   *  function.  The y_values must be the values for equally spaced
   *  x_values, including the bounds of the domain. This function assumes
   *  that the sequence of value pairs defines an adequate approximation of
   *  a continuous function over the interval of x_values.
   *  @param name The name of the function represented.
   *  @param min_x_value The minimum x_value.
   *  @param max_x_value The maximum x_value.
   *  @param y_values The array of y-axis values.
   *  @throws Exception if discrepencies are identified.
   */
   public Function_1D(String name,
                      double min_x_value,
                      double max_x_value,
                      double[] y_values)
      throws Exception
      {
      if (y_values.length <= 1)
         {
         throw(new Exception("plf.Function_1D(name, x_values, y_values)"
                           + " constructor ERROR detected:\n"
                           + "   The y_values array has 1 or fewer entries.\n"
                            ));
         }
      if (min_x_value >= max_x_value)
         {
         throw(new Exception("plf.Function_1D(name, x_values, y_values)"
                           + " constructor ERROR detected:\n"
                           + "   The min_x_value is not less than the"
                           + " max_x_value.\n"
                            ));
         }
      min_y_value = y_values[0];
      max_y_value = y_values[0];
      for (int point_index = 1;
           point_index < y_values.length;
           point_index++)
         {
         if (min_y_value > y_values[point_index])
            {
            min_y_value = y_values[point_index];
            }
         if (max_y_value < y_values[point_index])
            {
            max_y_value = y_values[point_index];
            }
         }


      this.name = name;
      this.min_x_value = min_x_value;
      this.max_x_value = max_x_value;
      this.range_x_value = max_x_value - min_x_value;

      this.y_values = y_values;

      this.last_point_index = y_values.length - 1;
      double interval_count = this.last_point_index;

      this.x_scale_factor = interval_count / range_x_value;
      this.x_increment = 1.0 / this.x_scale_factor;
/*
      System.out.println("*** ["
                       + this.min_x_value
                       + "   "
                       + this.max_x_value
                       + "] "
                       + this.range_x_value
                       + "\n   "
                       + interval_count
                       + " "
                       + this.x_scale_factor
                       + " "
                       + this.x_increment
                       + "\n"
                        );
*/
      }

   /**
   *  The constructor stores the function value defintions.  The
   *  min_x-value and max_x_value represent the domain range for the
   *  function.  The y_values must be the values for equally spaced
   *  x_values, including the bounds of the domain. This function assumes
   *  that the sequence of value pairs defines an adequate approximation of
   *  a continuous function over the interval of x_values.
   *  @param name The name of the function represented.
   *  @param min_x_value The minimum x_value.
   *  @param max_x_value The maximum x_value.
   *  @param y_values The array of integer y-axis values.
   *  @throws Exception if discrepencies are identified.
   */
   public Function_1D(String name,
                      double min_x_value,
                      double max_x_value,
                      int[] y_values)
      throws Exception
      {
      if (y_values.length <= 1)
         {
         throw(new Exception("plf.Function_1D(name, x_values, y_values)"
                           + " constructor ERROR detected:\n"
                           + "   The y_values array has 1 or fewer entries.\n"
                            ));
         }
      if (min_x_value >= max_x_value)
         {
         throw(new Exception("plf.Function_1D(name, x_values, y_values)"
                           + " constructor ERROR detected:\n"
                           + "   The min_x_value is not less than the"
                           + " max_x_value.\n"
                            ));
         }
      this.y_values = new double[y_values.length];
      double y_value = (double) y_values[0];
      this.y_values[0] = y_value;
      min_y_value = y_value;
      max_y_value = y_value;
      for (int point_index = 1;
           point_index < y_values.length;
           point_index++)
         {
         y_value = (double) y_values[point_index];
         this.y_values[point_index] = y_value;
         if (min_y_value > y_value)
            {
            min_y_value = y_value;
            }
         if (max_y_value < y_value)
            {
            max_y_value = y_value;
            }
         }


      this.name = name;
      this.min_x_value = min_x_value;
      this.max_x_value = max_x_value;
      this.range_x_value = max_x_value - min_x_value;

      this.last_point_index = y_values.length - 1;
      double interval_count = this.last_point_index;

      this.x_scale_factor = interval_count / range_x_value;
      this.x_increment = 1.0 / this.x_scale_factor;
/*
      System.out.println("*** ["
                       + this.min_x_value
                       + "   "
                       + this.max_x_value
                       + "] "
                       + this.range_x_value
                       + "\n   "
                       + interval_count
                       + " "
                       + this.x_scale_factor
                       + " "
                       + this.x_increment
                       + "\n"
                        );
*/
      }

   /**
   *  Get the function value for a given input value.
   *  @param x_value the argument to the function.
   *  @return the y_value approximation.
   *  @throws Exception
   */
   public double getValue(double x_value)
      throws Exception
      {
      double y_value;
      if (x_value < this.min_x_value)
         {
         throw(new Exception("plf.getValue(x_value)"
                           + " ERROR detected:\n"
                           + "   The x_value ("
                           + x_value
                           + ") is less than the"
                           + " min_x_value ("
                           + min_x_value
                           + ").\n"
                            ));
         }
      if (x_value > this.max_x_value)
         {
         throw(new Exception("plf.getValue(x_value)"
                           + " ERROR detected:\n"
                           + "   The x_value ("
                           + x_value
                           + ") is greater than the"
                           + " max_x_value ("
                           + max_x_value
                           + ").\n"
                            ));
         }
      /*
      *  If not an integer then truncate to the next lower integer,
      *  and get the offset into the interval.
      */
      Double x_index = (x_value - this.min_x_value) * this.x_scale_factor;
      Integer low_index = x_index.intValue();

      Double x_offset_fraction = x_index - low_index.doubleValue();

      if (x_offset_fraction <= 0.0001)
         {
         /*
         *  No need to interpolate.
         */
         y_value = this.y_values[low_index];
         }
      else if (low_index >= last_point_index)
         {
         /*
         *  No need to interpolate.
         */
         y_value = this.y_values[last_point_index];
         }
      else
         {
         /*
         *  It is between x_values, so we need to interpolate.
         */
         y_value = (this.y_values[low_index] * (1.0 - x_offset_fraction)) +
                   (this.y_values[low_index + 1] * x_offset_fraction);
/*
         String out_line = String.format("-- %8.3g %5.2f %3d: "
                                       + " %8.3g -- [%8.3g, %8.3g] -- %8.3g",
                                         x_value,
                                         x_index,
                                         low_index,
                                         x_offset_fraction,
                                         this.y_values[low_index],
                                         this.y_values[low_index + 1],
                                         y_value
                                        );
         System.out.println(out_line);
*/
         }
      return(y_value);
      }

   /**
   *  Get the function y_values for all input values as an array.
   *  @return the y_values array.
   */
   public double [] getValues()
      {
      return(this.y_values);
      }

   /**
   *  Get the integrated curve values.
   *  @param name The name for the new function.
   *  @return the integrated function.
   *  @throws Exception
   */
   public Function_1D Integrate(String name)
      throws Exception
      {
      Function_1D result = null;

      double[] y_values = new double[this.y_values.length];
      y_values[0] = 0.0;
      for (int point_index = 1;
           point_index < this.y_values.length;
           point_index++)
         {
         /*
         *  Use the simple trapezoidal rule for the integration.
         */
         y_values[point_index] = y_values[point_index - 1] +
                                 (this.y_values[point_index - 1] +
                                  this.y_values[point_index]) * 0.5;
         }
      
      result = new Function_1D(name,
                               this.min_x_value,
                               this.max_x_value,
                               y_values);
      return(result);
      }

   /**
   *  Get the differentiated curve values.
   *  @param name The name for the new function.
   *  @throws Exception
   *  @return the differentiated function.
   */
   public Function_1D Differentiate(String name)
      throws Exception
      {
      Function_1D result = null;

      double[] y_values = new double[this.y_values.length];
      y_values[0] = this.y_values[1] -
                    this.y_values[0];
      for (int point_index = 1;
           point_index < this.y_values.length;
           point_index++)
         {
         y_values[point_index] = this.y_values[point_index] -
                                 this.y_values[point_index - 1];
         }
      
      result = new Function_1D(name,
                               this.min_x_value,
                               this.max_x_value,
                               y_values);
      return(result);
      }

   /**
   *  Get the zero crossing x_values, if any.  Remember that zero crossings
   *  can be lost by using too course a step in piecewise linear functions.
   *  @return the ArrayList of estimated zero crossing x-values.
   */
   public ArrayList<Double> getZeros()
      {
      ArrayList<Double> result = new ArrayList<Double>();

      int num_intervals = 1;
      double last_value = this.y_values[0];
      int last_sign = 0;
      if (last_value < 0.0)
         {
         last_sign = -1;
         }
      else if (last_value > 0.0)
         {
         last_sign = 1;
         }
      for (int point_index = 1;
           point_index < this.y_values.length;
           point_index++)
         {
         double curr_value = this.y_values[point_index];
         int curr_sign = 0;
         if (curr_value < 0.0)
            {
            curr_sign = -1;
            }
         else if (curr_value > 0.0)
            {
            curr_sign = 1;
            }

         if (last_sign == 0)
            {
            last_sign = curr_sign;
            last_value = curr_value;
            num_intervals = 1;
            }
         else if (curr_sign == 0)
            {
            /*
            *  Keep whatever is the last sign and value.
            */
            if (last_sign != 0)
               {
               num_intervals++;
               }
            }
         else if (last_sign * curr_sign < 0)
            {
            /*
            * It crossed zero, estimate the x-value of the crossing.
            */
            double delta_y = Math.abs(curr_value - last_value);
            double curr_y_ratio = Math.abs(curr_value) / delta_y;
            double last_y_ratio = Math.abs(last_value) / delta_y;

            double x_span = this.x_increment *
                                    (double) num_intervals;
         
            Double curr_x = (double) point_index * this.x_increment +
                            this.min_x_value;
            Double last_x = curr_x - x_span +
                            this.min_x_value;
            Double x_value = curr_x * last_y_ratio + last_x * curr_y_ratio;

            result.add(x_value);

/*
            System.out.println("plf.Function_1D.getZeros() STATUS:\n"
                             + "   last_value = " + last_value + "\n"
                             + "   last_sign = " + last_sign + "\n"
                             + "   curr_value = " + curr_value + "\n"
                             + "   curr_sign = " + curr_sign + "\n"
                             + "   delta_y = " + delta_y + "\n"
                             + "   curr_y_ratio = " + curr_y_ratio + "\n"
                             + "   last_y_ratio = " + last_y_ratio + "\n"
                             + "   num_intervals = " + num_intervals + "\n"
                             + "   x_span = " + x_span + "\n"
                             + "   x_scale_factor = " + x_scale_factor + "\n"
                             + "   this.x_scale_factor = "
                                + this.x_scale_factor
                                + "\n"
                             + "   this.x_increment = "
                                + this.x_increment
                                + "\n"
                             + "   this.min_x_value = "
                                + this.min_x_value
                                + "\n"
                             + "   this.max_x_value = "
                                + this.max_x_value
                                + "\n"
                             + "   this.last_point_index = "
                                + this.last_point_index
                                + "\n"
                             + "   curr_x = " + curr_x + "\n"
                             + "   last_x = " + last_x + "\n"
                             + "   x_value = " + x_value + "\n"
                              );
*/


            last_value = curr_value;
            last_sign = curr_sign;
            num_intervals = 1;
            }
         }
      return(result);
      }

   /**
   *  Formats the function definition into a string.
   */
   public String toString()
      {
      String out_string = "Function_1D: "
                        + this.name
                        + "() using "
                        + this.y_values.length
                        + " points.\n";
      String out_line = new String();
      for (int point_index = 0;
           point_index < this.y_values.length;
           point_index++)
         {
         String.format("   %3d:  (%9.3g, %9.3g)\n",
                       point_index,
                       (this.x_increment * point_index + this.min_x_value),
                       this.y_values[point_index]
                      );
         out_string += out_line;
         }
      out_string += "\n";
      return(out_string);
      }

   /**
   *  Formats the function definition into a string and prints it.
   */
   public void Print()
      {
      System.out.println("Function_1D: "
                       + this.name
                       + "() using "
                       + this.y_values.length
                       + " points.\n");
      for (int point_index = 0;
           point_index < this.y_values.length;
           point_index++)
         {
         System.out.format("   %3d:  (%9.3g, %9.3g)\n",
                           point_index,
                           (this.x_increment * point_index + this.min_x_value),
                           this.y_values[point_index]
                          );
         }
      System.out.println("\n");
      }
   }

