
/*
*  The definition for a single plot item (one line).
*/

package plot;


import java.util.*;
import java.awt.*;


/**
*  This class defines the basic PlotItem structure used by the PlotPanel
*  to display the plot data.
*/

public class PlotItem
   {

   String title = null;

   int num_points = 0;

   String marker_type = null; /* Code for marker type. */

   boolean is_selected;       /* Set true if item selected for plot. */
   int line_flag;             /* True for line, False for vertices. */
   int line_type;             /* Code for line type. */


   int radius = 3;            /* The radius of the dot, or none if &lt; 0 */
   Color color;               /* The color to use for this item. */


   Double [] x_values = null;
   Double [] x_log_values = null;

   Double x_data_max = null;
   Double x_data_min = null;
   Double x_data_range = null;
   Double x_data_scale = null;

   Double [] y_values = null;
   Double [] y_log_values = null;

   Double y_data_max = null;
   Double y_data_min = null;
   Double y_data_range = null;
   Double y_data_scale = null;

   Double [] z_values = null;
   Double [] z_log_values = null;

   Double z_data_max = null;
   Double z_data_min = null;
   Double z_data_range = null;
   Double z_data_scale = null;


   /**
   * Create a plot item to include in the plot and key.
   * @param key_string the title to give this item's line in the plot.
   * @param x_values the array of x values for the plot line.
   * @param y_values the array of y values for the plot line.
   * @param color the color to give the line.
   */
   public PlotItem(String key_string,
                   double [] x_values,
                   double [] y_values,
                   Color color)
      {
      System.out.println("PlotItem::constructor(\""
                       + key_string
                       + "\", "
                       + "double[] x_values, "
                       + "double[] y_values, "
                       + "color)"
                       + " ON ENTRY"
                        );
      this.title = key_string;

      this.num_points = x_values.length;

      this.x_values = new Double[this.num_points];
      this.y_values = new Double[this.num_points];

      /*
      *  Copy the values into the local array.
      */
      for (int value_index = 0;
           value_index < this.num_points;
           value_index++)
         {
         this.x_values[value_index] = x_values[value_index];
         this.y_values[value_index] = y_values[value_index];
         }

      this.color = color;
      this.is_selected = true;
      this.getStatistics();
      }


   /**
   * Create a 3-D plot item to include in the plot and key.
   * @param key_string the title to give this item's line in the plot.
   * @param x_values the array of x values for the plot line.
   * @param y_values the array of y values for the plot line.
   * @param z_values the array of z values for the plot line.
   * @param color the color to give the line.
   */
   public PlotItem(String key_string,
                   double [] x_values,
                   double [] y_values,
                   double [] z_values,
                   Color color)
      {
      System.out.println("PlotItem::constructor(\""
                       + key_string
                       + "\", "
                       + "double[] x_values, "
                       + "double[] y_values, "
                       + "double[] z_value, "
                       + "color)"
                       + " ON ENTRY"
                        );
      this.title = key_string;

      this.num_points = x_values.length;

      this.x_values = new Double[this.num_points];
      this.y_values = new Double[this.num_points];
      this.z_values = new Double[this.num_points];

      /*
      *  Copy the values into the local array.
      */
      for (int value_index = 0;
           value_index < this.num_points;
           value_index++)
         {
         this.x_values[value_index] = x_values[value_index];
         this.y_values[value_index] = y_values[value_index];
         this.z_values[value_index] = z_values[value_index];
         }

      this.color = color;
      this.is_selected = true;
      this.getStatistics();
      }


   /**
   * Create a plot item to include in the plot and key.
   * @param key_string the title to give this item's line in the plot.
   * @param x_values the ArrayList of x values for the plot line.
   * @param y_values the ArrayList of y values for the plot line.
   * @param color the color to give the line.
   */
   public PlotItem(String key_string,
                   ArrayList<Double> x_values,
                   ArrayList<Double> y_values,
                   Color color)
      {
      this.title = key_string;
      System.out.println("PlotItem::constructor(\""
                       + key_string
                       + "\", "
                       + "ArrayList<Double> x_values, "
                       + "ArrayList<Double> y_values, "
                       + "color)"
                       + " ON ENTRY"
                        );

      this.num_points = x_values.size();

      this.x_values = new Double[this.num_points];
      this.y_values = new Double[this.num_points];

      /*
      *  Copy the values into the local array.
      */
      for (int point_index = 0;
           point_index < this.num_points;
           point_index++)
         {
         this.x_values[point_index] = x_values.get(point_index);
         this.y_values[point_index] = y_values.get(point_index);
         }

      this.color = color;
      this.is_selected = true;
      this.getStatistics();
      }


   /**
   * Create a 3-D plot item to include in the plot and key.
   * @param key_string the title to give this item's line in the plot.
   * @param x_values the ArrayList of x values for the plot line.
   * @param y_values the ArrayList of y values for the plot line.
   * @param z_values the ArrayList of z values for the plot line.
   * @param color the color to give the line.
   */
   public PlotItem(String key_string,
                   ArrayList<Double> x_values,
                   ArrayList<Double> y_values,
                   ArrayList<Double> z_values,
                   Color color)
      {
      System.out.println("PlotItem::constructor(\""
                       + key_string
                       + "\","
                       + " ArrayList<Double> x_values,"
                       + " ArrayList<Double> y_values,"
                       + " ArrayList<Double> z_value,"
                       + " color)"
                       + " ON ENTRY"
                        );
      this.title = key_string;

      this.num_points = x_values.size();

      this.x_values = new Double[this.num_points];
      this.y_values = new Double[this.num_points];
      this.z_values = new Double[this.num_points];

      /*
      *  Copy the values into the local array.
      */
      for (int point_index = 0;
           point_index < this.num_points;
           point_index++)
         {
         this.x_values[point_index] = x_values.get(point_index);
         this.y_values[point_index] = y_values.get(point_index);
         this.z_values[point_index] = z_values.get(point_index);
         }

      this.color = color;
      this.is_selected = true;
      this.getStatistics();
      }


   /**
   * Create a plot item to include in the plot and key.
   * @param key_string the title to give this item's line in the plot.
   * @param x_values the array of x values for the plot line.
   * @param y_values the array of integer y values for the plot line.
   * @param color the color to give the line.
   */
   public PlotItem(String key_string,
                   double [] x_values,
                   int [] y_values,
                   Color color)
      {
      System.out.println("PlotItem::constructor(\""
                       + key_string
                       + "\", double x_values, int y_values, color)"
                       + " ON ENTRY"
                        );
      this.title = key_string;

      this.num_points = x_values.length;

      this.x_values = new Double[this.num_points];
      this.y_values = new Double[this.num_points];

      /*
      *  Copy the values into the local array.
      */
      for (int value_index = 0;
           value_index < this.num_points;
           value_index++)
         {
         this.x_values[value_index] = x_values[value_index];
         this.y_values[value_index] = (double) y_values[value_index];
         }

      this.color = color;
      this.is_selected = true;
      this.getStatistics();
      }


   /**
   * Create a 3-D plot item to include in the plot and key.
   * @param key_string the title to give this item's line in the plot.
   * @param x_values the array of x values for the plot line.
   * @param y_values the array of integer y values for the plot line.
   * @param z_values the array of integer y values for the plot line.
   * @param color the color to give the line.
   */
   public PlotItem(String key_string,
                   double [] x_values,
                   int [] y_values,
                   int [] z_values,
                   Color color)
      {
      System.out.println("PlotItem::constructor(\""
                       + key_string
                       + "\","
                       + " double x_values, int y_values, int z_values,"
                       + " color)"
                       + " ON ENTRY"
                        );
      this.title = key_string;

      this.num_points = x_values.length;

      this.x_values = new Double[this.num_points];
      this.y_values = new Double[this.num_points];
      this.z_values = new Double[this.num_points];

      /*
      *  Copy the values into the local array.
      */
      for (int value_index = 0;
           value_index < this.num_points;
           value_index++)
         {
         this.x_values[value_index] = x_values[value_index];
         this.y_values[value_index] = (double) y_values[value_index];
         this.z_values[value_index] = (double) z_values[value_index];
         }

      this.color = color;
      this.is_selected = true;
      this.getStatistics();
      }


   /**
   * Create a plot item to include in the plot and key.
   * @param key_string the title to give this item's line in the plot.
   * @param values the ArrayList of entries of ArrayList (x, y) pairs.
   * @param first_entry_index the index of the first (x) value.
   * @param second_entry_index the index of the second (y) value.
   * @param color the color to give the line.
   */
   public PlotItem(String key_string,
                   ArrayList<ArrayList<Double>> values,
                   int first_entry_index,
                   int second_entry_index,
                   Color color)
      {
      this.title = key_string;

      this.num_points = values.size();

      this.x_values = new Double[this.num_points];
      this.y_values = new Double[this.num_points];

      /*
      *  Copy the values into the local array.
      */
      for (int point_index = 0;
           point_index < this.num_points;
           point_index++)
         {
         ArrayList<Double> entry = values.get(point_index);
         this.x_values[point_index] = entry.get(first_entry_index);
         this.y_values[point_index] = entry.get(second_entry_index);
         }

      this.color = color;
      this.is_selected = true;
      this.getStatistics();
      }


   /**
   * Create a plot item to include in the plot and key.
   * @param key_string the title to give this item's line in the plot.
   * @param values the ArrayList of entries of ArrayList values.
   * @param first_entry_index the index of the first (x) value.
   * @param second_entry_index the index of the second (y) value.
   * @param third_entry_index the index of the third (z) value.
   * @param radius the radius of the dot to use (&lt; 1 for just a line).
   * @param color the color to give the line.
   */
   public PlotItem(String key_string,
                   ArrayList<ArrayList<Double>> values,
                   int first_entry_index,
                   int second_entry_index,
                   int third_entry_index,
                   int radius,
                   Color color)
      {
      this.title = key_string;

      this.num_points = values.size();

      this.x_values = new Double[this.num_points];
      this.y_values = new Double[this.num_points];
      this.z_values = new Double[this.num_points];

      /*
      *  Copy the values into the local array.
      */
      for (int point_index = 0;
           point_index < this.num_points;
           point_index++)
         {
         ArrayList<Double> entry = values.get(point_index);
         this.x_values[point_index] = entry.get(first_entry_index);
         this.y_values[point_index] = entry.get(second_entry_index);
         this.z_values[point_index] = entry.get(third_entry_index);
         }

      this.radius = radius;
      this.color = color;
      this.is_selected = true;
      this.getStatistics();
      }


  /**
  *  Get the statistics on the value stored.
  */
  public void getStatistics()
      {
      System.out.println("Statistics:\n"
                       + "   num_points = " + num_points
                        );
/*
*/

      if (this.x_values != null)
         {
         this.x_data_max = this.x_values[0];
         this.x_data_min = this.x_values[0];

         for (int value_index = 1;
              value_index < this.num_points;
              value_index++)
            {
            Double x_value = this.x_values[value_index];

            if (this.x_data_max < x_value)
               {
               this.x_data_max = x_value;
               }
            else if (this.x_data_min > x_value)
               {
               this.x_data_min = x_value;
               }
            }

         this.x_data_range = this.x_data_max - this.x_data_min;
         if (this.x_data_range != 0)
            {
            this.x_data_scale = 1.0 / this.x_data_range;
            }
         else
            {
            this.x_data_scale = 1.0;
            }

         System.out.println("   x values --\n"
                          + "      max = " + this.x_data_max + "\n"
                          + "      min = " + this.x_data_min + "\n"
                          + "      range = " + this.x_data_range + "\n"
                          + "      scale = " + this.x_data_scale
                           );
/*
*/
         }

      if (this.y_values != null)
         {
         this.y_data_max = this.y_values[0];
         this.y_data_min = this.y_values[0];

         for (int value_index = 1;
              value_index < this.num_points;
              value_index++)
            {
            Double y_value = this.y_values[value_index];

            if (this.y_data_max < y_value)
               {
               this.y_data_max = y_value;
               }
            else if (this.y_data_min > y_value)
               {
               this.y_data_min = y_value;
               }
            }

         this.y_data_range = this.y_data_max - this.y_data_min;
         if (this.y_data_range != 0)
            {
            this.y_data_scale = 1.0 / this.y_data_range;
            }
         else
            {
            this.y_data_scale = 1.0;
            }

         System.out.println("   y values --\n"
                          + "      max = " + this.y_data_max + "\n"
                          + "      min = " + this.y_data_min + "\n"
                          + "      range = " + this.y_data_range + "\n"
                          + "      scale = " + this.y_data_scale
                           );
/*
*/
         }

      if (this.z_values != null)
         {
         this.z_data_max = this.z_values[0];
         this.z_data_min = this.z_values[0];

         for (int value_index = 1;
              value_index < this.num_points;
              value_index++)
            {
            Double z_value = this.z_values[value_index];

            if (this.z_data_max < z_value)
               {
               this.z_data_max = z_value;
               }
            else if (this.z_data_min > z_value)
               {
               this.z_data_min = z_value;
               }
            }

         this.z_data_range = this.z_data_max - this.z_data_min;
         if (this.z_data_range != 0)
            {
            this.z_data_scale = 1.0 / this.z_data_range;
            }
         else
            {
            this.z_data_scale = 1.0;
            }

         System.out.println("   z values --\n"
                          + "      max = " + this.z_data_max + "\n"
                          + "      min = " + this.z_data_min + "\n"
                          + "      range = " + this.z_data_range + "\n"
                          + "      scale = " + this.z_data_scale
                           );
/*
*/
         }

         System.out.println();
/*
*/
      }


   /**
   *  Get the linear collection of values to be plotted.  These are
   *  the logs of the values if the log_flag is true.
   *  @param log_flag if true the log10 of each value is to be returned.
   *  @param values the base axis values.
   *  @param log_values the log axis values (in relevant or stored).
   */
   public Double [] getValues(boolean log_flag,
                              Double [] values,
                              Double [] log_values)
      {
      Double [] res_values = new Double[values.length];
      if (log_flag)
         {
         for (int value_index = 0;
              value_index < values.length;
              value_index++)
            {
            Double log_value = null;
            if (log_values != null)
               {
               log_value = log_values[value_index];
               }
            if (log_value == null)
               {
               Double value = values[value_index];
               if (value != null &&
                   value != Double.NaN &&
                   value != Double.POSITIVE_INFINITY &&
                   value != Double.NEGATIVE_INFINITY)
                  {
                  log_value = Math.log10(value);
                  }
               if (log_values != null)
                  {
                  log_values[value_index] = log_value;
                  }
               }
            res_values[value_index] = log_value;
            }
         }
      else
         {
         for (int value_index = 0;
              value_index < values.length;
              value_index++)
            {
            Double value = values[value_index];
            if (value != null &&
                value != Double.NaN &&
                value != Double.POSITIVE_INFINITY &&
                value != Double.NEGATIVE_INFINITY)
               {
               res_values[value_index] = value;
               }
            }
         }
      return(res_values);
      }


   /**
   *  Get the plot item's title.
   *  @return the title string.
   */
   public String getTitle()
      {
      return(this.title);
      }

   /**
   *  See if the plot item is selected.
   *  @return the "is_selected" flag value.
   */
   public boolean isSelected()
      {
      return(this.is_selected);
      }

   /**
   *  See if the plot item is selected.
   *  @param new_radius the new radius for the item's marker.
   */
   public void setMarkerRadius(int new_radius)
      {
      this.radius = new_radius;
      }

   /**
   *  Set the marker type string for the item.
   *  @param plot_2d the parent plot context.
   *  @param type null sets it to the default.
   */
   public void setMarkerType(Plot2D plot_2d,
                             String type)
      {

      if (plot_2d != null)
         {
         HashMap<String, Boolean> marker_available =
                                 plot_2d.getMarkerAvailabilities();
         if (this.marker_type != null &&
             marker_available.containsKey(this.marker_type))
            {
            marker_available.put(this.marker_type, true);
            }
         this.marker_type = null;
         if (marker_available.containsKey(type))
            {
            this.marker_type = type;
            marker_available.put(this.marker_type, false);
            }
         }
      }

   /**
   *  Get the marker type string for the item.
   *  @return null is for the default.
   */
   public String getMarkerType()
      {
      return(this.marker_type);
      }

   }

