
/*
*  The Plot2D d3finition.
*/

package plot;


import java.util.*;

import java.awt.*;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.awt.geom.Rectangle2D;

import util.*;
import plot.*;

/**
*  This class defines the scrolled plot structure used by the PlotPanel
*  to display the plot data using a scrolled image in a plot area with
*  straight plot lines for each plot item (data line), and the lines
*  vary by color or brightness based on each point value.
*/
public class ScrolledPlot
   {

   String title = "Title";
   String x_axis_title = "X-Axis";
   String y_axis_title = "Y-Axis";

   boolean colors_selected = false;

   double x_min;
   double x_max;
   double y_min;
   double y_max;

   int plot_width = 0;
   int plot_height = 0;
   int col_width = 0;
   int col_height = 0;
   int cell_height = 0;


   Axis axis_x = null;
   Axis axis_y = null;


   Color title_color = Color.BLACK;
   Color grid_color = Color.darkGray;
   Color plot_color = Color.BLACK;

   /*
   *  These are based on the range of values for the value arrays.
   */
   double max_value;
   double min_value;

   /*
   *  These are based on the number of samples and what the horizontal
   *  axis should range over.
   */
   int num_cols_max = 0;

   ArrayList<Double> x_values = null;

   double x_values_max = 1.0;
   double x_values_min = 0.0;
   double x_values_range = 1.0;
   double x_values_incr = 0.0;


   /*
   *  These are based on the value magnitudes.
   */
   double value_min = -1.0;
   double value_max = 1.0;
   double value_range = 2.0;

   /*
   *  The values for each column of magnitudes.
   */
   ArrayList<ArrayList<Double>> magnitudes = null;

   ArrayList<Double> y_values = null;
   double y_values_max = 1.0;
   double y_values_min = 0.0;

   double y_values_range = 1.0;
   double y_values_scale = 1.0;


   /**
   *  These are the evenly spaced values in the rainbow.
   */
   static float[][] rb_rgb_values =
      {
         { 0.5F,  0.05F, 0.5F},   // Purple       at index 0 of 101 values
         { 0.0F,  0.0F,  0.75F},  // Blue         at index 15
         { 0.0F,  0.75F, 1.0F},   // Blue-green   at index 29
         { 0.0F,  0.6F,  0.0F},   // Green        at index 43
         { 0.0F,  1.0F,  0.0F},   // Light green  at index 57
         { 1.0F,  1.0F,  0.0F},   // Yellow       at index 73
         { 1.0F,  0.5F,  0.0F},   // Orange       at index 85
         { 1.0F,  0.0F,  0.0F}    // Red          at index 100 of 101 values
      };


   /**
   *  Create a ScrolledPlot instance, setting titles and assuming a
   *  linear horizontal axis.
   *  @param title the plot title
   *  @param x_axis_title the horizontal axis title.
   *  @param y_axis_title the vertical axis title.
   *  @param num_x_values_max the number of horizontal values to include,
   *         dropping extra ones off the left side as the plot scrolls.
   *  @param x_values_incr the amount to increment the horizontal "tics"
   *         for each new sample column.
   */
   public ScrolledPlot(String title,
                       String x_axis_title,
                       String y_axis_title,
                       int num_x_values_max,
                       double x_values_min,
                       double x_values_incr)
      {
      this.title = title;
      this.x_axis_title = x_axis_title;
      this.y_axis_title = y_axis_title;

      this.num_cols_max = num_x_values_max;

      this.x_values_incr = x_values_incr;
      this.x_values_min = x_values_min;
      this.x_values_max = this.x_values_min +
                          this.x_values_incr * (this.num_cols_max - 1);
      this.x_values_range = this.x_values_max - this.x_values_min;

      this.col_width = 100 / this.num_cols_max;
      this.col_height = 20;

      this.plot_width = 100;
/*
      System.out.println("ScrolledPlot Constructor on ENTRY:\n"
                       + "   title = " + this.title + "\n"
                       + "   x_axis_title = " + this.x_axis_title + "\n"
                       + "   y_axis_title = " + this.y_axis_title + "\n"
                       + "   num_cols_max = " + this.num_cols_max + "\n"
                       + "   x_values_min = " + this.x_values_min + "\n"
                       + "   x_values_incr = " + this.x_values_incr + "\n"
                       + "   x_values_max = " + this.x_values_max + "\n"
                        );
*/

      this.axis_x = new Axis(this.x_axis_title);
      this.axis_y = new Axis(this.y_axis_title);

      try
         {
         this.axis_x.updateAxis(this.x_axis_title,
                                this.x_values_min,
                                this.x_values_max,
                                false,
                                true);
         }
      catch (Exception except)
         {
         System.out.println("ScrolledPlot::constructor ERROR:\n"
                            + "   \"" + except + "\"");
         except.printStackTrace(System.out);
         }

      this.x_values = new ArrayList<Double>();

      this.magnitudes = new ArrayList<ArrayList<Double>>();

/*
      System.out.println("ScrolledPlot Constructor on EXIT.");
*/
      }


   /**
   *  Set the vertical values for the lines to include.  This gets the
   *  min and scale factor for later rescaling of the input values as
   *  needed for various plot sizes.
   *  @param y_values the values for each of the vertical locations.
   */
   public void setVerticals(double [] y_values)
      {
      System.out.println("ScrolledPlot::setVerticals(y_values) on ENTRY:\n"
                       + "   title = " + this.title + "\n"
                       + "   y_values.length = " + y_values.length + "\n"
                        );
/*
*/
      if (y_values != null && y_values.length > 0)
         {
         this.y_values = new ArrayList<Double>();

         System.out.println("ScrolledPlot::setVerticals() y_values:");
/*
*/
         for (int value_index = 0;
              value_index < y_values.length;
              value_index++)
            {
            double y_value = y_values[value_index];
            this.y_values.add(y_value);

            System.out.println("   value #" + value_index + ": " + y_value);
/*
*/

            if (value_index == 0)
               {
               this.y_values_max = y_value;
               this.y_values_min = y_value;
               }
            else
               {
               if (this.y_values_max < y_value)
                  {
                  this.y_values_max = y_value;
                  }
               if (this.y_values_min > y_value)
                  {
                  this.y_values_min = y_value;
                  }
               }
            }

         double values_range = this.y_values_max - this.y_values_min;

         System.out.println("   values min = " + this.y_values_min);
         System.out.println("   values max = " + this.y_values_max);
/*

         this.y_values_max = this.y_values_max + 0.05 * values_range;
         this.y_values_min = this.y_values_min - 0.05 * values_range;
*/

         this.y_values_range = this.y_values_max - this.y_values_min;

         if (this.y_values_range == 0.0)
            {
            this.y_values_max += 0.1;
            this.y_values_min -= 0.1;
            this.y_values_range = 0.2;
            }
         this.y_values_scale = 1.0 / this.y_values_range;


         this.cell_height = 3;
         if (this.y_values.size() < 5)
            {
            this.cell_height = 20;
            }

         this.col_height = this.y_values.size() * this.cell_height;

         try
            {
            this.axis_y.updateAxis(this.y_axis_title,
                                   this.y_values_min,
                                   this.y_values_max,
                                   false,
                                   true);
            }
         catch (Exception except)
            {
            System.out.println("ScrolledPlot::setVerticals ERROR:\n"
                               + "   \"" + except + "\"");
            except.printStackTrace(System.out);
            }
         this.plot_height = this.col_height;
         }
      }


   /**
   *  Set the vertical values for the lines to include.  This gets the
   *  min and scale factor for later rescaling of the input values as
   *  needed for various plot sizes.
   *  @param y_values the values for each of the vertical locations.
   */
   public void setVerticals(ArrayList<Double> y_values)
      {
      System.out.println("ScrolledPlot::setVerticals(y_values) on ENTRY:\n"
                       + "   title = " + this.title + "\n"
                       + "   y_values.size() = " + y_values.size() + "\n"
                        );
/*
*/
      if (y_values != null && y_values.size() > 0)
         {
         this.y_values = new ArrayList<Double>();

         System.out.println("ScrolledPlot::setVerticals() y_values:");
/*
*/
         for (int value_index = 0;
              value_index < y_values.size();
              value_index++)
            {
            double y_value = y_values.get(value_index);
            this.y_values.add(y_value);

            System.out.println("   value #" + value_index + ": " + y_value);
/*
*/

            if (value_index == 0)
               {
               this.y_values_max = y_value;
               this.y_values_min = y_value;
               }
            else
               {
               if (this.y_values_max < y_value)
                  {
                  this.y_values_max = y_value;
                  }
               if (this.y_values_min > y_value)
                  {
                  this.y_values_min = y_value;
                  }
               }
            }

         double values_range = this.y_values_max - this.y_values_min;

         this.y_values_max = this.y_values_max + 0.05 * values_range;
         this.y_values_min = this.y_values_min - 0.05 * values_range;

         this.y_values_range = this.y_values_max - this.y_values_min;

         if (this.y_values_range == 0.0)
            {
            this.y_values_max += 0.1;
            this.y_values_min -= 0.1;
            this.y_values_range = 0.2;
            }
         this.y_values_scale = 1.0 / this.y_values_range;


         this.cell_height = 3;
         if (this.y_values.size() < 5)
            {
            this.cell_height = 20;
            }

         this.col_height = this.y_values.size() * this.cell_height;

         try
            {
            this.axis_y.updateAxis(this.y_axis_title,
                                   this.y_values_min,
                                   this.y_values_max,
                                   false,
                                   true);
            }
         catch (Exception except)
            {
            System.out.println("ScrolledPlot::setVerticals ERROR:\n"
                               + "   \"" + except + "\"");
            except.printStackTrace(System.out);
            }
         this.plot_height = this.col_height;
         }
      }

   /**
   *  Set display magnitude range to map into the rainbow colors.
   *  @param mag_min the minimum value expected.
   *  @param mag_max the maximum value expected.
   */
   public void setValuesRange(double mag_min, double mag_max)
      {
      if (mag_min == mag_max)
         {
         this.value_min = 0.98 * mag_min;
         this.value_max = 1.02 * mag_max;
         }
      else if (mag_min < mag_max)
         {
         this.value_min = mag_min;
         this.value_max = mag_max;
         }
      else
         {
         this.value_min = mag_max;
         this.value_max = mag_min;
         }

      this.value_range = this.value_max - this.value_min;

      System.out.println("value(min, max, range) = ("
                       + this.value_min
                       + ", "
                       + this.value_max
                       + ", "
                       + this.value_range
                       + ")"
                        );

      }


   /**
   *  Store another time instance set of plot points (a single vertical line).
   *  @param key_string the key string for the line.
   *  @param x_value the time value for the samples.
   *  @param magnitudes the array of magnitudes, one each for the vertical
   *         values reserved.
   */
   public void storePlotValues(String key_string,
                               double x_value,
                               Double [] magnitudes)
      {
      System.out.println("ScrolledPlot::storePlotValues on ENTRY:\n"
                       + "   key_string = " + key_string + "\n"
                       + "   x_value = " + x_value + "\n"
                       + "   magnitudes.length = "
                       + magnitudes.length
                       + "\n"
                       + "   magnitudes.size() = "
                       + this.magnitudes.size()
                       + "\n"
                        );
/*
*/

      if (this.magnitudes.size() >= this.num_cols_max)
         {
         /*
         *  Drop a column before adding the new one.
         */
         this.x_values.remove(0);
         this.magnitudes.remove(0);

         this.x_values_min += this.x_values_incr;
         this.x_values_max += this.x_values_incr;
         }

      try
         {
         this.axis_x.updateAxis(this.x_axis_title,
                                this.x_values_min,
                                this.x_values_max,
                                false,
                                false);
         }
      catch (Exception except)
         {
         System.out.println("ScrolledPlot::storePlotValues ERROR:\n"
                            + "   \"" + except + "\"");
         except.printStackTrace(System.out);
         }

      /*
      *  Add the new column to the list of columns.
      */

      /*
      *  Copy the magnitudes.
      */
      ArrayList<Double> mags = new ArrayList<Double>();

      for (int mag_index = 0;
           mag_index < this.y_values.size();
           mag_index++)
         {
         Double magnitude = magnitudes[mag_index];
         mags.add(magnitude);
         }

      /*
      *  Append the column to the collection.
      */
      this.x_values.add(x_value);

      this.magnitudes.add(mags);

      System.out.println("ScrolledPlot::storePlotValues on EXIT:\n"
                       + "   magnitudes.size() = "
                       + this.magnitudes.size()
                       + "\n"
                        );
/*
*/
      }


   /**
   *  Get a plot of the column lines.
   */
   public Plot2D getColumnLinesPlot()
      {
      Plot2D column_lines_plot = new Plot2D(title, y_axis_title, "Magnitudes");

      /*
      *  Give it the PlotItem lines, one for each column (across rows).
      */
      ArrayList<Double> mags = this.magnitudes.get(0);
      int num_rows = mags.size();
      int num_cols = this.magnitudes.size();

      for (int col_index = 0;
           col_index < num_cols;
           col_index++)
         {
         double [] plot_x_values = new double[num_rows];
         double [] plot_y_values = new double[num_rows];

         mags = this.magnitudes.get(col_index);

         for (int row_index = 0;
              row_index < num_rows;
              row_index++)
            {
            Double magnitude = mags.get(row_index);

            plot_x_values[row_index] = this.y_values.get(row_index);

            if (magnitude != null)
               {
               plot_y_values[row_index] = magnitude;
               }
            else
               {
               plot_y_values[row_index] = 0.0;
               }
            }

         double col_value = this.x_values.get(col_index);
         String col_title = String.format(x_axis_title + " (%9.3g)", col_value);

         Color line_color = getColorForValue((double) col_index,
                                             0.0,
                                             (double) num_cols,
                                             true);

         column_lines_plot.storePlotValues(col_title,
                                           plot_x_values,
                                           plot_y_values,
                                           line_color);
         }

      return(column_lines_plot);
      }


   /**
   *  Get a plot of the row lines, each entry in a column for that row
   *  becomes a point in that row line.
   */
   public Plot2D getRowLinesPlot()
      {
      Plot2D row_lines_plot = new Plot2D(title, x_axis_title, "Magnitudes");

      /*
      *  Give it the PlotItem lines, one for each row (across columns).
      */
      ArrayList<Double> mags = this.magnitudes.get(0);
      int num_rows = mags.size();
      int num_cols = this.magnitudes.size();

      for (int row_index = 0;
           row_index < num_rows;
           row_index++)
         {
         double [] plot_x_values = new double[num_cols];
         double [] plot_y_values = new double[num_cols];

         for (int col_index = 0;
              col_index < num_cols;
              col_index++)
            {
            mags = this.magnitudes.get(col_index);
            Double magnitude = mags.get(row_index);

            plot_x_values[col_index] = this.x_values.get(col_index);
            if (magnitude != null)
               {
               plot_y_values[col_index] = magnitude;
               }
            else
               {
               plot_y_values[col_index] = 0.0;
               }
            }

         double row_value = this.y_values.get(row_index);
         String row_title = String.format(y_axis_title + " (%9.3g)", row_value);

         Color line_color = getColorForValue((double) row_index,
                                             0.0,
                                             (double) num_rows,
                                             true);

         row_lines_plot.storePlotValues(row_title,
                                        plot_x_values,
                                        plot_y_values,
                                        line_color);
         }

      return(row_lines_plot);
      }


   /**
   *  Generate a color to use given the current value.
   *  @param value the value to convert to a color.
   *  @return the color for that value.
   */
   public static Color getColorForValue(double value,
                                        double min_val,
                                        double max_val,
                                        boolean colors_selected)
      {
      Color color = Color.BLACK;

/*
      System.out.println("Getting the color for a magnitude ON ENTRY:\n"
                       + "   value = " + value + "\n"
                       + "   min_val = " + min_val + "\n"
                       + "   max_val = " + max_val + "\n"
                        );
*/

      if (value < min_val)
         {
         /*
         *  The default is BLACK.
         */
         }
      else if (value > max_val)
         {
         color = Color.WHITE;
         }
      else
         {
         float [] f_color = new float[3];

         double range_val = max_val - min_val;

         if (colors_selected)
            {
            float [] low_f_color = new float[3];
            float [] high_f_color = new float[3];

            double indexing_offset = 7.0 * (value - min_val) / range_val;

            int low_index = (int) Math.floor(indexing_offset);
            int high_index = (int) Math.ceil(indexing_offset);

            double fraction = indexing_offset - (double) low_index;

            float low_w = (float) (fraction);
            float high_w = (float) (1.0 - fraction);

            System.out.println("value = " + value
                             + ", " + low_index
                             + ", " + high_index);

            low_f_color[0] = rb_rgb_values[low_index][0];
            low_f_color[1] = rb_rgb_values[low_index][1];
            low_f_color[2] = rb_rgb_values[low_index][2];

            high_f_color[0] = rb_rgb_values[high_index][0];
            high_f_color[1] = rb_rgb_values[high_index][1];
            high_f_color[2] = rb_rgb_values[high_index][2];

            f_color[0] = low_f_color[0] * low_w + high_f_color[0] * high_w;
            f_color[1] = low_f_color[1] * low_w + high_f_color[1] * high_w;
            f_color[2] = low_f_color[2] * low_w + high_f_color[2] * high_w;
            }
         else
            {
            float grey = (float) ((value - min_val) / range_val);
/*
            System.out.println("value = " + value + ", " + grey);
*/

            f_color[0] = grey;
            f_color[1] = grey;
            f_color[2] = grey;
            }
         color = new Color(f_color[0], f_color[1], f_color[2]);
         }

/*
      System.out.println("Getting the color for a magnitude ON EXIT:\n"
                       + "   color = " + color + "\n"
                        );
*/
      return(color);
      }
   }

