
package util;

import java.util.*;
import java.io.*;

import util.geom.*;

/**
*  This class defines an interval in a sequence of data values.  If the
*  coordinates are
*     -- 1-dimensional the base space is a sequence of indexed values.
*     -- 2-dimensional it is a line of interpolated locations and their
*        associated values in a 2-d array space.
*     -- 3-dimensional it is a line of interpolated locations and their
*        associated values in a 3-d array space.
*  The line is characterized by the base point and the unit direction vector.
*  An interval is defined only by its endpoints, which must lie along the line.
*/
public class F_Interval extends Feature
   {

   /**
   * The (optional) line along which the interval lies.  This is not essential
   * given the two end points.
   */
   F_Sequence base_line = null;

   /**
   *  The start point for the interval.
   */
   F_Point start_point = null;

   /**
   * The end point for the interval.
   */
   F_Point end_point = null;

   /**
   * The direction vector for this interval (1, 2, 0r 3 dimensions).
   */
   VectorND direction = null;

   /**
   *  Constructor for a sequence line with a base point on a base line, with
   *  this line perpendicular to the base line in 2-D.
   *  @param start_point the starting end location of the sequence.
   *  @param end_point the end location of the interval.
   *  @param base_line the base line for the sequence if 2-dimensional.
   */
   public F_Interval(F_Point start_point,
                     F_Point end_point,
                     F_Sequence base_line)
      {
      super("Interval", start_point.getNumDims());

      this.start_point = start_point;
      this.end_point = end_point;
      this.base_line = base_line;

      /*
      * Get the normalized direction vector along this interval.
      */
      this.direction = this.start_point.getDirectionTo(this.end_point);
      }

   /**
   *  Get the base point for this line.
   *  @return the base point reference.
   */
   public F_Point getStartPoint()
      {
      return(this.start_point);
      }

   /**
   *  Get the other point for this line.
   *  @return the other point reference.
   */
   public F_Point getEndPoint()
      {
      return(this.end_point);
      }

   /**
   *  Get the base line for this line.
   *  @return the base line reference.
   */
   public F_Sequence getBaseLine()
      {
      return(this.base_line);
      }

   /**
   *  Get the direction for the line on which this interval exists.
   *  @return the direction vector reference.
   */
   public VectorND getDirection()
      {
      return(this.start_point.getDirectionTo(this.end_point));
      }

   /**
   *  Get the location of the center of the interval.
   *  @return the direction vector reference.
   */
   public VectorND getCenter()
      {
      return(this.start_point.location.
                              add(this.end_point.location).
                                 scale(0.5));
      }

   /**
   *  Get the next location along this interval.
   *  @return the next point's coordinates.
   */
   public VectorND getNextPosition(VectorND curr_pos,
                                   Double step)
      {
      return(this.direction.getScaled(step).add(curr_pos));
      }

   /**
   *  Get the previous location along this interval.
   *  @return the previous point's coordinates.
   */
   public VectorND getPrevPosition(VectorND curr_pos,
                                   Double step)
      {
      return(this.direction.getScaled(-step).add(curr_pos));
      }

  /**
  *  Get the step size scale factor for single integer increments
  *  along this line.  The largest axis will be incremented to the next
  *  integer value by this.
  *  @return the step size scale factor.
  */
   public Double getStepScaleFactor()
      {
      Double scale_factor = null;
      double max_axis = this.direction.getMagnitudeLI();
      if (max_axis != 0.0)
         {
         scale_factor = 1.0 / max_axis;
         }
      return(scale_factor);
      }

   /**
   *  Generate a string representation.
   *  @return the string representation.
   */
   public String toString()
      {
      String out_string = super.toString();

      this.start_point.setPrefix(this.prefix + "      ");
      out_string += this.prefix
                  + "   Start Point:\n"
                  + this.start_point;
      out_string += "\n";
      this.start_point.setPrefix("");

      this.end_point.setPrefix(this.prefix + "      ");
      out_string += this.prefix
                  + "   End Point:\n"
                  + this.end_point;
      out_string += "\n";
      this.end_point.setPrefix("");

      out_string += this.prefix
                  + "   Direction: " + this.direction;
      out_string += "\n";

      if (this.base_line != null)
         {
         this.base_line.setPrefix(this.prefix + "      ");
         out_string += this.prefix
                     +"   Base Line:\n"
                  + this.base_line;
         out_string += "\n";
         this.base_line.setPrefix("");
         }
      return(out_string);
      }

   /**
   *  Print the string representation.
   */
   public void Print()
      {
      System.out.print(this);
      }

   /*
   * The functions for determining the feature type efficiently.
   */

   /*
   * Is it a Point?
   * @returns true if it is.
   */
   public boolean isPoint()
      {
      return(false);
      }

   /*
   * Is it a Sequence?
   * @returns true if it is.
   */
   public boolean isSequence()
      {
      return(false);
      }

   /*
   * Is it a Interval?
   * @returns true if it is.
   */
   public boolean isInterval()
      {
      return(true);
      }

   /*
   * Is it a Region?
   * @returns true if it is.
   */
   public boolean isRegion()
      {
      return(false);
      }
   }

