
package util;

import java.util.*;
import java.io.*;

import util.geom.*;

/**
*  This class defines a sequence of data values.  If the CoordType is
*     -- 1-dimensional it is a sequence of indexed values.
*     -- 2-dimensional it is a line of interpolated locations and their
*        associated values in a 2-d array space.
*     -- 3-dimensional it is a line of interpolated locations and their
*        associated values in a 3-d array space.
*  The line is characterized by the base point and the unit increment
*  coefficients:
*
*  In 1 dimension:
*     +/- 1
*
*  In 2 dimensions:
*     x = a t
*     y = b t
*  giving (a, b) where |(a, b)| == 1.0
*
*  In 3 dimensions:
*     x = a t
*     y = b t
*     z = c t
*  giving (a, b, c) where |(a, b, c)| == 1.0
*
*/
public class F_Sequence extends Feature
   {

   /**
   *  The base point for the sequence.
   */
   F_Point base_point = null;

   /**
   * The (optional) second point if it is a sequence defined by two points.
   * If this is null then the sequence is defined by the base point and
   * the direction vector below.
   */
   F_Point other_point = null;

   /**
   * The (optional) line to which it is perpendicular at the base point.
   */
   F_Sequence base_line = null;

   /**
   * The direction vector for this line (1, 2, 0r 3 dimensions).
   */
   VectorND direction = null;


   /**
   *  Constructor for a sequence with a start and stop location point.
   *  @param base_point the starting end location of the sequence.
   *  @param other_point the other end location of the sequence.
   */
   public F_Sequence(F_Point base_point,
                     F_Point other_point)
      {
      super("Sequence", base_point.getNumDims());

      this.base_point = base_point;
      this.other_point = other_point;

      this.direction = base_point.getDirectionTo(other_point);
      }

  /**
  *  Constructor for a sequence line with a base point on a base line, with
  *  this line perpendicular to the base line in 2-D.
  *  @param base_point the starting end location of the sequence.
  *  @param base_line the base line for the sequence if 2-dimensional.
  *  @throws Exception
  */
   public F_Sequence(F_Point base_point,
                     F_Sequence base_line)
      throws Exception
      {
      super("Sequence", base_point.getNumDims());

      this.base_point = base_point;
      this.base_line = base_line;
      if (num_dims == 2)
         {
         this.direction = this.getPerpendicularTo(base_line);
         }
      else
         {
         System.out.println("F_Sequence(point, line) ERROR:\n"
                          + "   The location is not 2 dimensional.\n");
         }
      }

  /**
  *  Get the base point for this line.
  *  @return the base point reference.
  */
   public VectorND getPerpendicularTo(F_Sequence base_line)
      {
      VectorND direction = null;

      double x_value = base_line.direction.values[0];
      double y_value = base_line.direction.values[1];
      if (x_value == 0.0)
         {
         direction = new VectorND(1.0, 0.0);
         }
      else if (y_value == 0.0)
         {
         direction = new VectorND(0.0, 1.0);
         }
      else
         {
         direction = new VectorND(-y_value / x_value, 1.0);
         direction.normalizeL2();
         }

      return(direction);
      }

  /**
  *  Get the base point for this line.
  *  @return the base point reference.
  */
   public F_Point getBasePoint()
      {
      return(this.base_point);
      }

  /**
  *  Get the other point for this line.
  *  @return the other point reference.
  */
   public F_Point getOtherPoint()
      {
      return(this.other_point);
      }

  /**
  *  Get the base line for this line.
  *  @return the base line reference.
  */
   public F_Sequence getBaseLine()
      {
      return(this.base_line);
      }

  /**
  *  Get the direction for this line.
  *  @return the direction vector reference.
  */
   public VectorND getDirection()
      {
      return(this.direction);
      }

   /**
   *  Get the next location along this interval.
   *  @return the next point's coordinates.
   */
   public VectorND getNextPosition(VectorND curr_pos,
                                   Double step)
      {
      return(this.direction.getScaled(step).add(curr_pos));
      }

   /**
   *  Get the previous location along this interval.
   *  @return the previous point's coordinates.
   */
   public VectorND getPrevPosition(VectorND curr_pos,
                                   Double step)
      {
      return(this.direction.getScaled(-step).add(curr_pos));
      }

  /**
  *  Get the step size scale factor for single integer increments
  *  along this line.  The largest axis will be incremented to the next
  *  integer value by this.
  *  @return the step size scale factor.
  */
   public Double getStepScaleFactor()
      {
      Double scale_factor = null;
      double max_axis = this.direction.getMagnitudeLI();
      if (max_axis != 0.0)
         {
         scale_factor = 1.0 / max_axis;
         }
      return(scale_factor);
      }

  /**
  *  Generate a string representation.
  *  @return the string representation.
  */
   public String toString()
      {
      String out_string = super.toString();

      this.base_point.setPrefix(this.prefix + "      ");
      out_string += this.prefix
                  + "   Base Point:\n"
                  + this.base_point;
      out_string += "\n";
      this.base_point.setPrefix("");

      if (this.other_point != null)
         {
         this.other_point.setPrefix(this.prefix + "      ");
         out_string += this.prefix
                     + "   Other Point:\n"
                     + this.other_point;
         out_string += "\n";
         this.other_point.setPrefix("");
         }

      if (this.base_line != null)
         {
         this.base_line.setPrefix(this.prefix + "      ");
         out_string += this.prefix
                     + "   Base Line:\n"
                     + this.base_line;
         out_string += "\n";
         this.base_line.setPrefix("");
         }

      out_string += this.prefix
                  + "   Direction: " + this.direction;
      out_string += "\n";
      return(out_string);
      }

  /**
  *  Print the string representation.
  */
   public void Print()
      {
      System.out.print(this);
      }

   /*
   * The functions for determining the feature type efficiently.
   */

   /*
   * Is it a Point?
   * @returns true if it is.
   */
   public boolean isPoint()
      {
      return(false);
      }

   /*
   * Is it a Sequence?
   * @returns true if it is.
   */
   public boolean isSequence()
      {
      return(true);
      }

   /*
   * Is it a Interval?
   * @returns true if it is.
   */
   public boolean isInterval()
      {
      return(false);
      }

   /*
   * Is it a Region?
   * @returns true if it is.
   */
   public boolean isRegion()
      {
      return(false);
      }
   }
