
/*
*  The Histogram color histogram.
*/

package util;


import java.util.*;

import java.awt.*;


/**
*  This class defines the basic colormap structure used by the various
*  colormap control and colormap using classes in this library.
*/
public class Histogram
   {

   int num_values = 0;


   int bins[] = null;

   int num_bins = 0;
   double max_percent = 1.0;
   double max_fraction = 1.0;
   int max_count = 0;
   int count_range = 0;

   double value_mean = 0;
   double value_var = 0;
   double value_std = 0;
   int value_max = -10000;
   int value_min = 10000;
   int value_range = 0;

   /*
   *  The range of values to place in the bins.
   */
   int bin_max = -10000;
   int bin_min = 10000;
   int bin_range = 0;

   double bin_size = 0.0;
   double bin_scale = 1.0;

   /**
   *  Create a histogram based on the given values, with the
   *  given number of bins.
   *  @param values the array of integer values.
   *  @param d_num_bins the number of bins to include.
   */
   public Histogram(int values[], int d_num_bins)
      {
      num_values = values.length;

      num_bins = d_num_bins;
      /*
      *  Zero the histogram bin counts.
      */
      bins = new int[num_bins];
      for (int bin_index = 0;
           bin_index < num_bins;
           bin_index++)
         {
         bins[bin_index] = 0;
         }

      /*
      *  Get the statistics on the values to be used.
      */
      double sum = 0.0;
      double sumsq = 0.0;
      for (int value_index = 0;
           value_index < num_values;
           value_index++)
         {
         int value = values[value_index];
         if (value_index == 0)
            {
            value_max = value;
            value_min = value;
            }
         else if (value_max < value)
            {
            value_max = value;
            }
         else if (value_min > value)
            {
            value_min = value;
            }
         sum += value;
         sumsq += value * value;
         }

      value_mean = sum / num_values;
      value_var = sumsq / num_values - value_mean * value_mean;
      value_std = Math.sqrt(value_var);

      value_range = value_max - value_min;

      bin_min = value_min;
      bin_max = value_max;
      bin_range = value_range;


      bin_size = (double) value_range / (double) num_bins;
      bin_scale = (double) num_bins / (double) value_range;

      for (int value_index = 0;
           value_index < num_values;
           value_index++)
         {
         double value = bin_scale * (values[value_index] - value_min);
         int hist_index = (int) Math.floor(value);
         if (hist_index < 0)
            {
            hist_index = 0;
            }
         else if (hist_index >= num_bins)
            {
            hist_index = num_bins - 1;
            }
         bins[hist_index]++;
         int count = bins[hist_index];
         if (max_count < count)
            {
            max_count = count;
            }
         }

      if (num_values < 1)
         {
         max_fraction = 1.0;
         }
      else
         {
         max_fraction = (float) max_count / (float) num_values;
         }

      max_percent = 100.0 * max_fraction;

      System.out.println("Histogram::constructor #1 -- Statistics:\n"
                       + "   num_values = " + num_values + "\n"
                       + "   num_bins = " + num_bins + "\n"
                       + "   value_mean = " + value_mean + "\n"
                       + "   value_var = " + value_var + "\n"
                       + "   value_std = " + value_std + "\n"
                       + "   value_max = " + value_max + "\n"
                       + "   value_min = " + value_min + "\n"
                       + "   value_range = " + value_range + "\n"
                       + "   bin_max = " + bin_max + "\n"
                       + "   bin_min = " + bin_min + "\n"
                       + "   bin_range = " + bin_range + "\n"
                       + "   bin_size = " + bin_size + "\n"
                       + "   bin_scale = " + bin_scale + "\n"
                       + "   max_count = " + max_count + "\n"
                       + "   max_percent = " + max_percent + "\n"
                       + "   max_fraction = " + max_fraction + "\n"
                        );
/*
*/
   

/*
      System.out.println("Histogram values:");
      for (int bin_index = 0;
           bin_index < num_bins;
           bin_index++)
         {
         System.out.println("   " + bin_index + ": " + bins[bin_index]);
         }
*/
      }

   /**
   *  Create a histogram based on the given values, with the
   *  given number of bins, using the given min and max.  All values
   *  lower than the min will be counted in the first bin, and all
   *  higher than the max will be counted in the last bin.
   *  @param values the array of integer values.
   *  @param d_num_bins the number of bins to include.
   *  @param min the minimum value to include in the histogram binning
   *  @param max the maximum value to include in the histogram binning
   */
   public Histogram(int values[],
                    int d_num_bins,
                    int min,
                    int max)
      {
      num_values = values.length;

      num_bins = d_num_bins;
      /*
      *  Zero the histogram bin counts.
      */
      bins = new int[num_bins];
      for (int bin_index = 0;
           bin_index < num_bins;
           bin_index++)
         {
         bins[bin_index] = 0;
         }

      /*
      *  Get the statistics on the values to be used.
      */
      double sum = 0.0;
      double sumsq = 0.0;
      for (int value_index = 0;
           value_index < num_values;
           value_index++)
         {
         int value = values[value_index];
         if (value_index == 0)
            {
            value_max = value;
            value_min = value;
            }
         else if (value_max < value)
            {
            value_max = value;
            }
         else if (value_min > value)
            {
            value_min = value;
            }
         sum += value;
         sumsq += value * value;
         }

      value_mean = sum / num_values;
      value_var = sumsq / num_values - value_mean * value_mean;
      value_std = Math.sqrt(value_var);

      value_range = value_max - value_min;

      bin_min = min;
      bin_max = max;
      bin_range = bin_max - bin_min;


      bin_size = (double) bin_range / (double) num_bins;
      bin_scale = (double) num_bins / (double) bin_range;

      for (int value_index = 0;
           value_index < num_values;
           value_index++)
         {
         double value = bin_scale * (values[value_index] - bin_min);
         int hist_index = (int) Math.floor(value);
         if (hist_index < 0)
            {
            hist_index = 0;
            }
         else if (hist_index >= num_bins)
            {
            hist_index = num_bins - 1;
            }
         bins[hist_index]++;
         int count = bins[hist_index];
         if (max_count < count)
            {
            max_count = count;
            }
         }

      if (num_values < 1)
         {
         max_fraction = 1.0;
         }
      else
         {
         max_fraction = (float) max_count / (float) num_values;
         }

      max_percent = 100.0 * max_fraction;

      System.out.println("Histogram::constructor #2 -- Statistics:\n"
                       + "   num_values = " + num_values + "\n"
                       + "   num_bins = " + num_bins + "\n"
                       + "   value_mean = " + value_mean + "\n"
                       + "   value_var = " + value_var + "\n"
                       + "   value_std = " + value_std + "\n"
                       + "   value_max = " + value_max + "\n"
                       + "   value_min = " + value_min + "\n"
                       + "   value_range = " + value_range + "\n"
                       + "   bin_max = " + bin_max + "\n"
                       + "   bin_min = " + bin_min + "\n"
                       + "   bin_range = " + bin_range + "\n"
                       + "   bin_size = " + bin_size + "\n"
                       + "   bin_scale = " + bin_scale + "\n"
                       + "   max_count = " + max_count + "\n"
                       + "   max_percent = " + max_percent + "\n"
                       + "   max_fraction = " + max_fraction + "\n"
                        );
/*
*/
   

/*
      System.out.println("Histogram values:");
      for (int bin_index = 0;
           bin_index < num_bins;
           bin_index++)
         {
         System.out.println("   " + bin_index + ": " + bins[bin_index]);
         }
*/
      }

   /**
   *  Get the histogram of counts for the given number of
   *  equally spaced bins.
   *  @return an array of "count" values.
   */
   public int[] getHistogram()
      {
      return(bins);
      }

   /**
   *  Get the normalized histogram (all in [0.0, 1.0]) for the given
   *  number of equally spaced bins.  They are normalized by the max count
   *  in any one bin.
   *  @return an array of normalized "fraction" (in [0, 1.0]) values.
   */
   public double[] getNormalizedHistogram()
      {
      double d_hist[] = new double[num_bins];

      double d_max_count = (double) max_count;
      for (int bin_index = 0;
           bin_index < num_bins;
           bin_index++)
         {
         d_hist[bin_index] = (double) bins[bin_index] / d_max_count;
         }
      return(d_hist);
      }

   /**
   *  Get the number of values stored.
   *  @return the count.
   */
   public int get_num_values()
      {
      return(this.num_values);
      }

   /**
   *  Get the minimum of the values stored.
   *  @return the minimum value.
   */
   public int get_value_min()
      {
      return(this.value_min);
      }

   /**
   *  Get the maximum of the values stored.
   *  @return the maximum value.
   */
   public int get_value_max()
      {
      return(this.value_max);
      }

   /**
   *  Get the mean of the values stored.
   *  @return the mean.
   */
   public double get_value_mean()
      {
      return(this.value_mean);
      }

   /**
   *  Get the standard deviation of the values stored.
   *  @return the standard deviation.
   */
   public double get_value_std()
      {
      return(this.value_std);
      }

   /**
   *  Get the number of bins.
   *  @return the standard deviation.
   */
   public int get_num_bins()
      {
      return(this.num_bins);
      }

   /**
   *  Get the range of the bins.
   *  @return the range.
   */
   public int get_bin_range()
      {
      return(this.bin_range);
      }

   /**
   *  Get the min index for the bins.  They are indexed from 0 to this index value.
   *  @return the bin maximum index.
   */
   public int get_bin_min()
      {
      return(this.bin_min);
      }

   /**
   *  Get the max index for the bins.  They are indexed from 0 to this index value.
   *  @return the bin maximum index.
   */
   public int get_bin_max()
      {
      return(this.bin_max);
      }

   /**
   *  Get the maximum count stored in any of the bins.
   *  @return the count.
   */
   public int get_max_count()
      {
      return(this.max_count);
      }
   }

