
package util.geom;

import java.util.*;
import java.io.*;

import util.*;

/**
*  This class allows storing and manipulating 1-D interval components.
*/
public class BoundedError1D extends Vector2D
   {

   public final String default_format = "[%9.3f, %9.3f]";
   public String format = default_format;

   /**
   *  Creates a BoundedError1D object with default values of
   *  [0.0, 1.0].
   */
   public BoundedError1D()
      {
      super(0.0, 1.0);
      }

   /**
   *  Creates a BoundedError1D object with (x, y) component values.
   *  @param x the x-component.
   *  @param y the y-component.
   */
   public BoundedError1D(double x, double y)
      {
      super(x, y);
      }


   /**
   *  Creates a BoundedError1D object from a Vector2D argument's
   *  component values.
   *  @param vector the vector.
   */
   public BoundedError1D(Vector2D vector)
      {
      super(vector);
      }

   /**
   * Creates a new BoundedError1D object from the BoundedError1D's component
   * values.
   * @return the new BoundedError1D object reference.
   */
   public BoundedError1D clone()
      {
      BoundedError1D interval = new BoundedError1D(this);
      return(interval);
      }


   /**
   *  Returns the width (range) of the interval's values.
   *  @return the width of the interval.
   */
   public double range()
      {
      double range = this.y - this.x;
      
      return(range);
      }


   /**
   *  Returns the scaled bounded error.
   *  @param scale the scale multiplier.
   *  @return a scaled interval.
   */
   public BoundedError1D getScaled(double scale)
      {
      double x_val = scale * this.x;
      double y_val = scale * this.y;

      BoundedError1D interval = new BoundedError1D(x_val, y_val);
      
      return(interval);
      }


   /**
   *  Returns the sum of the two BoundedError1D vectors.
   *  @param vector1 the first interval
   *  @param vector2 the second interval
   *  @return the sum BoundedError1D.
   */
   public static BoundedError1D sum(Vector2D vector1, Vector2D vector2)
      {
      double sum_x = vector1.x + vector2.x;
      double sum_y = vector1.y + vector2.y;

      BoundedError1D interval = new BoundedError1D(sum_x, sum_y);
      
      return(interval);
      }


   /**
   *  Returns the sum of the two component vectors.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @return the sum BoundedError1D.
   */
   public static BoundedError1D sum(double x1, double y1, double x2, double y2)
      {
      BoundedError1D interval = new BoundedError1D(x1 + x2, y1 + y2);
      
      return(interval);
      }

   /**
   *  Returns the sum of the two BoundedError1D vectors.
   *  @param vector the other vector.
   *  @return the sum BoundedError1D.
   */
   public BoundedError1D add(Vector2D vector)
      {
      BoundedError1D interval = new BoundedError1D(this.x + vector.x,
                                                   this.y + vector.y);
      
      return(interval);
      }

   /**
   *  Returns the sum of the two component vectors.
   *  @param x the interval's X component
   *  @param y the interval's Y component
   *  @return the sum BoundedError1D.
   */
   public BoundedError1D add(double x, double y)
      {
      BoundedError1D interval = new BoundedError1D(this.x + x, this.y + y);
      
      return(interval);
      }


   /**
   * Calculates the difference of the bounded intervals, creating a new bounded
   * interval.
   * @return the bounded interval for the difference.
   */
   public BoundedError1D subtract(Vector2D vector)
      {
      double value1 = this.x - vector.x;
      double value2 = this.y - vector.y;
      BoundedError1D interval = new BoundedError1D(Math.min(value1, value2),
                                                   Math.max(value1, value2));
      return(interval);
      }

   /**
   *  Returns the difference between the two bounded error intervals.
   *  @param x the interval's X component
   *  @param y the interval's Y component
   *  @return the difference BoundedError1D.
   */
   public BoundedError1D subtract(double x, double y)
      {
      double value1 = this.x - x;
      double value2 = this.y - y;
      BoundedError1D interval = new BoundedError1D(Math.min(value1, value2),
                                                   Math.max(value1, value2));
      return(interval);
      }


   /**
   *  Returns the difference between the two bounded error intervals.
   *  @param vector1 the first interval as a vector
   *  @param vector2 the second interval as a vector
   *  @return the new bounded error interval.
   */
   public static BoundedError1D diff(Vector2D vector1, Vector2D vector2)
      {
      double value1 = vector1.x - vector2.x;
      double value2 = vector1.y - vector2.y;
      BoundedError1D interval = new BoundedError1D(Math.min(value1, value2),
                                                   Math.max(value1, value2));
      return(interval);
      }

   /**
   *  Returns the difference between the two bounded error intervals.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @return the difference BoundedError1D.
   */
   public static BoundedError1D diff(double x1,
                                     double y1,
                                     double x2,
                                     double y2)
      {
      double value1 = x1 - x2;
      double value2 = y1 - y2;
      BoundedError1D interval = new BoundedError1D(Math.min(value1, value2),
                                                   Math.max(value1, value2));
      return(interval);
      }


   /**
   * Calculates the product of the bounded intervals, creating a new bounded
   * interval.
   * @return the bounded interval for the product.
   */
   public BoundedError1D product(Vector2D vector)
      {
      BoundedError1D interval = new BoundedError1D();

      double value1 = this.x * vector.x;
      double value2 = this.y * vector.y;
      double value3 = this.x * vector.y;
      double value4 = this.y * vector.x;

      double min1 = Math.min(value1, value2);
      double min2 = Math.min(value3, value4);
      interval.x = Math.min(min1, min2);

      double max1 = Math.max(value1, value2);
      double max2 = Math.max(value3, value4);
      interval.y = Math.max(max1, max2);

      return(interval);
      }

   /**
   * Calculates the result of dividing the first variable by the second
   * variable where each is described by bounded intervals, creating a
   * new bounded interval.  Division by zero can be within the interval,
   * but will put + or - inifinity as one of the bounds. If the first
   * interval includes 0.0 then both bounds are NaN.
   * @return the bounded interval for the division.
   */
   public BoundedError1D divide(Vector2D vector)
      {
      BoundedError1D interval = new BoundedError1D();

      if (vector.x <= 0.0 && vector.y >= 0.0)
         {
         /*
         * Zero is in the interval for the denominator, so the default
         * is applicable unless 0.0 is included in the numerator.
         */
         if (this.x <= 0.0 && this.y >= 0.0)
            {
            interval.x = Double.NaN;
            interval.y = Double.NaN;
            }
         else
            {
            interval.x = Double.NEGATIVE_INFINITY;
            interval.y = Double.POSITIVE_INFINITY;
            }
         }
      else
         {
         double value1 = this.x / vector.x;
         double value2 = this.y / vector.y;
         double value3 = this.x / vector.y;
         double value4 = this.y / vector.x;

         double min1 = Math.min(value1, value2);
         double min2 = Math.min(value3, value4);
         interval.x = Math.min(min1, min2);

         double max1 = Math.max(value1, value2);
         double max2 = Math.max(value3, value4);
         interval.y = Math.max(max1, max2);
         }

      return(interval);
      }


   /**
   * Calculates the intersection of the bounded intervals, creating a
   * new bounded interval.
   * @return the bounded interval for the intersection.
   */
   public BoundedError1D Intersection(Vector2D vector)
      {
      BoundedError1D interval = null;

      double value1 = Math.max(this.x, vector.x);
      double value2 = Math.min(this.y, vector.y);
      if (value1 <= value2)
         {
         interval = new BoundedError1D(value1, value2);
         }
      else
         {
         interval = new BoundedError1D(Double.NaN, Double.NaN);
         }
      return(interval);
      }

   /**
   * Calculates the intersection of the bounded intervals, creating a
   * new bounded interval.
   * @param x the interval's X component
   * @param y the interval's Y component
   * @return the bounded interval for the intersection.
   */
   public BoundedError1D Intersection(double x, double y)
      {
      BoundedError1D interval = null;

      double value1 = Math.max(this.x, x);
      double value2 = Math.min(this.y, y);
      if (value1 <= value2)
         {
         interval = new BoundedError1D(value1, value2);
         }
      else
         {
         interval = new BoundedError1D(Double.NaN, Double.NaN);
         }
      return(interval);
      }


   /**
   *  Returns the intersection between the two bounded error intervals.
   *  @param vector1 the first interval as a vector
   *  @param vector2 the second interval as a vector
   *  @return the new bounded error interval.
   */
   public static BoundedError1D Intersection(Vector2D vector1,
                                             Vector2D vector2)
      {
      BoundedError1D interval = null;

      double value1 = Math.max(vector1.x, vector2.x);
      double value2 = Math.min(vector1.y, vector2.y);
      if (value1 <= value2)
         {
         interval = new BoundedError1D(value1, value2);
         }
      else
         {
         interval = new BoundedError1D(Double.NaN, Double.NaN);
         }
      return(interval);
      }

   /**
   *  Returns the intersection between the two bounded error intervals.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @return the intersection BoundedError1D.
   */
   public static BoundedError1D Intersection(double x1,
                                             double y1,
                                             double x2,
                                             double y2)
      {
      BoundedError1D interval = null;

      double value1 = Math.max(x1, x2);
      double value2 = Math.min(y1, y2);
      if (value1 <= value2)
         {
         interval = new BoundedError1D(value1, value2);
         }
      else
         {
         interval = new BoundedError1D(Double.NaN, Double.NaN);
         }
      return(interval);
      }


   /**
   * Calculates the union of the bounded intervals, creating a
   * new bounded interval.  If the two are disjoint then the interval
   * between them is included in the union.
   * @return the bounded interval for the union.
   */
   public BoundedError1D Union(Vector2D vector)
      {
      double value1 = Math.min(this.x, vector.x);
      double value2 = Math.max(this.y, vector.y);
      BoundedError1D interval = new BoundedError1D(value1, value2);
      return(interval);
      }

   /**
   * Calculates the union of the bounded intervals, creating a
   * new bounded interval.  If the two are disjoint then the interval
   * between them is included in the union.
   * @param x the interval's X component
   * @param y the interval's Y component
   * @return the bounded interval for the union.
   */
   public BoundedError1D Union(double x, double y)
      {
      double value1 = Math.min(this.x, vector.x);
      double value2 = Math.max(this.y, vector.y);
      BoundedError1D interval = new BoundedError1D(value1, value2);
      return(interval);
      }


   /**
   *  Returns the union between the two bounded error intervals.
   *  If the two are disjoint then the interval
   *  between them is included in the union.
   *  @param vector1 the first interval as a vector
   *  @param vector2 the second interval as a vector
   *  @return the new bounded error interval.
   */
   public static BoundedError1D Union(Vector2D vector1,
                                             Vector2D vector2)
      {
      double value1 = Math.min(this.x, vector.x);
      double value2 = Math.max(this.y, vector.y);
      BoundedError1D interval = new BoundedError1D(value1, value2);
      return(interval);
      }

   /**
   *  Returns the union between the two bounded error intervals.
   *  If the two are disjoint then the interval
   *  between them is included in the union.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @return the union BoundedError1D.
   */
   public static BoundedError1D Union(double x1,
                                      double y1,
                                      double x2,
                                      double y2)
      {
      double value1 = Math.min(this.x, vector.x);
      double value2 = Math.max(this.y, vector.y);
      BoundedError1D interval = new BoundedError1D(value1, value2);
      return(interval);
      }

   /**
   * Returns a String that represents the value interval of this BoundedError1D.
   * @return the string representation.
   */
   public String toString()
      {
      String out_string = String.format(this.format, this.x, this.y);
      return(out_string);
      }

   /**
   * Prints the representation the value interval.
   */
   public void Print()
      {
      System.out.print(this);
      }
   }

