
package util.geom;

import java.util.*;
import java.io.*;

import util.*;

/**
*  This class allows storing 2-D complex number components.
*/
public class Complex extends Vector2D
   {

   public final String default_format = "(%9.3f %+9.3f i)";
   public String format = default_format;

   /**
   *  Creates a Complex object from the given magnitude and angle.
   *  @param mag the magnitude to use.
   *  @param angle the angle (in radians) to use.
   */
   public static Complex getFromMagAngle(double mag, double angle)
      {
      Complex complex_num = new Complex(mag * Math.cos(angle),
                                        mag * Math.sin(angle));
      return(complex_num);
      }

   /**
   *  Creates a Complex object with default values of 0.0.
   */
   public Complex()
      {
      super();
      }

   /**
   *  Creates a Complex object with (x, y) component values.
   *  @param x the x-component.
   *  @param y the y-component.
   */
   public Complex(double x, double y)
      {
      super(x, y);
      }

   /**
   *  Creates a Complex object from a Complex argument's component values.
   *  @param vector the other complex number as a vector.
   */
   public Complex(Vector2D vector)
      {
      super(vector);
      }

   /**
   * Creates a new Complex object from the Complex's component values.
   * @return the new Complex object reference.
   */
   public Complex clone()
      {
      Complex complex_num = new Complex(this);
      return(complex_num);
      }


   /**
   *  Returns the scaled complex number.
   *  @param scale the scale multiplier.
   *  @return a scaled complex number.
   */
   public Complex getScaled(double scale)
      {
      double x_val = scale * this.x;
      double y_val = scale * this.y;

      Complex complex_num = new Complex(x_val, y_val);
      
      return(complex_num);
      }


   /**
   *  Returns the sum of the two Complex vectors.
   *  @param vector1 the first complex number as a vector.
   *  @param vector2 the second complex number as a vector.
   *  @return the sum Complex.
   */
   public static Complex sum(Vector2D vector1, Vector2D vector2)
      {
      double sum_x = vector1.x + vector2.x;
      double sum_y = vector1.y + vector2.y;

      Complex diff_complex_num = new Complex(sum_x, sum_y);
      
      return(diff_complex_num);
      }


   /**
   *  Returns the sum of the two component vectors.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @return the sum Complex.
   */
   public static Complex sum(double x1, double y1, double x2, double y2)
      {
      Complex sum_complex_num = new Complex(x1 + x2, y1 + y2);
      
      return(sum_complex_num);
      }

   /**
   *  Returns the sum of the two Complex vectors.
   *  @param vector the other complex number as a vector.
   *  @return the sum Complex.
   */
   public Complex plus(Vector2D vector)
      {
      double sum_x = this.x + vector.x;
      double sum_y = this.y + vector.y;

      Complex sum_complex_num = new Complex(sum_x, sum_y);
      
      return(sum_complex_num);
      }

   /**
   *  Returns the sum of the two component vectors.
   *  @param x the complex number's X component
   *  @param y the complex number's Y component
   *  @return the sum Complex.
   */
   public Complex plus(double x, double y)
      {
      Complex sum_complex_num = new Complex(this.x + x, this.y + y);
      
      return(sum_complex_num);
      }


   /**
   *  Returns the difference between the two Complex vectors.
   *  @param vector1 the first complex number as a vector.
   *  @param vector2 the second complex number as a vector.
   */
   public static Complex diff(Vector2D vector1, Vector2D vector2)
      {
      double diff_x = vector1.x - vector2.x;
      double diff_y = vector1.y - vector2.y;

      Complex complex_num = new Complex(diff_x, diff_y);
      
      return(complex_num);
      }

   /**
   *  Returns the difference between the two component complex numbers.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @return the difference Complex.
   */
   public static Complex diff(double x1, double y1, double x2, double y2)
      {
      Complex diff_complex_num = new Complex(x1 - x2, y1 - y2);
      
      return(diff_complex_num);
      }

   /**
   *  Returns the difference between the two Complex vectors.
   *  @param vector the other complex number as a vector.
   *  @return the difference Complex.
   */
   public Complex minus(Vector2D vector)
      {
      double diff_x = this.x - vector.x;
      double diff_y = this.y - vector.y;

      Complex diff_complex_num = new Complex(diff_x, diff_y);
      
      return(diff_complex_num);
      }

   /**
   *  Returns the difference between the two component vectors.
   *  @param x the complex number's X component
   *  @param y the complex number's Y component
   *  @return the difference Complex.
   */
   public Complex minus(double x, double y)
      {
      Complex diff_complex_num = new Complex(this.x - x, this.y - y);
      
      return(diff_complex_num);
      }


   /**
   *  Get the negative of the complex number.
   *  @return the negated complex number.
   */
   public Complex negative()
      {
      Complex complex_num = new Complex(-this.x, -this.y);
      return(complex_num);
      }

   /**
   * Calculates the complex conjugate as if it is a complex number.
   * @return the complex conjugate.
   */
   public Complex conjugate()
      {
      Complex conjugate = new Complex(this.x, -this.y);
      return(conjugate);
      }

   /**
   * Calculates the product as if they are both complex numbers.
   * <pre>
   *    (a + i * b) = (c * e - d * f) + i * (c * f + d * e)
   * </pre>
   * @param complex_num the other complex number.
   * @return the complex product.
   */
   public Complex times(Complex complex_num)
      {
      Complex product = new Complex(this.x * complex_num.x -
                                    this.y * complex_num.y,
                                    this.x * complex_num.y +
                                    this.y * complex_num.x);
      return(product);
      }

   /**
   * Calculates the reciprocal as if it is a complex number.
   * <pre>
   *    1.0 / (a + i * b) = (a - i * b) / (a^2 + b^2);
   * </pre>
   * @return the complex reciprocal.
   */
   public Complex reciprocal()
      {
      double mag_sq = this.dotProd(this);
      Complex reciprocal = new Complex(this.x / mag_sq,
                                         -this.y / mag_sq);
      return(reciprocal);
      }

   /**
   * Returns a String that represents the value of this Complex.
   * @return the string representation of the complex_num.
   */
   public String toString()
      {
      String out_string = String.format(this.format, this.x, this.y);
      return(out_string);
      }

   /**
   * Prints the representation the value of this Complex.
   */
   public void Print()
      {
      System.out.print(this);
      }
   }

