
package util.geom;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.awt.font.*;

import java.util.*;
import java.io.*;


import util.geom.*;

/**
*  This is a class for keeping track of a viewing eye's position, orientation,
*  and aim direction.
*/
public class Eye
   {

   /*
   *  Camera vectors:
   *  -- orientation 
   *     x is right,
   *     y is up, and
   *     z is into the image.
   *  -- position 
   *     This starts 5 in front of the object, on the x-y axes, and
   *     looking toward the center (origin) for the display "world".
   *
   *  The base is the original eye state, and is returned to by a "reset",
   *  while the other is the current eye_state, updated by all updates.
   */
   private Vector3D base_position = null;

   private Vector3D base_front = new Vector3D(0.0, 0.0, 1.0);
   private Vector3D base_up = new Vector3D(0.0, 1.0, 0.0);
   private Vector3D base_right = new Vector3D(1.0, 0.0, 0.0);


   private Vector3D curr_position = new Vector3D();

   private Vector3D curr_front = new Vector3D();
   private Vector3D curr_up = new Vector3D();
   private Vector3D curr_right = new Vector3D();


   /*
   *  The base_increment is the smallest increment, and the curr_increment
   *  is a multiplier.
   */
   private double base_increment = 0.01;
   private double curr_increment = 0.01;

   /*
   *  The viewable object center.
   */
   private Vector3D base_object_position = new Vector3D();
   private Vector3D curr_object_position = null;


   public boolean debug_flag = false;


   /**
   *  This constructor creates a basic eye and aims it at the object
   *  for later use and manipulation.
   *  @param eye_position the initial base position.
   *  @param object_center the point around and at which the eye is looking.
   */
   public Eye(Vector3D eye_position, Vector3D object_center)
      {

      this.base_object_position.set(object_center);
      this.setBasePositionVector(eye_position);

      this.resetView();

      Vector3D front_to_obj = (object_center.subtract(eye_position)).
                                                  getNormalizedL2();
      /*
      * Get the needed rotation axis to aim the front at the object.
      */
      Vector3D rotation_axis = this.curr_front.crossProd(front_to_obj);
      if (rotation_axis.magnitudeL2() > 0.00002)
         {
         /*
         * The rotation is needed.
         * Since the vectors are normalized we get the cos of the angle from
         * their dot product.
         */
         double angle = Math.acos(this.curr_front.dotProd(front_to_obj));
         Matrix_4x4 rotator = Matrix_4x4.GetAxisRotated(angle, rotation_axis);

         this.curr_front = Matrix_4x4.Multiply(rotator, this.curr_front);
         this.curr_up = Matrix_4x4.Multiply(rotator, this.curr_up);
         this.curr_right = Matrix_4x4.Multiply(rotator, this.curr_right);

         this.curr_object_position = this.base_object_position.clone();
         }
      }


   /**
   *  This constructor creates a basic eye and aims it at the "front" using
   *  the given "up" ane "right" directions.  The vectors are copied and
   *  normalized.
   *  @param eye_position the initial base position.
   *  @param eye_front the direction to the "front".
   *  @param eye_up the direction to the "up".
   *  @param eye_right the direction to the "right".
   */
   public Eye(Vector3D eye_position,
              Vector3D eye_front,
              Vector3D eye_up,
              Vector3D eye_right)
      {
      this.setBasePositionVector(eye_position);
      this.base_front = eye_front.getNormalizedL2();
      if (Math.abs(eye_up.dotProd(this.base_front)) <=
          Math.abs(eye_right.dotProd(this.base_front)))
         {
         this.base_right = this.base_front.crossProd(eye_up);
         this.base_up = this.base_right.crossProd(this.base_front);
         }
      else
         {
         this.base_up = eye_right.crossProd(this.base_front);
         this.base_right = this.base_front.crossProd(this.base_up);
         }

      this.base_object_position.set(eye_position.add(eye_front));

      /*
      * Now set the current vectors to this state.
      */
      this.resetView();
      }


   /**
   * Resets the viewing position.
   */
   public void resetView()
      {
      this.curr_front.set(this.base_front);
      this.curr_up.set(this.base_up);
      this.curr_right.set(this.base_right);

      this.curr_position.set(this.base_position);

      this.curr_object_position = this.base_object_position.clone();

      this.curr_increment = 1;

      if (this.debug_flag)
         {
         System.out.println("Navigate3D.display() on Entry:\n"
                          + "   curr_position = "
                             + this.curr_position
                             + "\n"
                          + "   curr_up = " + this.curr_up + "\n"
                          + "   curr_front = " + this.curr_front + "\n"
                          + "   curr_right = " + this.curr_right + "\n"
                          + "   curr_increment = "
                             + this.curr_increment
                             + "\n"
                           );
         }
      }

   /*
   * Needed functions:
   *    1. Fetch the eye vectors.
   *       a. "front"
   *       b. "up"
   *       c. "right"
   *       d. "position"
   *    2. Move "in" and "out" -- reposition along the "front" line axis.
   *    3. Move "up" and "down" -- reposition along the "up" line axis.
   *    4. Move "left" and "right" -- reposition along the "right" line axis.
   *    5. Rotate "CW" and "CCW" -- around the "front" line axis (in degrees).
   *    6. Rotate "up" and "down" -- around the "right" line axis (in degrees).
   *    7. Rotate "left" and "right" -- around the "up" line axis (in degrees).
   */

   /**
   *  Fetch the "front" vector.
   *  @return the "front" vector.
   */
   public Vector3D getCurrentFrontVector()
      {
      return(this.curr_front);
      }

   /**
   *  Fetch the "up" vector.
   *  @return the "up" vector.
   */
   public Vector3D getCurrentUpVector()
      {
      return(this.curr_up);
      }
   /**
   *  Fetch the "right" vector.
   *  @return the "right" vector.
   */
   public Vector3D getCurrentRightVector()
      {
      return(this.curr_right);
      }

   /**
   *  Set the "front" vector, adjusting the other vectors to maintain
   *  orthogonality and unit lengths..
   *  @param vector the "front" vector.
   */
   public void setCurrentFrontVector(Vector3D vector)
      {
      if (vector != null &&
          vector.normalizeL2() <= 0.0)
         {
         this.curr_front = vector.clone();
         if (Math.abs(this.curr_up.dotProd(this.curr_front)) <=
             Math.abs(this.curr_right.dotProd(this.curr_front)))
            {
            this.curr_right = this.curr_front.crossProd(this.curr_up);
            this.curr_up = this.curr_right.crossProd(this.curr_front);
            }
         else
            {
            this.curr_up = this.curr_right.crossProd(this.curr_front);
            this.curr_right = vector.crossProd(this.curr_up);
            }
         }
      }

   /**
   *  Set the "front" vector, adjusting the other vectors to maintain
   *  orthogonality and unit lengths..
   *  @param x the "front" vector x-component.
   *  @param y the "front" vector y-component.
   *  @param z the "front" vector z-component.
   */
   public void setCurrentFrontVector(double x, double y, double z)
      {
      Vector3D vector = new Vector3D(x, y, z);
      if (vector != null &&
          vector.normalizeL2() <= 0.0)
         {
         this.curr_front = vector.clone();
         if (Math.abs(this.curr_up.dotProd(this.curr_front)) <=
             Math.abs(this.curr_right.dotProd(this.curr_front)))
            {
            this.curr_right = this.curr_front.crossProd(this.curr_up);
            this.curr_up = this.curr_right.crossProd(this.curr_front);
            }
         else
            {
            this.curr_up = this.curr_right.crossProd(this.curr_front);
            this.curr_right = vector.crossProd(this.curr_up);
            }
         }
      }

   /**
   *  Set the "up" vector, adjusting the other vectors to maintain
   *  orthogonality and unit lengths..
   *  @param vector the "up" vector.
   */
   public void setCurrentUpVector(Vector3D vector)
      {
      if (vector != null &&
          vector.normalizeL2() <= 0.0)
         {
         this.curr_up = vector.clone();
         if (Math.abs(this.curr_front.dotProd(this.curr_up)) <=
             Math.abs(this.curr_right.dotProd(this.curr_up)))
            {
            this.curr_right = this.curr_front.crossProd(this.curr_up);
            this.curr_front = this.curr_up.crossProd(this.curr_right);
            }
         else
            {
            this.curr_front = this.curr_up.crossProd(this.curr_right);
            this.curr_right = this.curr_front.crossProd(this.curr_up);
            }
         }
      }

   /**
   *  Set the "up" vector, adjusting the other vectors to maintain
   *  orthogonality and unit lengths..
   *  @param x the "front" vector x-component.
   *  @param y the "front" vector y-component.
   *  @param z the "front" vector z-component.
   */
   public void setCurrentUpVector(double x, double y, double z)
      {
      Vector3D vector = new Vector3D(x, y, z);
      if (vector != null &&
          vector.normalizeL2() <= 0.0)
         {
         this.curr_up = vector.clone();
         if (Math.abs(this.curr_front.dotProd(this.curr_up)) <=
             Math.abs(this.curr_right.dotProd(this.curr_up)))
            {
            this.curr_right = this.curr_front.crossProd(this.curr_up);
            this.curr_front = this.curr_up.crossProd(this.curr_right);
            }
         else
            {
            this.curr_front = this.curr_up.crossProd(this.curr_right);
            this.curr_right = this.curr_front.crossProd(this.curr_up);
            }
         }
      }

   /**
   *  Set the "right" vector, adjusting the other vectors to maintain
   *  orthogonality and unit lengths..
   *  @param vector the "right" vector.
   */
   public void setCurrentRightVector(Vector3D vector)
      {
      if (vector != null &&
          vector.normalizeL2() <= 0.0)
         {
         this.curr_right = vector.clone();
         if (Math.abs(this.curr_up.dotProd(this.curr_right)) <=
             Math.abs(this.curr_front.dotProd(this.curr_right)))
            {
            this.curr_front = this.curr_up.crossProd(this.curr_right);
            this.curr_up = this.curr_right.crossProd(this.curr_front);
            }
         else
            {
            this.curr_up = this.curr_right.crossProd(this.curr_front);
            this.curr_front = this.curr_up.crossProd(this.curr_right);
            }
         }
      }

   /**
   *  Set the "right" vector, adjusting the other vectors to maintain
   *  orthogonality and unit lengths..
   *  @param x the "front" vector x-component.
   *  @param y the "front" vector y-component.
   *  @param z the "front" vector z-component.
   */
   public void setCurrentRightVector(double x, double y, double z)
      {
      Vector3D vector = new Vector3D(x, y, z);
      if (vector != null &&
          vector.normalizeL2() <= 0.0)
         {
         this.curr_right = vector.clone();
         if (Math.abs(this.curr_up.dotProd(this.curr_right)) <=
             Math.abs(this.curr_front.dotProd(this.curr_right)))
            {
            this.curr_front = this.curr_up.crossProd(this.curr_right);
            this.curr_up = this.curr_right.crossProd(this.curr_front);
            }
         else
            {
            this.curr_up = this.curr_right.crossProd(this.curr_front);
            this.curr_front = this.curr_up.crossProd(this.curr_right);
            }
         }
      }

   /**
   *  Fetch the base "position" vector.
   *  @return the base "position" vector.
   */
   public Vector3D getBasePositionVector()
      {
      return(this.base_position);
      }

   /**
   *  Set the base "position" vector.
   *  @param position the new base position.
   */
   public void setBasePositionVector(Vector3D position)
      {
      this.base_position = position;
      }

   /**
   *  Fetch the current "position" vector.
   *  @return the current "position" vector.
   */
   public Vector3D getCurrentPositionVector()
      {
      return(this.curr_position);
      }

   /**
   *  Set the current "position" vector.
   *  @param position the new current position.
   */
   public void setCurrentPositionVector(Vector3D position)
      {
      this.curr_position = position;
      }

   /**
   *  Fetch the current "position" vector increment.
   *  @return the current "position" vector increment.
   */
   public double getPositionIncrement()
      {
      return(this.curr_increment);
      }

   /**
   *  Set the current "position" vector increment.
   *  @param increment the new position increment.
   */
   public void setPositionIncrement(double increment)
      {
      this.curr_increment = increment;
      }

   /**
   *  Move an incremental distance along the "front" vector.
   *  @param positive true for increment, false for decrement.
   *  @return the new current positiion vector.
   */
   public Vector3D moveAlongFront(boolean positive)
      {
      if (positive)
         {
         this.curr_position.z += this.curr_increment;
         this.curr_object_position.z += this.curr_increment;
         }
      else
         {
         this.curr_position.z -= this.curr_increment;
         this.curr_object_position.z -= this.curr_increment;
         }

      return(this.curr_position);
      }

   /**
   *  Move an incremental distance along the "up" vector.
   *  @param positive true for increment, false for decrement.
   *  @return the new current positiion vector.
   */
   public Vector3D moveAlongUp(boolean positive)
      {
      if (positive)
         {
         this.curr_position.y += this.curr_increment;
         this.curr_object_position.y += this.curr_increment;
         }
      else
         {
         this.curr_position.y -= this.curr_increment;
         this.curr_object_position.y -= this.curr_increment;
         }
      return(this.curr_position);
      }

   /**
   *  Move an incremental distance along the "right" vector.
   *  @param positive true for decrement, false for increment.
   *  @return the new current positiion vector.
   */
   public Vector3D moveAlongRight(boolean positive)
      {
      if (positive)
         {
         this.curr_position.x -= this.curr_increment;
         }
      else
         {
         this.curr_position.x += this.curr_increment;
         }
      return(this.curr_position);
      }

   /**
   *  Rotate around the "front" vector, angle measured from origin looking
   *  along the vector axis.
   *  @param angle in clockwise direction from origin along the vector, in
   *     radians.
   */
   public void rotateAroundFront(double angle)
      {
      Matrix_3x3 rotation = Matrix_3x3.getZRotated(angle);

      this.curr_up = Matrix_3x3.Multiply(rotation, this.curr_up);
      this.curr_right = Matrix_3x3.Multiply(rotation, this.curr_right);
      }

   /**
   *  Rotate around the "up" vector, angle measured from origin looking along
   *  the vector axis.
   *  @param angle in clockwise direction from origin along the vector, in
   *     radians.
   */
   public void rotateAroundUp(double angle)
      {
      Matrix_3x3 rotation = Matrix_3x3.getYRotated(angle);

      this.curr_right = Matrix_3x3.Multiply(rotation, this.curr_right);
      this.curr_front = Matrix_3x3.Multiply(rotation, this.curr_front);
      }

   /**
   *  Rotate around the "right" vector, angle measured from origin looking
   *  along the vector axis.
   *  @param angle in clockwise direction from origin along the vector, in
   *     radians.
   */
   public void rotateAroundRight(double angle)
      {
      Matrix_3x3 rotation = Matrix_3x3.getXRotated(angle);

      this.curr_up = Matrix_3x3.Multiply(rotation, this.curr_up);
      this.curr_front = Matrix_3x3.Multiply(rotation, this.curr_front);
      }

   }


