
package util.geom;

import java.util.*;
import java.io.*;
import java.awt.geom.*;

import util.*;

/**
*  This class allows storing 2-D vector components.
*/
public class IntVector2D
   {

   public static String default_format = "(%3d, %3d)";
   public String format = default_format;

   public int x;
   public int y;

   /**
   *  Creates a IntVector2D object with default values of 0.0.
   */
   public IntVector2D()
      {
      this.x = 0;
      this.y = 0;
      }

   /**
   *  Creates an IntVector2D object with (x, y) component values.
   *  @param x the x-component.
   *  @param y the y-component.
   */
   public IntVector2D(int x, int y)
      {
      this.x = x;
      this.y = y;
      }

   /**
   *  Creates a IntVector2D object from an IntVector2D argument's component
   *  values.
   *  @param vector the other vector.
   */
   public IntVector2D(IntVector2D vector)
      {
      this.x = vector.x;
      this.y = vector.y;
      }

   /**
   *  Creates a IntVector2D object from a Point2D argument's component values.
   *  @param point the point to use.
   */
   public IntVector2D(Point2D point)
      {
      this.x = (int) point.getX();
      this.y = (int) point.getY();
      }

   /**
   *  Creates a IntVector2D object from a Vector2D argument's component values.
   *  @param vector the vector to use.
   */
   public IntVector2D(Vector2D vector)
      {
      this.x = (int) vector.x;
      this.y = (int) vector.y;
      }

   /**
   * Creates a new IntVector2D object from the IntVector2D's component values.
   * @return the new IntVector2D object reference.
   */
   public IntVector2D clone()
      {
      IntVector2D vector = new IntVector2D(this);
      return(vector);
      }

   /**
   * Returns the X component of this IntVector2D.
   * @return the X component value.
   */
   public int getX()
      {
      return(this.x);
      }

   /**
   * Returns the Y component of this IntVector2D.
   * @return the Y component value.
   */
   public int getY()
      {
      return(this.y);
      }

   /**
   * Returns the component values of this IntVector2D in an array.
   * @return the Y component value.
   */
   public int [] getArray()
      {
      int [] array =
         {
         this.x,
         this.y
         };
      return(array);
      }

   /**
   *  Sets the IntVector2D component values to the x and y component values.
   *  @param x the x-component.
   *  @param y the y-component.
   */
   public void set(int x, int y)
      {
      this.x = x;
      this.y = y;
      }

   /**
   *  Sets the IntVector2D object to the x and y component values of
   *  another vector.
   *  @param vector the other vector.
   */
   public void set(IntVector2D vector)
      {
      this.x = vector.x;
      this.y = vector.y;
      }

   /**
   *  Sets the IntVector2D object to the x and y component values of
   *  a Vector2D.
   *  @param vector the other vector.
   */
   public void set(Vector2D vector)
      {
      this.x = (int) vector.x;
      this.y = (int) vector.y;
      }

   /**
   *  Sets the IntVector2D object to the x and y component values of
   *  a Point2D.
   *  @param point the point.
   */
   public void set(Point2D point)
      {
      this.x = (int) point.getX();
      this.y = (int) point.getY();
      }

   /**
   *  Compares the vector and returns true if they are equal.
   *  @param vector the other vector.
   *  @return true if the vectors are the same vector or equal.
   */
   public boolean equals(IntVector2D vector)
      {
      boolean are_equal = false;
      if (this == vector ||
          (this.x == vector.x &&
           this.y == vector.y))
         {
         are_equal = true;
         }
      return(are_equal);
      }

   /**
   *  Returns the Euclidian magnitude of the Vector2D.
   */
   public double magnitude()
      {
      double sumsq = this.x * this.x + this.y * this.y;
      double mag = Math.sqrt(sumsq);
      
      return(mag);
      }

   /**
   *  Returns the square of the  magnitude of the Vector2D.
   */
   public double magnitudeSQ()
      {
      double sumsq = this.x * this.x + this.y * this.y;
      return(sumsq);
      }

   /**
   *  Returns the Euclidian angle of the Vector2D.
   */
   public double angle()
      {
      double angle = Math.atan2((double) this.y, (double) this.x);
      
      return(angle);
      }

   /**
   *  Scales the vector in-place.
   *  @param value the scale multiplier.
   */
   public void scale(double value)
      {
      this.x = (int) (value * this.x);
      this.y = (int) (value * this.y);
      }


   /**
   *  Returns the scaled vector.
   *  @param scale the scale multiplier.
   *  @return a scaled vector.
   */
   public IntVector2D getScaled(double scale)
      {
      int x_val = (int) (scale * this.x);
      int y_val = (int) (scale * this.y);

      IntVector2D vector = new IntVector2D(x_val, y_val);
      
      return(vector);
      }

   /**
   *  Returns the difference between the two Vector2D vectors.
   *  @param vector the other vector
   *  @return the difference IntVector2D.
   */
   public IntVector2D subtract(IntVector2D vector)
      {
      int diff_x = this.x - vector.x;
      int diff_y = this.y - vector.y;

      IntVector2D diff_vector = new IntVector2D(diff_x, diff_y);
      
      return(diff_vector);
      }

   /**
   *  Returns the difference between the two component vectors.
   *  @param x the vector's X component
   *  @param y the vector's Y component
   *  @return the difference IntVector2D.
   */
   public IntVector2D subtract(int x, int y)
      {
      IntVector2D diff_vector = new IntVector2D(this.x - x, this.y - y);
      
      return(diff_vector);
      }

   /**
   *  Returns the difference between the Vector2D and point coords vectors.
   *  @param point the point
   *  @return the difference IntVector2D.
   */
   public IntVector2D subtract(Point2D point)
      {
      int diff_x = this.x - (int) point.getX();
      int diff_y = this.y - (int) point.getY();

      IntVector2D diff_vector = new IntVector2D(diff_x, diff_y);
      
      return(diff_vector);
      }


   /**
   *  Add the input vector to the current vector.
   *  @param vector the vector to add.
   */
   public void addInto(IntVector2D vector)
      {
      this.x += vector.x;
      this.y += vector.y;
      }


   /**
   *  Add the input vector to the current vector.
   *  @param vector the vector to add.
   */
   public void addInto(Vector2D vector)
      {
      this.x += (int) vector.x;
      this.y += (int) vector.y;
      }


   /**
   *  Add the input vector to the current vector.
   *  @param point the point to add.
   */
   public void addInto(Point2D point)
      {
      this.x += (int) point.getX();
      this.y += (int) point.getY();
      }


   /**
   *  Add the input vector to the current vector.
   *  @param x the x-value to add.
   *  @param y the y-value to add.
   */
   public void addInto(int x, int y)
      {
      this.x += x;
      this.y += y;
      }


   /**
   *  Subtract the input vector from the current vector.
   *  @param vector the vector to subtract.
   */
   public void subtractOut(IntVector2D vector)
      {
      this.x -= vector.x;
      this.y -= vector.y;
      }


   /**
   *  Subtract the input vector from the current vector.
   *  @param vector the vector to subtract.
   */
   public void subtractOut(Vector2D vector)
      {
      this.x -= (int) vector.x;
      this.y -= (int) vector.y;
      }


   /**
   *  Subtract the input point from the current vector.
   *  @param point the point to subtract.
   */
   public void subtractOut(Point2D point)
      {
      this.x -= (int) point.getX();
      this.y -= (int) point.getY();
      }


   /**
   *  Subtract the input vector from the current vector.
   *  @param x the x-value to subtract.
   *  @param y the y-value to subtract.
   */
   public void subtractOut(int x, int y)
      {
      this.x -= x;
      this.y -= y;
      }

   /**
   *  Returns the sum of the two Vector2D vectors.
   *  @param vector the other vector
   *  @return the sum Vector2D.
   */
   public IntVector2D add(IntVector2D vector)
      {
      int sum_x = this.x + vector.x;
      int sum_y = this.y + vector.y;

      IntVector2D sum_vector = new IntVector2D(sum_x, sum_y);
      
      return(sum_vector);
      }

   /**
   *  Returns the sum of the two Vector2D vectors.
   *  @param vector the other vector
   *  @return the sum Vector2D.
   */
   public IntVector2D add(Vector2D vector)
      {
      int sum_x = this.x + (int) vector.x;
      int sum_y = this.y + (int) vector.y;

      IntVector2D sum_vector = new IntVector2D(sum_x, sum_y);
      
      return(sum_vector);
      }

   /**
   *  Returns the sum of the two Vector2D vectors.
   *  @param point the point
   *  @return the sum Point2D.
   */
   public IntVector2D add(Point2D point)
      {
      int sum_x = this.x + (int) point.getX();
      int sum_y = this.y + (int) point.getY();

      IntVector2D sum_vector = new IntVector2D(sum_x, sum_y);
      
      return(sum_vector);
      }

   /**
   *  Returns the sum of the two component vectors.
   *  @param x the vector's X component
   *  @param y the vector's Y component
   *  @return the sum Vector2D.
   */
   public IntVector2D add(int x, int y)
      {
      IntVector2D sum_vector = new IntVector2D(this.x + x, this.y + y);
      
      return(sum_vector);
      }

   /**
   *  Convert the IntVector2D into a Point2D.
   *  @return the Point2D refrence.
   */
   public Point2D toPoint2D()
      {
      return(new Point2D.Double((double) this.x, (double) this.y));
      }



   /**
   *  Returns the vector dot product between the given Vector2D vectors.
   *  @param vector1 the first vector.
   *  @param vector2 the second vector.
   *  @return the dot product scalar.
   */
   public static double dotProd(IntVector2D vector1, IntVector2D vector2)
      {
      double dot_prod = vector1.x * vector2.x + vector1.y * vector2.y;
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given
   *  vector components.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @return the dot product scalar.
   */
   public static double dotProd(int x1, int y1, int x2, int y2)
      {
      double dot_prod = x1 * x2 + y1 * y2;
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given Vector2D.
   *  @param vector the other vector.
   *  @return the dot product scalar.
   */
   public double dotProd(IntVector2D vector)
      {
      double dot_prod = this.x * vector.x + this.y * vector.y;
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given
   *  vector components.
   *  @param x the X component
   *  @param y the Y component
   *  @return the dot product scalar.
   */
   public double dotProd(int x, int y)
      {
      double dot_prod = this.x * x + this.y * y;
      
      return(dot_prod);
      }


   /**
   *  Get the negative of the vector
   *  @return the negated vector
   */
   public IntVector2D negative()
      {
      IntVector2D vector = new IntVector2D(-this.x, -this.y);
      return(vector);
      }


   /**
   *  Negate the vector.
   */
   public void negate()
      {
      this.x = -this.x;
      this.y = -this.y;
      }

   /**
   * Returns a String that represents the value of this IntVector2D.
   * @return the string representation of the vector.
   */
   public String toString()
      {
      String out_string = String.format(this.format, this.x, this.y);
      return(out_string);
      }

   /**
   * Prints the representation the value of this IntVector2D.
   */
   public void Print()
      {
      System.out.print(this);
      }
   }

