
package util.geom;

import java.util.*;
import java.io.*;

/**
*  This class allows storing 3-D point coordinates.
*  The center of the sphere is the x, y, z of the coords..
*/
public class Sphere3D extends Vector3D
   {

   /*
   *  The radius of the sphere.
   */
   double radius = 1.0;
   double radius_sq = 1.0;

   /**
   *  Creates a Sphere3D object with (0, 0, 0) as the center coordinate values,
   *  and 1.0 as the radius.
   */
   public Sphere3D()
      {
      }

   /**
   *  Creates a Sphere3D object with (x, y, and z) center coordinate values and
   *  the specified radius.
   *  @param x the x-coordinate.
   *  @param y the y-coordinate.
   *  @param z the z-coordinate.
   *  @param radius the radius.
   */
   public Sphere3D(double x,
                   double y,
                   double z,
                   double radius)
      {
      super(x, y, z);

      this.radius = radius;
      this.radius_sq = this.radius * this.radius;
      }

   /**
   *  Creates a Sphere3D object with coords for the center coordinate values
   *  and the specified radius.
   *  @param coords the center coords.
   *  @param radius the radius.
   */
   public Sphere3D(Vector3D coords,
                   double radius)
      {
      super(coords.x, coords.y, coords.z);

      this.radius = radius;
      this.radius_sq = this.radius * this.radius;
      }

   /**
   *  Creates a Sphere3D object from a Sphere3D argument's coordinate
   *  values and sizes.
   *  @param sphere the other sphere.
   */
   public Sphere3D(Sphere3D sphere)
      {
      super(sphere.x, sphere.y, sphere.z);

      this.radius = sphere.radius;
      this.radius_sq = sphere.radius_sq;
      }

   /**
   * Creates a new Sphere3D object from the Sphere3D's coordinate values.
   * @return the new Sphere3D object reference.
   */
   public Sphere3D clone()
      {
      Sphere3D sphere = new Sphere3D(this);
      return(sphere);
      }

   /**
   *  Sets the Sphere3D radius to the given value.
   *  @param radius the radius.
   */
   public void setRadius(double radius)
      {
      this.radius = radius;
      this.radius_sq = this.radius * this.radius;
      }

   /**
   *  Gets the Sphere3D center as a Vector3D.
   */
   public Vector3D getCenter()
      {
      return(new Vector3D(this.x, this.y, this.z));
      }

   /**
   *  Returns true if the Sphere3D contains the point.
   *  @param x the x-coordinate.
   *  @param y the y-coordinate.
   *  @param z the z-coordinate.
   *  @return true if the point is inside the sphere.
   */
   public boolean contains(double x, double y, double z)
      {
      boolean contains_point = true;

      double distance_sq = this.distanceL2SQ(x, y, z);
      if (distance_sq > this.radius_sq)
         {
         contains_point = false;
         }

      return(contains_point);
      }

   /**
   *  Returns true if the Sphere3D contains the point.
   *  @param point the point.
   *  @return true if the point is inside the sphere.
   */
   public boolean contains(Vector3D point)
      {
      boolean contains_point = true;

      double distance_sq = this.distanceL2SQ(point);
      if (distance_sq > this.radius_sq)
         {
         contains_point = false;
         }

      return(contains_point);
      }

   /**
   *  Returns true if the spheres intersect.
   *  @param sphere the other sphere.
   *  @return true if the two intersect.
   */
   public boolean intersect(Sphere3D sphere)
      {
      boolean intersects = false;

      /*
      *  Do their shells overlap?  Check the distance between centers
      *  relative to the sum of their radii.
      */
      Vector3D sphere_center = sphere.getCenter();
      double distance = this.distanceL2(sphere_center);
      if (distance < this.radius + sphere.radius)
         {
         intersects = true;
         }

      return(intersects);
      }

   /**
   *  Returns true if the spheres and the box intersect.
   *  @param box the box.
   *  @return true if the two intersect.
   */
   public boolean intersect(Box3D box)
      {
      boolean intersects = false;

      /*
      *  See if the sphere center is in the box.
      */
      if (box.contains(this.x, this.y, this.z))
         {
         intersects = true;
         }
      /*
      *  See if the sphere intesects any of the faces of the box.
      */
      else if (this.intersectFace(box.getFace(0)) ||
               this.intersectFace(box.getFace(1)) ||
               this.intersectFace(box.getFace(2)) ||
               this.intersectFace(box.getFace(3)) ||
               this.intersectFace(box.getFace(4)) ||
               this.intersectFace(box.getFace(5)))
         {
         intersects = true;
         }

      return(intersects);
      }

   /**
   *  Returns true if the sphere and the flat quadrilateral face intersect.
   *  The nodes must be ordered as in the illustration below.
   *  <pre>
   *
   *   Face node indexing.
   *
   *     0         1
   *      *-------*
   *      |       |
   *      |       |
   *      |       |
   *      *-------*
   *     3         2
   *
   *  </pre>
   *  @param face the face points array.
   *  @return true if the two intersect.
   */
   public boolean intersectFace(Vector3D [] face)
      {
      boolean intersects = false;

      /*
      *  See if any of the face corners are in the sphere.
      */
      if (this.contains(face[0]) ||
          this.contains(face[1]) ||
          this.contains(face[2]) ||
          this.contains(face[3]))
         {
         intersects = true;
         }
      else
         {
         /*
         *  See if the distance from the face plane to the sphere center is
         *  less than the sphere radius.
         */
         Vector3D center = this.getCenter();

         /*
         *  Get the plane normal.
         */
         Vector3D vector_0_to_1 = face[0].subtract(face[1]);
         Vector3D vector_1_to_2 = face[1].subtract(face[2]);
         Vector3D normal = vector_0_to_1.crossProd(vector_1_to_2);

         /*
         *  Project the vector to the center onto the plane normal.
         */
         Vector3D unit_normal = normal.getNormalizedL2();
         Vector3D vector_0_to_center = center.subtract(face[0]);
         double projection = Math.abs(unit_normal.
                                         dotProd(vector_0_to_center));

         if (projection < this.radius_sq)
            {
            /*
            *  It overlaps the plane, so see if it is within the bounds of
            *  the face.
            *  Project it onto the plane, and make sure the projected point
            *  is inside the defined region.
            *
            *  Use equation:
            *
            *     ax + by + cz + d = 0
            *
            *  giving
            *
            *     normal = (a, b, c)
            *
            *  and satisfied by any one of the points in the plane.
            */
            double d_value = normal.dotProd(face[0]);
            Vector3D projected_point = center.projectOntoPlane(normal.x,
                                                              normal.y,
                                                              normal.z,
                                                              d_value);
            if (projected_point.between(face[0], face[2]) &&
                projected_point.between(face[1], face[3]))
               {
               intersects = true;
               }
            }
         }

      return(intersects);
      }

   /**
   *  Returns true if the sphere intesects the line.
   *  @param line the array of two points for the line.
   *  @return true if the two intersect.
   */
   public boolean intersectLine(Vector3D [] line)
      {
      boolean intersects = false;

      if (this.contains(line[0].x, line[0].y, line[0].y) ||
          this.contains(line[1].x, line[1].y,  line[1].z))
         {
         intersects = true;
         }
      else
         {
         Vector3D line_midpoint = line[0].midpoint(line[1]);
         double line_half_length_sq = line[0].distanceL2SQ(line_midpoint);

         double distance_sq = line_midpoint.distanceL2SQ(this.x,
                                                         this.y,
                                                         this.z);
         if (distance_sq < this.radius_sq)
            {
            intersects = true;
            }
         }

      return(intersects);
      }

   /**
   *  Returns true if the sphere intesects the line.
   *  @param point1 one end of the line.
   *  @param point2 the other end of the line.
   *  @return true if the two intersect.
   */
   public boolean intersectLine(Vector3D point1, Vector3D point2)
      {
      boolean intersects = false;

      if (this.contains(point1.x, point1.y, point1.y) ||
          this.contains(point2.x, point2.y,  point2.z))
         {
         intersects = true;
         }
      else
         {
         Vector3D line_midpoint = point1.midpoint(point2);
         double line_half_length_sq = point1.distanceL2SQ(line_midpoint);

         double distance_sq = line_midpoint.distanceL2SQ(this.x,
                                                         this.y,
                                                         this.z);
         if (distance_sq < this.radius_sq)
            {
            intersects = true;
            }
         }

      return(intersects);
      }

   public String toString()
      {
      String out_string = "("
                        + this.x
                        + ", "
                        + this.y
                        + ", "
                        + this.z
                        + ": "
                        + this.radius
                        + ")"
                         ;
      return(out_string);
      }

   public void Print()
      {
      System.out.print(this);
      }
   }

