
package util.geom;

import java.util.*;
import java.io.*;
import java.util.regex.*;

import util.*;

/**
*  This class allows storing 3-D point coordinates.
*/
public class Vector3D
   {

   /**
   * Set true to print out debug information.
   */
   public static boolean debug_flag = false;

   /**
   * Vector2D basic (not detailed) value pattern
   */
   static String basic_pattern_regex = "\\s*"
                                     + "("
                                     + "\\("
                                     + "[^\\)]*"
                                     + "\\)"
                                     + ")"
                                     ;


   static Pattern basic_pattern = null;

   static String value_pattern_regex = "\\s*"
                                     + "\\("
                                     + "\\s*"
                                     + "([^,\\s]*)"
                                     + "\\s*"
                                     + ","
                                     + "\\s*"
                                     + "([^,\\s]*)"
                                     + "\\s*"
                                     + ","
                                     + "\\s*"
                                     + "([^\\)\\s]*)"
                                     + "\\s*"
                                     + "\\)"
                                     ;

   static Pattern value_pattern = null;


   /*
   *  The (x, y, z) coordinates for the point.
   */
   public double x = 0.0;
   public double y = 0.0;
   public double z = 0.0;

   /**
   *  Creates a Vector3D object with (0, 0, 0) as the coordinate values.
   */
   public Vector3D()
      {
      }

   /**
   *  Creates a Vector3D object with (x, y, and z) coordinate values.
   *  @param x the x-coordinate.
   *  @param y the y-coordinate.
   *  @param z the z-coordinate.
   */
   public Vector3D(double x, double y, double z)
      {
      this.x = x;
      this.y = y;
      this.z = z;
      }

   /**
   *  Creates a Vector3D object from a Vector3D argument's coordinate values.
   *  @param coords the other coords.
   */
   public Vector3D(Vector3D coords)
      {
      this.x = coords.x;
      this.y = coords.y;
      this.z = coords.z;
      }

   /**
   * Creates a new Vector3D object from the Vector3D's coordinate values.
   * @return the new Vector3D object reference.
   */
   public Vector3D clone()
      {
      Vector3D coords = new Vector3D(this);
      return(coords);
      }

   /**
   * Returns the X component of this Vector3D in double precision.
   * @return the X component value.
   */
   public double getX()
      {
      return(this.x);
      }

   /**
   * Returns the Y component of this Vector3D in double precision.
   * @return the Y component value.
   */
   public double getY()
      {
      return(this.y);
      }

   /**
   * Returns the Z component of this Vector3D in double precision.
   * @return the Z component value.
   */
   public double getZ()
      {
      return(this.z);
      }

   /**
   * Returns the x, y, and z components of this Vector3D in an array
   * of double precision values.
   * @return the array of coordinate values.
   */
   public double [] getArray()
      {
      double [] array =
         {
         this.x,
         this.y,
         this.x
         };
      return(array);
      }

   /**
   *  Sets the Vector3D location to the x, y, and Z coordinate values.
   *  @param x the x-component.
   *  @param y the y-component.
   *  @param z the z-component.
   */
   public void set(double x, double y, double z)
      {
      this.x = x;
      this.y = y;
      this.z = z;
      }

   /**
   *  Sets the Vector3D object to the x, y, and Z coordinate values of
   *  another point.
   *  @param coords the other coords.
   */
   public void set(Vector3D coords)
      {
      this.x = coords.x;
      this.y = coords.y;
      this.z = coords.z;
      }

   /**
   * Sets the x, y, and z components of this Vector3D from an array
   * of double precision values.
   */
   public void setArray(double [] array)
      {
      this.x = array[0];
      this.y = array[1];
      this.x = array[2];
      }

   /**
   *  Returns the Min axis (L0) magnitude of the Vector3D.
   */
   public double magnitudeL0()
      {
      double mag = Math.min(this.x, Math.min(this.y, this.z));
      return(mag);
      }

   /**
   *  Returns the Taxicab (L1) magnitude of the Vector3D.
   */
   public double magnitudeL1()
      {
      double mag = Math.abs(this.x) + Math.abs(this.y) + Math.abs(this.z);
      return(mag);
      }

   /**
   *  Returns the Cartesion (L2) magnitude of the Vector3D.
   */
   public double magnitudeL2()
      {
      double sumsq = this.x * this.x +
                     this.y * this.y +
                     this.z * this.z;
      double mag = Math.sqrt(sumsq);
      
      return(mag);
      }

   /**
   *  Returns the Max axis (LI or L-infinity) magnitude of the Vector3D.
   */
   public double magnitudeLI()
      {
      double mag = Math.max(this.x, Math.max(this.y, this.z));
      return(mag);
      }


   /**
   *  Returns the Min axis (L0) magnitude of the coordinates.
   *  @param x the X component
   *  @param y the Y component
   *  @param z the Z component
   */
   public static double magnitudeL0(double x, double y, double z)
      {
      double mag = Math.min(x, Math.min(y, z));
      return(mag);
      }

   /**
   *  Returns the Taxicab (L1) magnitude of the coordinates.
   *  @param x the X component
   *  @param y the Y component
   *  @param z the Z component
   */
   public static double magnitudeL1(double x, double y, double z)
      {
      double mag = Math.abs(x) + Math.abs(y) + Math.abs(z);
      return(mag);
      }

   /**
   *  Returns the Cartesion (L2) magnitude of the coordinates.
   *  @param x the X component
   *  @param y the Y component
   *  @param z the Z component
   */
   public static double magnitudeL2(double x, double y, double z)
      {
      double sumsq = x * x + y * y + z * z;
      double mag = Math.sqrt(sumsq);
      
      return(mag);
      }

   /**
   *  Returns the Max axis (LI or L-infinity) magnitude of the coordinates.
   *  @param x the X component
   *  @param y the Y component
   *  @param z the Z component
   */
   public static double magnitudeLI(double x, double y, double z)
      {
      double mag = Math.max(x, Math.max(y, z));
      return(mag);
      }

   /**
   *  Scales the coords.
   *  @param scale the scale multiplier.
   *  @return reference to the Vector instance.
   */
   public Vector3D scale(double scale)
      {
      this.x *= scale;
      this.y *= scale;
      this.z *= scale;
      return(this);
      }

   /**
   *  Returns a Vector3D with the coords scaled .
   *  @param scale the scale multiplier.
   *  @return a scaled vector.
   */
   public Vector3D getScaled(double scale)
      {
      double x_val = this.x * scale;
      double y_val = this.y * scale;
      double z_val = this.z * scale;

      Vector3D coords = new Vector3D(x_val, y_val, z_val);
      
      return(coords);
      }

   /**
   *  Returns the difference between the two sets of coordinates.
   *  @param x the coordinate's X component
   *  @param y the coordinate's Y component
   *  @param z the coordinate's Z component
   *  @return the difference Vector3D.
   */
   public Vector3D subtract(double x, double y, double z)
      {
      Vector3D diff_vector = new Vector3D(this.x - x, this.y - y, this.z - z);
      
      return(diff_vector);
      }

   /**
   *  Returns the difference between the two Vector3D or derivations.
   *  @param coords the other coords
   *  @return the difference Vector3D.
   */
   public Vector3D subtract(Vector3D coords)
      {
      double diff_x = this.x - coords.x;
      double diff_y = this.y - coords.y;
      double diff_z = this.z - coords.z;

      Vector3D diff_vector = new Vector3D(diff_x, diff_y, diff_z);

      return(diff_vector);
      }

   /**
   *  Subtracts the coordinate values from this coords set.
   *  @param x the vector's X component
   *  @param y the vector's Y component
   *  @param z the vector's Z component
   *  @return reference to the Vector instance.
   */
   public Vector3D subtractFrom(double x, double y, double z)
      {
      this.x -= x;
      this.y -= y;
      this.z -= z;
      return(this);
      }

   /**
   *  Subtracts the coords values from this coords set.
   *  @param coords the other coords
   *  @return reference to the Vector instance.
   */
   public Vector3D subtractFrom(Vector3D coords)
      {
      this.x -= coords.x;
      this.y -= coords.y;
      this.z -= coords.z;
      return(this);
      }

   /**
   *  Returns the difference between the two sets of coordinates.
   *  @param x1 the first coords's X component
   *  @param y1 the first coords's Y component
   *  @param z1 the first coords's Z component
   *  @param x2 the second coords's X component
   *  @param y2 the second coords's Y component
   *  @param z2 the second coords's Z component
   *  @return the difference Vector3D.
   */
   public static Vector3D getDiff(double x1, double y1, double z1,
                                  double x2, double y2, double z2)
      {
      Vector3D diff_vector = new Vector3D(x1 - x2, y1 - y2, z1 - z2);
      
      return(diff_vector);
      }


   /**
   *  Returns the difference between two Vector3D derivations.
   *  @param coords1 the first coords
   *  @param coords2 the second coords
   */
   public static Vector3D getDiff(Vector3D coords1, Vector3D coords2)
      {
      double diff_x = coords1.x - coords2.x;
      double diff_y = coords1.y - coords2.y;
      double diff_z = coords1.z - coords2.z;

      Vector3D vector = new Vector3D(diff_x, diff_y, diff_z);
      
      return(vector);
      }

   /**
   *  Returns the sum of the two sets of coordinates
   *  @param coords the other coords
   *  @return the sum Vector3D.
   */
   public Vector3D add(Vector3D coords)
      {
      double sum_x = this.x + coords.x;
      double sum_y = this.y + coords.y;
      double sum_z = this.z + coords.z;

      Vector3D sum_vector = new Vector3D(sum_x, sum_y, sum_z);
      
      return(sum_vector);
      }

   /**
   *  Returns the sum of the two sets of coordinates.
   *  @param x the coords's X component
   *  @param y the coords's Y component
   *  @param z the coords's Z component
   *  @return the sum Vector3D.
   */
   public Vector3D add(double x, double y, double z)
      {
      Vector3D sum_vector = new Vector3D(this.x + x, this.y + y, this.z + z);
      
      return(sum_vector);
      }


   /**
   *  Add the input coords to the current coords.
   *  @param coords the coords to add.
   *  @return reference to the Vector instance.
   */
   public Vector3D addTo(Vector3D coords)
      {
      this.x += coords.x;
      this.y += coords.y;
      this.z += coords.z;
      return(this);
      }


   /**
   *  Add the input coords to the current coords.
   *  @param x the x-value to add.
   *  @param y the y-value to add.
   *  @param z the z-value to add.
   *  @return reference to the Vector instance.
   */
   public Vector3D addTo(double x, double y, double z)
      {
      this.x += x;
      this.y += y;
      this.z += z;
      return(this);
      }


   /**
   *  Returns the sum of the two Vector3D sets of coordinates.
   *  @param coords1 the first coords
   *  @param coords2 the second coords
   *  @return the difference Vector3D.
   */
   public static Vector3D getSum(Vector3D coords1, Vector3D coords2)
      {
      double sum_x = coords1.x + coords2.x;
      double sum_y = coords1.y + coords2.y;
      double sum_z = coords1.z + coords2.z;

      Vector3D diff_vector = new Vector3D(sum_x, sum_y, sum_z);
      
      return(diff_vector);
      }


   /**
   *  Returns the sum of the two component sets of coordinates.
   *  @param x1 the first coords's X component
   *  @param y1 the first coords's Y component
   *  @param z1 the first coords's Z component
   *  @param x2 the second coords's X component
   *  @param y2 the second coords's Y component
   *  @param z2 the second coords's Z component
   *  @return the sum Vector3D.
   */
   public static Vector3D getSum(double x1, double y1, double z1,
                                 double x2, double y2, double z2)
      {
      Vector3D sum_vector = new Vector3D(x1 + x2, y1 + y2, z1 + z2);
      
      return(sum_vector);
      }

   /**
   *  Normalizes the coordinates, using Taxicab (L1) distance.
   *  @return the normalized Vector3D.
   */
   public Vector3D getNormalizedL1()
      {
      Vector3D result = new Vector3D();

      double mag = Math.abs(this.x) + Math.abs(this.y) + Math.abs(this.z);
      if (mag != 0.0)
         {
         double mag_i = 1.0 / mag;

         result = new Vector3D(this.x * mag_i,
                               this.y * mag_i,
                               this.z * mag_i);
         }
      else
         {
         result = new Vector3D(Double.NaN,
                               Double.NaN,
                               Double.NaN);
         }
      
      return(result);
      }

   /**
   *  Normalizes the coordinates, using Cartesian (L2) distance.
   *  @return the normalized Vector3D.
   */
   public Vector3D getNormalizedL2()
      {
      Vector3D result = new Vector3D();
      double sumsq = this.x * this.x +
                     this.y * this.y +
                     this.z * this.z;
      double mag = Math.sqrt(sumsq);
      if (mag != 0.0)
         {
         double mag_i = 1.0 / mag;

         result = new Vector3D(this.x * mag_i,
                               this.y * mag_i,
                               this.z * mag_i);
         }
      else
         {
         result = new Vector3D(Double.NaN,
                               Double.NaN,
                               Double.NaN);
         }
      
      return(result);
      }

   /**
   *  Normalizes the coordinates, using Taxicab (L1) distance.
   *  @return the magnitude before normalizing.
   */
   public double normalizeL1()
      {
      double sum = Math.abs(this.x) + Math.abs(this.y) + Math.abs(this.z);
      if (sum != 0.0)
         {
         double sum_i = 1.0 / sum;

         this.x *= sum_i;
         this.y *= sum_i;
         this.z *= sum_i;
         }
      else
         {
         this.x = Double.NaN;
         this.y = Double.NaN;
         this.z = Double.NaN;
         }
      
      return(sum);
      }

   /**
   *  Normalizes the coordinates, using Cartesian (L2) distance.
   *  @return the magnitude before normalizing.
   */
   public double normalizeL2()
      {
      double sumsq = this.x * this.x +
                     this.y * this.y +
                     this.z * this.z;
      double mag = Math.sqrt(sumsq);
      if (mag != 0.0)
         {
         double mag_i = 1.0 / mag;

         this.x *= mag_i;
         this.y *= mag_i;
         this.z *= mag_i;
         }
      else
         {
         this.x = Double.NaN;
         this.y = Double.NaN;
         this.z = Double.NaN;
         }
      
      return(mag);
      }

   /**
   *  Returns the Taxicab (L1) distance from this Vector3D to the
   *  specified coordinate.
   *  @param x the X coordinate
   *  @param y the Y coordinate
   *  @param z the Z coordinate
   */
   public double distanceL1(double x, double y, double z)
      {
      double distance = 0.0;
      double x_offset = Math.abs(this.x - x);
      double y_offset = Math.abs(this.y - y);
      double z_offset = Math.abs(this.z - z);

      double sum = x_offset + y_offset + z_offset;
      
      return(sum);
      }

   /**
   *  Returns the Cartesian (L2) distance from this Vector3D to the
   *  specified coordinate.
   *  @param x the X coordinate
   *  @param y the Y coordinate
   *  @param z the Z coordinate
   */
   public double distanceL2(double x, double y, double z)
      {
      double distance = 0.0;
      double x_offset = this.x - x;
      double y_offset = this.y - y;
      double z_offset = this.z - z;

      double sumsq = x_offset * x_offset +
                     y_offset * y_offset +
                     z_offset * z_offset;
      
      distance = Math.sqrt(sumsq);
      
      return(distance);
      }

   /**
   *  Returns the Taxicab (L1) distance from this Vector3D to the
   *  specified coordinates.
   *  @param coords the other coords.
   *  @return the L1 distance (sum of the magnitudes) for the separation.
   */
   public double distanceL1(Vector3D coords)
      {
      double distance = 0.0;
      double x_offset = Math.abs(this.x - coords.x);
      double y_offset = Math.abs(this.y - coords.y);
      double z_offset = Math.abs(this.z - coords.z);

      double sum = x_offset + y_offset + z_offset;
      
      return(sum);
      }

   /**
   *  Returns the Cartesian (L2) distance from this Vector3D to the
   *  specified coordinates.
   *  @param coords the other coords.
   *  @return the L2 (Euclidean) distance for the separation.
   */
   public double distanceL2(Vector3D coords)
      {
      double distance = 0.0;
      double x_offset = this.x - coords.x;
      double y_offset = this.y - coords.y;
      double z_offset = this.z - coords.z;

      double sumsq = x_offset * x_offset +
                     y_offset * y_offset +
                     z_offset * z_offset;
      
      distance = Math.sqrt(sumsq);
      
      return(distance);
      }

   /**
   *  Returns the distance (L2) squared from this Vector3D to the specified
   *  coordinates.
   *  @param x the X coordinate
   *  @param y the Y coordinate
   *  @param z the Z coordinate
   */
   public double distanceL2SQ(double x, double y, double z)
      {
      double x_offset = this.x - x;
      double y_offset = this.y - y;
      double z_offset = this.z - z;

      double sumsq = x_offset * x_offset +
                     y_offset * y_offset +
                     z_offset * z_offset;
      
      return(sumsq);
      }

   /**
   *  Returns the distance (L2) squared from this Vector3D to the specified
   *  coordinates.
   *  @param coords the other coords.
   */
   public double distanceL2SQ(Vector3D coords)
      {
      double x_offset = this.x - coords.x;
      double y_offset = this.y - coords.y;
      double z_offset = this.z - coords.z;

      double sumsq = x_offset * x_offset +
                     y_offset * y_offset +
                     z_offset * z_offset;
      
      return(sumsq);
      }

   /**
   *  Returns the distance (L2) squared between the two coordinate locations.
   *  @param x1 the first coords's X coordinate
   *  @param y1 the first coords's Y coordinate
   *  @param z1 the first coords's Z coordinate
   *  @param x2 the second coords's X coordinate
   *  @param y2 the second coords's Y coordinate
   *  @param z2 the second coords's Z coordinate
   */
   public static double distanceL2SQ(double x1, double y1, double z1,
                                     double x2, double y2, double z2)
      {
      double x_offset = x1 - x2;
      double y_offset = y1 - y2;
      double z_offset = z1 - z2;

      double sumsq = x_offset * x_offset +
                     y_offset * y_offset +
                     z_offset * z_offset;
      
      return(sumsq);
      }

   /**
   *  Returns the midpoint for the two sets of coordinates
   *  @param coords the other coords
   *  @return the midpoint.
   */
   public Vector3D midpoint(Vector3D coords)
      {
      double ave_x = 0.5 * (this.x + coords.x);
      double ave_y = 0.5 * (this.y + coords.y);
      double ave_z = 0.5 * (this.z + coords.z);

      Vector3D ave_point = new Vector3D(ave_x, ave_y, ave_z);
      
      return(ave_point);
      }

   /**
   *  Returns the sum of the two sets of coordinates.
   *  @param x the coords's X component
   *  @param y the coords's Y component
   *  @param z the coords's Z component
   *  @return the midpoint.
   */
   public Vector3D midpoint(double x, double y, double z)
      {
      Vector3D ave_point = new Vector3D(0.5 * (this.x + x),
                                      0.5 * (this.y + y),
                                      0.5 * (this.z + z));
      
      return(ave_point);
      }

   /**
   *  Returns the sum of the two sets of coordinates.
   *  @param x1 the first coords's X component
   *  @param y1 the first coords's Y component
   *  @param z1 the first coords's Z component
   *  @param x2 the second coords's X component
   *  @param y2 the second coords's Y component
   *  @param z2 the second coords's Z component
   *  @return the midpoint.
   */
   public static Vector3D midpoint(double x1, double y1, double z1,
                                  double x2, double y2, double z2)
      {
      Vector3D ave_point = new Vector3D(0.5 * (x1 + x2),
                                      0.5 * (y1 + y2),
                                      0.5 * (z1 + z2));
      
      return(ave_point);
      }

   /**
   *  Returns the midpoint for the two sets of coordinates
   *  @param coords_array the array of coordinates to average.
   *  @return the midpoint, or null if there are no coordinates.
   */
   public static Vector3D midpoint(Vector3D [] coords_array)
      {
      Vector3D ave_point = null;

      if (coords_array != null && coords_array.length > 0)
         {
         double ave_x = 0.0;
         double ave_y = 0.0;
         double ave_z = 0.0;

         double count = coords_array.length;
         if (count > 0.0)
            {
            double divisor = 1.0 / count;

            for (Vector3D coords: coords_array)
               {
/*
               System.out.println("Vector3D.midpoint(): coords = " + coords);
*/
               ave_x += coords.x;
               ave_y += coords.y;
               ave_z += coords.z;
               }
            ave_point = new Vector3D(divisor * ave_x,
                                    divisor * ave_y,
                                    divisor * ave_z);
            }
         else
            {
            ave_point = new Vector3D(Double.NaN,
                                     Double.NaN,
                                     Double.NaN);
            }
         }
      
      return(ave_point);
      }

   /**
   *  Returns the vector dot product between the given Vector3D vectors.
   *  @param vector1 the first vector.
   *  @param vector2 the second vector.
   *  @return the dot product scalar.
   */
   public static double dotProd(Vector3D vector1, Vector3D vector2)
      {
      double dot_prod = vector1.x * vector2.x +
                        vector1.y * vector2.y +
                        vector1.z * vector2.z;
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given
   *  vector components.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param z1 the first location's Z component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @param z2 the second location's Z component
   *  @return the dot product scalar.
   */
   public static double dotProd(double x1, double y1, double z1,
                                double x2, double y2, double z2)
      {
      double dot_prod = x1 * x2 + y1 * y2 + z1 * z2;
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given Vector3D.
   *  @param vector the other vector.
   *  @return the dot product scalar.
   */
   public double dotProd(Vector3D vector)
      {
      double dot_prod = this.x * vector.x +
                        this.y * vector.y +
                        this.z * vector.z;
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given
   *  vector components.
   *  @param x the X component
   *  @param y the Y component
   *  @param z the Z component
   *  @return the dot product scalar.
   */
   public double dotProd(double x, double y, double z)
      {
      double dot_prod = this.x * x + this.y * y + this.z * z;
      
      return(dot_prod);
      }

   /**
   *  Returns the vector from the cross product with the given Vector3D.
   *  @param vector the other vector.
   *  @return the cross product Vector3D.
   */
   public Vector3D crossProd(Vector3D vector)
      {
      double prod_x = this.y * vector.z - this.z * vector.y;
      double prod_y = this.z * vector.x - this.x * vector.z;
      double prod_z = this.x * vector.y - this.y * vector.x;

      Vector3D cross_prod = new Vector3D(prod_x, prod_y, prod_z);
      
      return(cross_prod);
      }

   /**
   *  Returns the vector from the cross product with the given vector
   *  components.
   *  @param x the X component
   *  @param y the Y component
   *  @param z the Z component
   *  @return the cross product Vector3D.
   */
   public Vector3D crossProd(double x, double y, double z)
      {
      double prod_x = this.y * z - this.z * y;
      double prod_y = this.z * x - this.x * z;
      double prod_z = this.x * y - this.y * x;

      Vector3D cross_prod = new Vector3D(prod_x, prod_y, prod_z);
      
      return(cross_prod);
      }

   /**
   *  Returns the vector from the cross product of the two given vectors.
   *  @param x1 the first X component
   *  @param y1 the first Y component
   *  @param z1 the first Z component
   *  @param x2 the second X component
   *  @param y2 the second Y component
   *  @param z2 the second Z component
   *  @return the cross product Vector3D.
   */
   public static Vector3D crossProd(double x1, double y1, double z1,
                                    double x2, double y2, double z2)
      {
      double prod_x = y1 * z2 - z1 * y2;
      double prod_y = z1 * x2 - x1 * z2;
      double prod_z = x1 * y2 - y1 * x2;

      Vector3D cross_prod = new Vector3D(prod_x, prod_y, prod_z);
      
      return(cross_prod);
      }

   /**
   *  Returns the vector from the cross product of the two given vectors.
   *  @param coords1 the first vector coordinates.
   *  @param coords2 the second vector coordinates.
   *  @return the cross product Vector3D.
   */
   public static Vector3D crossProd(Vector3D coords1, Vector3D coords2)
      {
      double prod_x = coords1.y * coords2.z - coords1.z * coords2.y;
      double prod_y = coords1.z * coords2.x - coords1.x * coords2.z;
      double prod_z = coords1.x * coords2.y - coords1.y * coords2.x;

      Vector3D cross_prod = new Vector3D(prod_x, prod_y, prod_z);
      
      return(cross_prod);
      }

   /**
   *  Returns the matrix from the exterior product with the given vector.
   *  @param vect the other vector to use.
   *  @return the exterior product 3x3 matrix.
   */
   public Matrix_3x3 exteriorProduct(Vector3D vect)
      {
      Matrix_3x3 matrix = new Matrix_3x3(this.x * vect.x,
                                         this.x * vect.y,
                                         this.x * vect.z,
                                         this.y * vect.x,
                                         this.y * vect.y,
                                         this.y * vect.z,
                                         this.z * vect.x,
                                         this.z * vect.y,
                                         this.z * vect.z);
      return(matrix);
      }

   /**
   *  Project the point onto the closest point on the plane.
   *  @param a the A coefficient.
   *  @param b the B coefficient.
   *  @param c the C coefficient.
   *  @param d the D term.
   *  @return the point on the plane whose normal passes thorugh this
   *       point.
   */
   public Vector3D projectOntoPlane(double a,
                                    double b,
                                    double c,
                                    double d)
      {
      Vector3D result_point = null;

      double numer = a * this.x + b * this.y + c * this.z + d;

      Vector3D vector = new Vector3D(a, b, c);
      double denom = vector.magnitudeL2();

      double t0 = numer / denom;

      result_point = new Vector3D(this.x + a * t0,
                                  this.y + b * t0,
                                  this.z + c * t0);
      return(result_point);
      }

   /**
   *  Return true if the point is between the endpoints, but not
   *  necessarily on the line between them.
   *  @param point1 the first endpoint.
   *  @param point2 the second endpoint.
   *  @return true if the point is between the endpoints.
   */
   public boolean between(Vector3D point1, Vector3D point2)
      {
      boolean between = false;

      if (Math.min(point1.x, point2.x) < this.x &&
          this.x < Math.max(point1.x, point2.x) &&
          Math.min(point1.y, point2.y) < this.y &&
          this.y < Math.max(point1.y, point2.y) &&
          Math.min(point1.z, point2.z) < this.z &&
          this.z < Math.max(point1.z, point2.z))
         {
         between = true;
         }
      return(between);
      }

   /**
   * Returns a String that represents the component values of this Vector3D.
   * @return the string representation of the coordinates.
   */
   public String toString()
      {
      String out_string = new String("");

      out_string = "(" + this.x + ", " + this.y + ", " + this.z + ")";
      return(out_string);
      }

   /**
   * Prints the representation the component values of this Vector3D.
   */
   public void Print()
      {
      System.out.print("(" + this.x + ", " + this.y + ", " + this.z + ")");
      }


   /**
   * The Parsable routines.
   */


   /**
   *  Get the basic "locatable" pattern for scanning text for an occurrance
   *  of a possible match..
   *  @return the pattern for locating candidate substrings.
   */
   public static Pattern getBasicPattern()
      {
      if (basic_pattern == null)
         {
         basic_pattern = Pattern.compile(basic_pattern_regex);
         }
      return(basic_pattern);
      }

   /**
   *  Determines whether the basic pattern is in the line.
   *  @param line the string of text to search for the pattern.
   *  @return the matcher for the occrrance of the object's pattern or null
   *     if it is not found.
   */
   public static Matcher findMatchInLine(String line)
      {
      Matcher matcher = null;
      if (basic_pattern == null)
         {
         basic_pattern = Pattern.compile(basic_pattern_regex);
         }
      if (line != null)
         {
         matcher = basic_pattern.matcher(line);
         }
      return(matcher);
      }

   /**
   *  Extracts the string where the basic pattern occurs in the line.
   *  @param line the string of text to search for the pattern.
   *  @return the string that was matched, or null if not found.
   */
   public static String getMatchingString(String line)
      {
      String match = null;
      return(match);
      }

   /**
   *  Parses the given string and returns the value.
   *  @param val_string the input string from which to extract the values.
   *  @return the reference to the new object with the included value
   *     or null if it failed.
   */
   public static Vector3D getVector3DValue(String val_string)
      {
      Vector3D value = null;

      if (debug_flag)
         {
         System.out.println("Vector2D.getValueObject(string) on ENTRY:\n"
                          + "   val_string = " + val_string + "\n"
                          + "   value_pattern_regex = \""
                          + value_pattern_regex
                          + "\""
                           );
         }
      Matcher value_matcher = null;

      if (value_pattern == null)
         {
         value_pattern = Pattern.compile(value_pattern_regex);
         }

      if (val_string != null)
         {
         value_matcher = value_pattern.matcher(val_string);
         }

      if (value_matcher != null &&
          value_matcher.find())
         {
         String val1_string = value_matcher.group(1).trim();
         String val2_string = value_matcher.group(2).trim();
         String val3_string = value_matcher.group(3).trim();

         if (debug_flag)
            {
            System.out.println("val1_string: " + val1_string);
            System.out.println("val2_string: " + val2_string);
            System.out.println("val3_string: " + val3_string);
            }

         Double value1 = Parsable.parseDoubleValue(val1_string);
         Double value2 = Parsable.parseDoubleValue(val2_string);
         Double value3 = Parsable.parseDoubleValue(val3_string);

         if (debug_flag)
            {
            System.out.println("   value = ("
                             + value1
                             + ", "
                             + value2
                             + ", "
                             + value3
                             + ")"
                              );
            }
         value = new Vector3D(value1, value2, value3);
         }
      return(value);
      }


   /**
   *  Parses the given string and returns the object value.
   *  @param val_string the input string from which to extract the values.
   *  @return the reference to the new object with the included value
   *     or null if it failed.
   */
   public static Object getValueObject(String val_string)
      {
      Object value_obj = getVector3DValue(val_string);

      return(value_obj);
      }

   }

