
package util.geom;

import java.util.*;
import java.io.*;
import java.util.regex.*;

import util.*;


/**
*  This class allows storing N-dimensional vectors of values.
*/
public class VectorND extends Parsable
   {

   public static boolean debug_flag = false;

   /**
   * Vector2D basic (not detailed) value pattern
   */
   static String basic_pattern_regex = "\\s*"
                                     + "("
                                     + "\\("
                                     + "[^\\)]*"
                                     + "\\)"
                                     + ")"
                                     ;


   static Pattern basic_pattern = null;

   static String value_pattern_regex = "\\s*"
                                     + "\\("
                                     + "(?:\\s*"
                                     + "([^,\\s]*)"
                                     + "\\s*"
                                     + ",)"
                                     + "(?:\\s*"
                                     + "([^,\\s]*)"
                                     + "\\s*)"
                                     + "\\)"
                                     ;

   static Pattern value_pattern = null;


   public static String default_format = "%9.3g";
   public String format = default_format;

   /*
   *  The N-dimensional vector for the point.
   */
   public double [] values = null;


   /**
   *  Creates a VectorND object with (0, ..., 0) as the vector values.
   *  @param num_dims the number of dimensions to initialize to 0.0
   */
   public VectorND(int num_dims)
      {
      this.values = new double[num_dims];
      for (int dim_index = 0;
           dim_index < num_dims;
           dim_index++)
         {
         this.values[dim_index] = 0.0;
         }
      }

   /**
   *  Creates a VectorND object using the array of values.
   *  values array.
   *  @param vector_values the vector component values array to reference.
   */
   public VectorND(double ... vector_values)
      {
      this.values = new double[vector_values.length];
      for (int entry_index = 0;
           entry_index < vector_values.length;
           entry_index++)
         {
         this.values[entry_index] = vector_values[entry_index];
         }
      }

   /**
   *  Creates a VectorND object using the array of values.
   *  values array.
   *  @param vector_values the vector component values array to reference.
   */
   public VectorND(Double [] vector_values)
      {
      this.values = new double[vector_values.length];
      for (int entry_index = 0;
           entry_index < vector_values.length;
           entry_index++)
         {
         this.values[entry_index] = vector_values[entry_index];
         }
      }

   /**
   *  Creates a VectorND object from a VectorND argument's values.
   *  @param vector the other vector.
   */
   public VectorND(VectorND vector)
      {
      this.values = new double[vector.values.length];
      for (int dim_index = 0;
           dim_index < vector.values.length;
           dim_index++)
         {
         this.values[dim_index] = vector.values[dim_index];
         }
      }

   /**
   * Creates a new VectorND object from the VectorND's values.
   * @return the new VectorND object reference.
   */
   public VectorND clone()
      {
      VectorND vector = new VectorND(this);
      return(vector);
      }

   /**
   * Get the size of the array of vector values (the dimensionality).
   * @return the number of elements in the vector.
   */
   public int size()
      {
      return(this.values.length);
      }

   /**
   *  Compares the vectors and returns true if they are equal.
   *  @param vector the other vector.
   *  @return true if the vectors are the same vector or equal.
   */
   public boolean equals(VectorND vector)
      {
      boolean are_equal = false;
      if (this == vector)
         {
         are_equal = true;
         }
      else if (this.values.length == vector.values.length)
         {
         are_equal = true;
         for (int entry_index = 0;
              entry_index < this.values.length && are_equal;
              entry_index++)
            {
            if (this.values[entry_index] != vector.values[entry_index])
               {
               are_equal = false;
               }
            }
         }
      return(are_equal);
      }

   /**
   *  Compares the vectors and returns true if they are close using the
   *  L0 distance measure (min difference).
   *  @param vector the other vector.
   *  @param threshold the threshold (it absolute value is used.)
   *  @return true if the vectors are as close as or closer than the
   *      threshold distance.
   */
   public boolean closeL0(VectorND vector, double threshold)
      {
      boolean are_close = false;
      if (this.equals(vector))
         {
         are_close = true;
         }
      else
         {
         double distance = this.distanceL0(vector.values);
         if (distance <= Math.abs(threshold))
            {
            are_close = true;
            }
         }
      return(are_close);
      }

   /**
   *  Compares the vectors and returns true if they are close using the
   *  L1 distance measure (Taxicab distance).
   *  @param vector the other vector.
   *  @param threshold the threshold (it absolute value is used.)
   *  @return true if the vectors are as close as or closer than the
   *      threshold distance.
   */
   public boolean closeL1(VectorND vector, double threshold)
      {
      boolean are_close = false;
      if (this.equals(vector))
         {
         are_close = true;
         }
      else
         {
         double distance = this.distanceL1(vector.values);
         if (distance <= Math.abs(threshold))
            {
            are_close = true;
            }
         }
      return(are_close);
      }

   /**
   *  Compares the vectors and returns true if they are close using the
   *  L2 distance measure (square root of the sum of squares).
   *  @param vector the other vector.
   *  @param threshold the threshold (it absolute value is used.)
   *  @return true if the vectors are as close as or closer than the
   *      threshold distance.
   */
   public boolean closeL2(VectorND vector, double threshold)
      {
      boolean are_close = false;
      if (this.equals(vector))
         {
         are_close = true;
         }
      else
         {
         double distanceSQ = this.distanceL2SQ(vector.values);
         if (distanceSQ <= threshold * threshold)
            {
            are_close = true;
            }
         }
      return(are_close);
      }

   /**
   *  Compares the vectors and returns true if they are close using the
   *  LI distance measure (max difference).
   *  @param vector the other vector.
   *  @param threshold the threshold (it absolute value is used.)
   *  @return true if the vectors are as close as or closer than the
   *      threshold distance.
   */
   public boolean closeLI(VectorND vector, double threshold)
      {
      boolean are_close = false;
      if (this.equals(vector))
         {
         are_close = true;
         }
      else
         {
         double distance = this.distanceLI(vector.values);
         if (distance <= Math.abs(threshold))
            {
            are_close = true;
            }
         }
      return(are_close);
      }

   /**
   * Returns the dim_index component of this VectorND in double precision.
   * @return the dim_index component's value.
   */
   public double getDimValue(int dim_index)
      {
      return(this.values[dim_index]);
      }

   /**
   * Returns the reference to the component arrays of this VectorND (an array
   * of double precision values).
   * @return the reference to the array of values.
   */
   public double [] getValues()
      {
      return(this.values);
      }

   /**
   * Returns an array of Double values.
   * @return the reference to the array of values.
   */
   public Double [] getDoubleValues()
      {
      Double [] values = new Double[this.values.length];

      for (int entry_index = 0;
           entry_index < this.values.length;
           entry_index++)
         {
         values[entry_index] = this.values[entry_index];
         }
      return(values);
      }

   /**
   * Returns a copy of the component values of this VectorND (an array
   * of double precision values).
   * @return the reference to the array of values.
   */
   public double [] getValuesArrayCopy()
      {
      double [] array = new double[this.values.length];
      for (int dim_index = 0;
           dim_index < this.values.length;
           dim_index++)
         {
         array[dim_index] = this.values[dim_index];
         }
      return(array);
      }

   /**
   *  Sets the VectorND values to the given values.
   *  @param vector_values the new array of values.
   */
   public void set(double [] vector_values)
      {
      this.values = new double[vector_values.length];
      for (int dim_index = 0;
           dim_index < vector_values.length;
           dim_index++)
         {
         this.values[dim_index] = vector_values[dim_index];
         }
      }

   /**
   *  Sets the VectorND reference to the given values array.
   *  @param vector_values the values array reference.
   */
   public void setRef(double [] vector_values)
      {
      this.values = vector_values;
      }

   /**
   *  Sets the VectorND object to the values of another vector.
   *  @param vector the other vector.
   */
   public void set(VectorND vector)
      {
      this.values = new double[vector.values.length];
      for (int dim_index = 0;
           dim_index < vector.values.length;
           dim_index++)
         {
         this.values[dim_index] = vector.values[dim_index];
         }
      }

   /**
   *  Returns the Min axis (L0) magnitude of the VectorND.
   */
   public double getMagnitudeL0()
      {
      double mag = Double.MAX_VALUE;
      for (int dim_index = 0;
           dim_index < this.values.length;
           dim_index++)
         {
         mag = Math.min(this.values[dim_index], mag);
         }
      return(mag);
      }

   /**
   *  Returns the Taxicab (L1) magnitude of the VectorND.
   */
   public double getMagnitudeL1()
      {
      double mag = 0.0;
      for (int dim_index = 0;
           dim_index < this.values.length;
           dim_index++)
         {
         mag += Math.abs(this.values[dim_index]);
         }
      return(mag);
      }

   /**
   *  Returns the Cartesion (L2) magnitude of the VectorND.
   */
   public double getMagnitudeL2()
      {
      double sumsq = 0.0;
      for (int dim_index = 0;
           dim_index < this.values.length;
           dim_index++)
         {
         sumsq += this.values[dim_index] * this.values[dim_index];
         }
      double mag = Math.sqrt(sumsq);
      
      return(mag);
      }

   /**
   *  Returns the Max axis (LI or L-infinity) magnitude of the VectorND.
   */
   public double getMagnitudeLI()
      {
      double mag = -Double.MAX_VALUE;
      for (int dim_index = 0;
           dim_index < this.values.length;
           dim_index++)
         {
         mag = Math.max(this.values[dim_index], mag);
         }
      return(mag);
      }

   /**
   *  Returns the Min axis (L0) magnitude of the vector.
   *  @param vector_values the array of values.
   */
   public static double getMagnitudeL0(double [] vector_values)
      {
      double mag = Double.MAX_VALUE;
      for (int dim_index = 0;
           dim_index < vector_values.length;
           dim_index++)
         {
         mag = Math.min(vector_values[dim_index], mag);
         }
      return(mag);
      }

   /**
   *  Returns the Taxicab (L1) magnitude of the vector.
   *  @param vector_values the array of values.
   */
   public static double getMagnitudeL1(double [] vector_values)
      {
      double mag = 0.0;
      for (int dim_index = 0;
           dim_index < vector_values.length;
           dim_index++)
         {
         mag += Math.abs(vector_values[dim_index]);
         }
      return(mag);
      }

   /**
   *  Returns the Cartesion (L2) magnitude of the vector.
   *  @param vector_values the array of vector values.
   */
   public static double getMagnitudeL2(double [] vector_values)
      {
      double sumsq = 0.0;
      for (int dim_index = 0;
           dim_index < vector_values.length;
           dim_index++)
         {
         sumsq += vector_values[dim_index] * vector_values[dim_index];
         }
      double mag = Math.sqrt(sumsq);
      
      return(mag);
      }

   /**
   *  Returns the Max axis (LI or L-infinity) magnitude of the vector.
   *  @param vector_values the array of values.
   */
   public static double getMagnitudeLI(double [] vector_values)
      {
      double mag = -Double.MAX_VALUE;
      for (int dim_index = 0;
           dim_index < vector_values.length;
           dim_index++)
         {
         mag = Math.max(vector_values[dim_index], mag);
         }
      return(mag);
      }

   /**
   *  Scales the vector.
   *  @param scale the scale multiplier.
   */
   public VectorND scale(double scale)
      {
      for (int dim_index = 0;
           dim_index < this.values.length;
           dim_index++)
         {
         this.values[dim_index] *= scale;
         }
      return(this);
      }

   /**
   *  Returns a VectorND with the vector scaled .
   *  @param scale the scale multiplier.
   *  @return a scaled vector.
   */
   public VectorND getScaled(double scale)
      {
      VectorND vector = new VectorND(this);
      vector.scale(scale);
      
      return(vector);
      }

   /**
   *  Returns the difference between the two vectors of values.
   *  @param vector_values the values to subtract.
   *  @return the difference VectorND.
   */
   public VectorND subtract(double [] vector_values)
      {
      VectorND diff_vector = new VectorND(this);
 
      for (int dim_index = 0;
           dim_index < vector_values.length;
           dim_index++)
         {
         diff_vector.values[dim_index] -= vector_values[dim_index];
         }
      
      return(diff_vector);
      }

   /**
   *  Returns the difference between the two VectorND or derivations.
   *  @param vector the other vector
   *  @return the difference VectorND.
   */
   public VectorND subtract(VectorND vector)
      {
      VectorND diff_vector = this.subtract(vector.values);

      return(diff_vector);
      }

   /**
   *  Subtracts the vector of values from the current vector of values.
   *  @param vector_values the values.
   */
   public void minus(double [] vector_values)
      {
      for (int dim_index = 0;
           dim_index < vector_values.length;
           dim_index++)
         {
         this.values[dim_index] -= vector_values[dim_index];
         }
      }

   /**
   *  Subtracts the vector of values from the current vector of values.
   *  @param vector the other vector
   */
   public void minus(VectorND vector)
      {
      this.minus(vector.values);
      }

   /**
   *  Returns the difference between the two vectors of values.
   *  @param vector_values1 the first vector values.
   *  @param vector_values2 the second vector values.
   *  @return the difference VectorND.
   */
   public static VectorND getDiff(double [] vector_values1,
                                  double [] vector_values2)
      {
      VectorND diff_vector = new VectorND(vector_values1);
      diff_vector.minus(vector_values2);
      return(diff_vector);
      }


   /**
   *  Returns the difference between two VectorND derivations.
   *  @param vector1 the first vector
   *  @param vector2 the second vector
   */
   public static VectorND getDiff(VectorND vector1, VectorND vector2)
      {
      VectorND diff_vector = new VectorND(vector1.values);
      diff_vector.minus(vector2.values);
      return(diff_vector);
      }

   /**
   *  Returns the sum of the two vectors of values.
   *  @param vector_values the values.
   *  @return the sum VectorND.
   */
   public VectorND add(double [] vector_values)
      {
      VectorND sum_vector = new VectorND(this);
      for (int dim_index = 0;
           dim_index < vector_values.length;
           dim_index++)
         {
         sum_vector.values[dim_index] += vector_values[dim_index];
         }
      
      return(sum_vector);
      }

   /**
   *  Returns the sum of the two vectors of values.
   *  @param vector the other vector
   *  @return the sum VectorND.
   */
   public VectorND add(VectorND vector)
      {
      VectorND sum_vector = new VectorND(this);
      sum_vector.add(vector.values);
      return(sum_vector);
      }


   /**
   *  Add the input vector of values to the current vector of values.
   *  @param vector_values the values.
   */
   public void addTo(double [] vector_values)
      {
      for (int dim_index = 0;
           dim_index < vector_values.length;
           dim_index++)
         {
         this.values[dim_index] += vector_values[dim_index];
         }
      }


   /**
   *  Add the input vector of values to the current vector of values.
   *  @param vector the vector to add.
   */
   public void addTo(VectorND vector)
      {
      this.addTo(vector.values);
      }


   /**
   *  Returns the sum of the two values arrays as a vector.
   *  @param vector_values1 the first values array.
   *  @param vector_values2 the second values array.
   *  @return the sum VectorND.
   */
   public static VectorND getSum(double [] vector_values1,
                                 double [] vector_values2)
      {
      VectorND sum_vector = new VectorND(vector_values1);
      sum_vector.addTo(vector_values2);
      
      return(sum_vector);
      }


   /**
   *  Returns the sum of the two VectorND vectors of values.
   *  @param vector1 the first vector
   *  @param vector2 the second vector
   *  @return the sum VectorND.
   */
   public static VectorND getSum(VectorND vector1, VectorND vector2)
      {
      VectorND sum_vector = new VectorND(vector1);
      sum_vector.addTo(vector2.values);
      
      return(sum_vector);
      }

   /**
   *  Normalizes the vector of values, using Min axis (L0) distance.
   *  @return the magnitude before normalizing.
   */
   public double normalizeL0()
      {
      double mag = this.getMagnitudeL0();
      if (mag != 0.0)
         {
         double mag_i = 1.0 / mag;
         this.scale(mag_i);
         }
      
      return(mag);
      }

   /**
   *  Normalizes the vector of values, using Taxicab (L1) distance.
   *  @return the magnitude before normalizing.
   */
   public double normalizeL1()
      {
      double mag = this.getMagnitudeL1();
      if (mag != 0.0)
         {
         double mag_i = 1.0 / mag;
         this.scale(mag_i);
         }
      
      return(mag);
      }

   /**
   *  Normalizes the vector of values, using Cartesian (L2) distance.
   *  @return the magnitude before normalizing.
   */
   public double normalizeL2()
      {
      double mag = this.getMagnitudeL2();
      if (mag != 0.0)
         {
         double mag_i = 1.0 / mag;
         this.scale(mag_i);
         }
      
      return(mag);
      }

   /**
   *  Normalizes the vector of values, using Max axis (LI or L-infinity)
   *  distance.
   *  @return the magnitude before normalizing.
   */
   public double normalizeLI()
      {
      double mag = this.getMagnitudeLI();
      if (mag != 0.0)
         {
         double mag_i = 1.0 / mag;
         this.scale(mag_i);
         }
      
      return(mag);
      }

   /**
   *  Get a vector of the normalized vector values, using Min axis (L0)
   *  distance.
   *  @return the normalized VectorND.
   */
   public VectorND getNormalizedL0()
      {
      VectorND result = new VectorND(this);
      result.normalizeL0();
      
      return(result);
      }

   /**
   *  Get a vector of the normalized vector values, using Taxicab (L1)
   *  distance.
   *  @return the normalized VectorND.
   */
   public VectorND getNormalizedL1()
      {
      VectorND result = new VectorND(this);
      result.normalizeL1();
      
      return(result);
      }

   /**
   *  Get a vector of the normalized vector values, using Cartesian (L2)
   *  distance.
   *  @return the normalized VectorND.
   */
   public VectorND getNormalizedL2()
      {
      VectorND result = new VectorND(this);
      result.normalizeL2();
      
      return(result);
      }

   /**
   *  Get a vector of the normalized vector values, using Max axis (LI or
   *  L-infinity) distance.
   *  @return the normalized VectorND.
   */
   public VectorND getNormalizedLI()
      {
      VectorND result = new VectorND(this);
      result.normalizeLI();
      
      return(result);
      }

   /**
   *  Returns the Min axis (L0) distance from one vector values array to
   *  the other values array (comparing only for the length of the shortest
   *  array.
   *  @param vector_values1 the first values array.
   *  @param vector_values2 the other values array.
   *  @return the L0 distance between their component entries.
   */
   public static double distanceL0(double [] vector_values1,
                                   double [] vector_values2)
      {
      double distance = Double.MAX_VALUE;
      for (int dim_index = 0;
           dim_index < vector_values1.length &&
           dim_index < vector_values2.length;
           dim_index++)
         {
         distance = Math.min(vector_values1[dim_index] -
                             vector_values2[dim_index],
                             distance);
         }
      return(distance);
      }

   /**
   *  Returns the Taxicab (L1) distance from one vector values array to the
   *  other values array (comparing only for the length of the shortest array.
   *  @param vector_values1 the first values array.
   *  @param vector_values2 the other values array.
   *  @return the L1 distance between their component entries.
   */
   public static double distanceL1(double [] vector_values1,
                                   double [] vector_values2)
      {
      double distance = 0.0;
      for (int dim_index = 0;
           dim_index < vector_values1.length &&
           dim_index < vector_values2.length;
           dim_index++)
         {
         distance += Math.abs(vector_values1[dim_index] -
                              vector_values2[dim_index]);
         }
      return(distance);
      }


   /**
   *  Returns the Cartesian (L2) distance from one vector values array to the
   *  other values array (comparing only for the length of the shortest array.
   *  @param vector_values1 the first values array.
   *  @param vector_values2 the other values array.
   *  @return the L2 distance between their component entries.
   */
   public static double distanceL2(double [] vector_values1,
                                   double [] vector_values2)
      {
      double distance = 0.0;
      double sumsq = 0.0;
      for (int dim_index = 0;
           dim_index < vector_values1.length &&
           dim_index < vector_values2.length;
           dim_index++)
         {
         double offset = vector_values1[dim_index] - vector_values2[dim_index];
         sumsq += offset * offset;
         }
      distance = Math.sqrt(sumsq);
      return(distance);
      }


   /**
   *  Returns the Max axis (LI) distance from one vector values array to
   *  the other values array (comparing only for the length of the shortest
   *  array. (The L-infinity distance)
   *  @param vector_values1 the first values array.
   *  @param vector_values2 the other values array.
   *  @return the LI distance between their component entries.
   */
   public static double distanceLI(double [] vector_values1,
                                   double [] vector_values2)
      {
      double distance = -Double.MAX_VALUE;
      for (int dim_index = 0;
           dim_index < vector_values1.length &&
           dim_index < vector_values2.length;
           dim_index++)
         {
         distance = Math.max(vector_values1[dim_index] -
                             vector_values2[dim_index],
                             distance);
         }
      return(distance);
      }

   /**
   *  Returns the Min axis (L0) distance from this VectorND to the
   *  specified values array.
   *  @param vector_values the values array.
   *  @return the L0 distance between their component entries.
   */
   public double distanceL0(double [] vector_values)
      {
      double distance = Double.MAX_VALUE;
      for (int dim_index = 0;
           dim_index < vector_values.length &&
           dim_index < this.values.length;
           dim_index++)
         {
         distance = Math.min(this.values[dim_index] -
                             vector_values[dim_index],
                             distance);
         }
      return(distance);
      }

   /**
   *  Returns the Taxicab (L1) distance from this VectorND to the
   *  specified values array.
   *  @param vector_values the values array.
   *  @return the L1 distance between their component entries.
   */
   public double distanceL1(double [] vector_values)
      {
      double distance = 0.0;
      for (int dim_index = 0;
           dim_index < vector_values.length &&
           dim_index < this.values.length;
           dim_index++)
         {
         distance += Math.abs(this.values[dim_index]
                            - vector_values[dim_index]);
         }
      return(distance);
      }


   /**
   *  Returns the Cartesian (L2) distance from this VectorND to the
   *  specified values array.
   *  @param vector_values the values array.
   *  @return the L2 distance between their component entries.
   */
   public double distanceL2(double [] vector_values)
      {
      double distance = 0.0;
      double sumsq = 0.0;
      for (int dim_index = 0;
           dim_index < vector_values.length &&
           dim_index < this.values.length;
           dim_index++)
         {
         double offset = this.values[dim_index] - vector_values[dim_index];
         sumsq += offset * offset;
         }
      distance = Math.sqrt(sumsq);
      
      return(distance);
      }

   /**
   *  Returns the Max axis (LI) distance from this VectorND to the
   *  specified values array. (The L-infinity distance)
   *  @param vector_values the values array.
   *  @return the LI distance between their component entries.
   */
   public double distanceLI(double [] vector_values)
      {
      double distance = -Double.MAX_VALUE;
      for (int dim_index = 0;
           dim_index < vector_values.length &&
           dim_index < this.values.length;
           dim_index++)
         {
         distance = Math.max(this.values[dim_index] -
                             vector_values[dim_index],
                             distance);
         }
      return(distance);
      }

   /**
   *  Returns the Min axis (L0) distance from this vector of values to the
   *  specified vector of values.
   *  @param vector the other vector.
   *  @return the L0 distance between their component entries.
   */
   public double distanceL0(VectorND vector)
      {
      double distance = this.distanceL0(vector.values);
      
      return(distance);
      }

   /**
   *  Returns the Taxicab (L1) distance from this vector of values to the
   *  specified vector of values.
   *  @param vector the other vector.
   *  @return the L1 distance between their component entries.
   */
   public double distanceL1(VectorND vector)
      {
      double distance = this.distanceL1(vector.values);
      
      return(distance);
      }

   /**
   *  Returns the Cartesian (L2) distance from this vector of values to the
   *  specified vector of values.
   *  @param vector the other vector.
   *  @return the L2 distance between their component entries.
   */
   public double distanceL2(VectorND vector)
      {
      double distance = this.distanceL2(vector.values);
      
      return(distance);
      }

   /**
   *  Returns the Max axis (LI) distance from this vector of values to the
   *  specified vector of values.
   *  @param vector the other vector.
   *  @return the LI distance between their component entries.
   */
   public double distanceLI(VectorND vector)
      {
      double distance = this.distanceLI(vector.values);
      
      return(distance);
      }


   /**
   *  Returns the Cartesian (L2) distance squared from one vector values
   *  array to the other values array (comparing only for the length of
   *  the shortest array.
   *  @param vector_values1 the first values array.
   *  @param vector_values2 the other values array.
   *  @return the L2 distance squared between their component entries.
   */
   public static double distanceL2SQ(double [] vector_values1,
                                     double [] vector_values2)
      {
      double sumsq = 0.0;
      for (int dim_index = 0;
           dim_index < vector_values1.length &&
           dim_index < vector_values2.length;
           dim_index++)
         {
         double offset = vector_values1[dim_index] - vector_values2[dim_index];
         sumsq += offset * offset;
         }
      return(sumsq);
      }

   /**
   *  Returns the distance (L2) squared from this vector of values to the
   *  specified array of values.
   *  @param vector_values the array of values.
   *  @return the L2 distance squared between their component entries.
   */
   public double distanceL2SQ(double [] vector_values)
      {
      double sumsq = 0.0;
      for (int dim_index = 0;
           dim_index < vector_values.length &&
           dim_index < this.values.length;
           dim_index++)
         {
         double offset = this.values[dim_index] - vector_values[dim_index];
         sumsq += offset * offset;
         }
      
      return(sumsq);
      }

   /**
   *  Returns the distance (L2) squared from this vector of values to the
   *  specified vector of values.
   *  @param vector the other vector.
   *  @return the L2 distance squared between their component entries.
   */
   public double distanceL2SQ(VectorND vector)
      {
      double sumsq = this.distanceL2SQ(vector.values);
      
      return(sumsq);
      }

   /**
   *  Returns the midpoint for the two vectors of values.
   *  @param vector the other vector
   *  @return the midpoint.
   */
   public PointND midpoint(VectorND vector)
      {
      PointND ave_point = new PointND(this);

      ave_point.addTo(vector.values);
      ave_point.scale(0.5);
      
      return(ave_point);
      }

   /**
   *  Returns the sum of the two vectors of values.
   *  @param vector_values the values array.
   *  @return the midpoint.
   */
   public PointND midpoint(double [] vector_values)
      {
      PointND ave_point = new PointND(this);

      ave_point.addTo(vector_values);
      ave_point.scale(0.5);
      
      return(ave_point);
      }

   /**
   *  Returns the sum of the two vectors of values.
   *  @param vector_values1 the first values array.
   *  @param vector_values2 the second values array.
   *  @return the midpoint.
   */
   public static PointND midpoint(double [] vector_values1,
                                  double [] vector_values2)
      {
      PointND ave_point = new PointND(vector_values1);
      ave_point.addTo(vector_values2);
      ave_point.scale(0.5);
      
      return(ave_point);
      }

   /**
   *  Returns the midpoint for the two vectors of values.
   *  @param vector_array the array of vectors of values to average.
   *  @return the midpoint, or null if there are no vectors of values.
   */
   public static PointND midpoint(VectorND [] vector_array)
      {
      PointND ave_point = null;

      if (vector_array != null && vector_array.length > 0)
         {
         ave_point = new PointND(vector_array[0]);

         double divisor = 1.0 / vector_array.length;

         for (int vector_index = 1;
              vector_index < vector_array.length;
              vector_index++)
            {
            VectorND vector = vector_array[vector_index];
            ave_point.addTo(vector);
            }
         ave_point.scale(divisor);
         }
      
      return(ave_point);
      }

   /**
   *  Returns the scalar value of the dot product with the given
   *  vector components.
   *  @param vector_values the vector values array.
   *  @return the dot product scalar.
   */
   public double dotProd(double [] vector_values)
      {
      double dot_prod = 0.0;
      for (int dim_index = 0;
           dim_index < vector_values.length;
           dim_index++)
         {
         dot_prod += this.values[dim_index] + vector_values[dim_index];
         }
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given VectorND.
   *  @param vector the other vector.
   *  @return the dot product scalar.
   */
   public double dotProd(VectorND vector)
      {
      double dot_prod = this.dotProd(vector.values);
      
      return(dot_prod);
      }

   /**
   *  Returns the vector dot product between the given VectorND vectors.
   *  @param vector1 the first vector.
   *  @param vector2 the second vector.
   *  @return the dot product scalar.
   */
   public static double dotProd(VectorND vector1, VectorND vector2)
      {
      double dot_prod = dotProd(vector1.values, vector2.values);
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given
   *  vector components.
   *  @param vector_values1 the first vector component values.
   *  @param vector_values2 the second vector component values.
   *  @return the dot product scalar.
   */
   public static double dotProd(double [] vector_values1,
                                double [] vector_values2)
      {
      double dot_prod = 0.0;
      for (int dim_index = 0;
           dim_index < vector_values1.length;
           dim_index++)
         {
         dot_prod += vector_values1[dim_index] + vector_values2[dim_index];
         }
      
      return(dot_prod);
      }

   /**
   *  Returns the vector from the element-by-element product with the
   *  given vector's components.
   *  @param vector_values the vector component values.
   *  @return the product VectorND.
   */
   public VectorND multiply(double [] vector_values)
      {
      VectorND prod = new VectorND(this);
      for (int dim_index = 0;
           dim_index < vector_values.length;
           dim_index++)
         {
         prod.values[dim_index] *= vector_values[dim_index];
         }
      return(prod);
      }

   /**
   *  Returns the vector from the element-by-element product with the
   *  given vector's components.
   *  @param vector the other vector.
   *  @return the product VectorND.
   */
   public VectorND multiply(VectorND vector)
      {
      VectorND prod = this.multiply(vector.values);
      return(prod);
      }

   /**
   *  Returns the vector from the element-by-element product of the
   *  two vectors' components.
   *  @param vector_values1 the first vector component values.
   *  @param vector_values2 the second vector component values.
   *  @return the product VectorND.
   */
   public static VectorND multiply(double [] vector_values1,
                                   double [] vector_values2)
      {
      VectorND prod = new VectorND(vector_values1);
      prod.multiply(vector_values2);
      return(prod);
      }

   /**
   *  Returns the vector from the cross product of the two given vectors.
   *  @param vector1 the first vector of values.
   *  @param vector2 the second vector of values.
   *  @return the product VectorND.
   */
   public static VectorND crossProd(VectorND vector1, VectorND vector2)
      {
      VectorND prod = new VectorND(vector1);
      prod.multiply(vector2.values);
      return(prod);
      }

   /**
   *  Returns the matrix from the exterior product with the given vector.
   *  @param vector the other vector to use.
   *  @return the NxM matrix (array of row vectors) for the  exterior product.
   */
   public VectorND [] exteriorProduct(VectorND vector)
      {
      VectorND [] matrix = new VectorND[this.values.length];
      for (int row_index = 0;
           row_index < this.values.length;
           row_index++)
         {
         VectorND row_vector = new VectorND(this);
         for (int col_index = 0;
              col_index < vector.values.length;
              col_index++)
            {
            row_vector.scale(vector.values[col_index]);
            }
         matrix[row_index] = row_vector;
         }
      return(matrix);
      }

   /**
   *  Project the vector as point coordinates onto the closest point on the
   *  hyperplane.
   *  @param a_coefs the hyperplane variable coefficients
   *  @param d_term the D term.
   *  @return the point on the hyperplane whose normal passes through this
   *       point.
   */
   public VectorND projectOntoHyperPlane(double [] a_coefs,
                                         double d_term)
      {
      VectorND result_vector = null;

      /*
      *  The hyperplane for the projection is given by
      *
      *   0.0 = A[0] * x[0]  + A[1] * x[1]  + ...  + A[n-1] * x[n-1]  + D
      *
      *  The a_coefs array of coefficients can be put into a vector:
      *
      *     a_coefs = (A[0], A[1], ... , A[n-1])
      *
      * or
      *
      *     A = a_coefs
      *
      * and
      *
      *     D = d_term
      * 
      */
      VectorND vector = new VectorND(a_coefs);

      /*
      * But the equation is nonzero for any point not on the plane, so the
      * equation for the parallel plane through the point of interest,
      * indicated as x', as
      *
      *   D` = A[0] * x'[0] + A[1] * x'[1] + ...  + A[n-1] * x'[n-1] + D
      *
      * where
      *
      *   D` = dp_term
      *
      * in the following code.
      *
      */
      double dp_term = this.dotProd(a_coefs) + d_term;

      /*
      * The A magnitude, a_mag, is given by |A| and can give the normalized
      * coefficients from:
      *
      *    a[i] = A[i] / |A|
      *
      * or
      *
      *    a = A / |A|
      *
      * giving
      *
      *     a = (a[0], a[1], ... , a[n-1])
      *
      * which defines the vector of direction cosines for the line normal to
      * the plane and
      *
      *     d = D / |A|
      *
      */
      double a_mag = vector.getMagnitudeL2();

      /*
      *  The parametric equation for the line normal to the plane is
      *  derived from
      *
      *  For the values for (x[0], x[1], ... , x[n-1]) the line normal to
      *  the plane has the direction cosines, (a[0], a[1], ... a[n - 1])
      *  and the equation for any line perpedicular to the plane can be
      *  expressed relative to a single variable, t0, as
      *
      *  t0 = x[0] / a[0] = x[1] / a[1] = ... = x[n-1] / a[n-1]
      *
      * with the point along the line that lies in the plane giving
      *
      *    t0 = d = D / |A|
      */
      double t0 = dp_term / a_mag;

      /*
      * Giving the normalized equations for points along the line as
      *
      *  t0 = (x[0] - x'[0]) /a[0]       ---> x[0] = x'[0] + a[0] * t0
      *  t0 = (x[1] - x'[1]) /a[1]       ---> x[1] = x'[1] + a[1] * t0
      *     o                                      o
      *     o                                      o
      *     o                                      o
      *  t0 = (x[n-1] - x'[n-1]) / a[n-1] --> x[n-1] = x'[n-1] + a[n-1] * t0
      *
      * where
      *
      *     t0 = d' = D' / |A|
      *
      */


      /*
      *  Thus the result point on the normal through the given point and
      *  lying in the plane is given by:
      *
      *     x[0] = x'[0] + a[0] * t0
      *     x[1] = x'[1] + a[1] * t0
      *          o
      *          o
      *          o
      *     x[n-1] = x'[n-1] + a[n-1] * t0
      *
      * Make
      *
      *    vector = a * t0
      *
      * then add it to the given point to get the point in the plane.
      *
      */
      vector.scale(t0);

      result_vector = new VectorND(this);
      result_vector.add(vector);

      return(result_vector);
      }

   /**
   *  Return true if the vector endpoint is between the given line endpoints,
   *  but not necessarily on the line.  This can be used for checking opposite
   *  corners on a hyper-rectangle to see if this vector endpoint is inside it.
   *  @param point1 the first endpoint.
   *  @param point2 the second endpoint.
   *  @return true if the vector endpoint is between the endpoints in all
   *     dimensions.
   */
   public boolean between(VectorND point1, VectorND point2)
      {
      boolean between = true;

      for (int dim_index = 0;
           dim_index < this.values.length;
           dim_index++)
         {
         /*
         * If any dimension falls outside the interval for that dimension
         * the vector endpoint is outside.
         */
         if (this.values[dim_index] < Math.min(point1.values[dim_index],
                                               point2.values[dim_index]) ||
             Math.max(point1.values[dim_index],
                      point2.values[dim_index]) < this.values[dim_index])
            {
            between = false;
            }
         }

      return(between);
      }

   /**
   * Returns a String that represents the component values of this VectorND.
   * @return the string representation of the vector of values.
   */
   public String toString()
      {
      String out_string = new String("");

      String prefix = "(";
      for (int dim_index = 0;
           dim_index < this.values.length;
           dim_index++)
         {
         out_string += String.format("%s" + format,
                                     prefix,
                                     this.values[dim_index]);
         prefix = ", ";
         }
      out_string += ")";

      return(out_string);
      }

   /**
   * Prints the representation the component values of this VectorND.
   */
   public void Print()
      {
      System.out.print(this);
      }


   /**
   * The Parsable routines.
   */


   /**
   *  Get the basic "locatable" pattern for scanning text for an occurrance
   *  of a possible match..
   *  @return the pattern for locating candidate substrings.
   */
   public static Pattern getBasicPattern()
      {
      if (basic_pattern == null)
         {
         basic_pattern = Pattern.compile(basic_pattern_regex);
         }
      return(basic_pattern);
      }

   /**
   *  Determines whether the basic pattern is in the line.
   *  @param line the string of text to search for the pattern.
   *  @return the matcher for the occrrance of the object's pattern or null
   *     if it is not found.
   */
   public static Matcher findMatchInLine(String line)
      {
      Matcher matcher = null;
      if (basic_pattern == null)
         {
         basic_pattern = Pattern.compile(basic_pattern_regex);
         }
      if (line != null)
         {
         matcher = basic_pattern.matcher(line);
         }
      return(matcher);
      }

   /**
   *  Extracts the string where the basic pattern occurs in the line.
   *  @param line the string of text to search for the pattern.
   *  @return the string that was matched, or null if not found.
   */
/*
   public static String getMatchingString(String line)
      {
      String match = null;
      return(match);
      }
*/

   /**
   *  Parses the given string and returns the value.
   *  @param val_string the input string from which to extract the values.
   *  @return the reference to the new object with the included value
   *     or null if it failed.
   */
   public static VectorND getVectorNDValue(String val_string)
      {
      VectorND value = null;

      if (debug_flag)
         {
         System.out.println("VectorND.getVectorNDValue(string) on ENTRY:\n"
                          + "   val_string = " + val_string + "\n"
                          + "   value_pattern_regex = \""
                          + value_pattern_regex
                          + "\""
                           );
         }
      Matcher value_matcher = null;
      Double [] values = null;

      if (value_pattern == null)
         {
         value_pattern = Pattern.compile(value_pattern_regex);
         }

      if (val_string != null)
         {
         value_matcher = value_pattern.matcher(val_string);
         }

      if (value_matcher != null &&
          value_matcher.find())
         {
         int value_count = value_matcher.groupCount();
         values = new Double[value_count];
         for (int value_index = 1;
              value_index <= value_count;
              value_index++)
            {
            String new_val_string = value_matcher.group(value_index).trim();

            if (debug_flag)
               {
               System.out.println("val_string["
                                + value_index
                                + "]: " + new_val_string);
               }

            values[value_index] = Parsable.parseDoubleValue(new_val_string);
            }

         if (debug_flag)
            {
            String prefix = "   value = (";
            for (int value_index = 1;
                 value_index <= value_count;
                 value_index++)
               {
               System.out.println(prefix + values[value_index]);
               }
            System.out.println(")");
            }
         value = new VectorND(values);
         }
      return(value);
      }


   /**
   *  Parses the given string and returns the value.
   *  @param val_string the input string from which to extract the values.
   *  @return the reference to the new object with the included value
   *     or null if it failed.
   */
   public static Object getValueObject(String val_string)
      {
      Object value_obj = getVectorNDValue(val_string);

      return(value_obj);
      }
   }

