
/*
*  The simulation events dispatcher class.
*/

package util;


import java.util.*;

import java.awt.*;

import util.SimEventListener;
import util.SimEvent;

/**
*  This class defines the sim events dispatcher.  It allows listeners to
*  be registered to receive events based on event type.  When an event is
*  passed to "dispatchEvent()" the listeners registered for the event type
*  are called successively in the order they were registered.
*/
public class SimEventTarget
   {

   /**
   *  The listeners to be informed whenever there is a time update.
   */
   private HashMap<String, ArrayList<SimEventListener>> typed_listeners =
                          new HashMap<String, ArrayList<SimEventListener>>();

   private ArrayList<SimEventListener> untyped_listeners =
                          new ArrayList<SimEventListener>();

   /**
   *  Create an instance of a sim event target object.
   */
   public SimEventTarget()
      {
      }

   /**
   *  Add an event listener for a given event type.  Multiple listeners
   *  for a single event type will be called in the order they are registered.
   *  @param type the event type for this listener.  If null the
   *  listener will be called for every event type before the listeners for
   *  any specific type.
   *  @param listener the event listener to add.
   */
   public void addEventListener(String type,
                                SimEventListener listener)
      {
      if (listener != null)
         {
         if (type == null || type.equals(""))
            {
            if (!untyped_listeners.contains(listener))
               {
               untyped_listeners.add(listener);
               }
            }
         else
            {
            ArrayList<SimEventListener> listeners = null;
            if (typed_listeners.containsKey(type))
               {
               listeners = typed_listeners.get(type);
               }
            else
               {
               listeners = new ArrayList<SimEventListener>();
               typed_listeners.put(type, listeners);
               }
            if (!listeners.contains(listener))
               {
               listeners.add(listener);
               }
            }
         }
      }

   /**
   *  Remove an event listener for a given event type.
   *  @param type the event type for this listener.
   *  @param listener the event listener to remove.
   */
   public void removeEventListener(String type,
                                   SimEventListener listener)
      {
      if (listener != null)
         {
         if (type == null || type.equals(""))
            {
            if (untyped_listeners.contains(listener))
               {
               untyped_listeners.remove(listener);
               }
            }
         else
            {
            ArrayList<SimEventListener> listeners = null;
            if (typed_listeners.containsKey(type))
               {
               listeners = typed_listeners.get(type);
               if (listeners.contains(listener))
                  {
                  listeners.remove(listener);
                  }
               }
            }
         }
      }


   /**
   *  Dispatch an event to the listeners registered for its type.
   *  @param event the event to dispatch to its registered listeners.
   */
   public void dispatchEvent(SimEvent event)
      {
      if (event != null)
         {
         for (SimEventListener listener: untyped_listeners)
            {
            listener.handleEvent(event);
            }
         if (typed_listeners.containsKey(event.type))
            {
            ArrayList<SimEventListener> listeners =
                                       typed_listeners.get(event.type);
            for (SimEventListener listener: listeners)
               {
               listener.handleEvent(event);
               }
            }
         }
      }


   /**
   *  Get a string representation of the queue and its current contents.
   *  @return the string representation of the contents.
   */
   public String toString()
      {
      String out_string = "SimEventTarget listeners:\n";

      if (this.untyped_listeners == null || this.untyped_listeners.size() < 1)
         {
         out_string += "   Untyped Listeners list is empty.\n";
         }
      else
         {
         for (SimEventListener listener: untyped_listeners)
            {
            out_string += "   listener[]: " + listener + "\n";
            }
         }

      if (this.typed_listeners == null ||
          this.typed_listeners.size() < 1)
         {
         out_string += "   Typed Listeners lists are empty.\n";
         }
      else
         {
         for (String type: typed_listeners.keySet())
            {
            ArrayList<SimEventListener> listeners = typed_listeners.get(type);
            if (listeners == null || listeners.size() < 1)
               {
               out_string += "   listener[" + type + "]: list is empty\n";
               }
            else
               {
               for (SimEventListener listener: listeners)
                  {
                  out_string += "   listener["
                              + type
                              + "]: "
                              + listener
                              + "\n";
                  }
               }
            }
         }

      return(out_string);
      }

   /**
   *  Print a string representation of the queue and its current contents.
   */
   public void Print()
      {
      System.out.print(this);
      }

   }

