
package util;

import java.util.*;
import java.io.*;

public class Terms
   {
   /*
   *  This is a class that parses and stores the "terms" and "alias"
   *  strings as needed.  It includes a parser for loading definitions
   *  from file, as well as an output formatting routine (toString()).
   */

   private HashMap <String, Integer> words_hash;
   private ArrayList <String> words;
   private ArrayList <ArrayList<Integer>> word_aliases;
   private ArrayList <Float> word_weights;
   private ArrayList <Boolean> word_base_flags;
   private boolean print_all_flag;

   public Terms()
      {
      words_hash = new HashMap<String, Integer>();
      words = new ArrayList<String>();
      word_weights = new ArrayList<Float>();
      word_aliases = new ArrayList<ArrayList<Integer>>();
      word_base_flags = new ArrayList<Boolean>();
      print_all_flag = false;
      }

   public void set_print_all_flag(boolean d_value)
      {
      print_all_flag = d_value;
      }

   public int get_num_words()
      {
      return(words.size());
      }

   public int get_num_base_words()
      {
      int num_words = 0;
      int word_index;
      for (word_index = 0;
           word_index < words.size();
           word_index++)
         {
         if (word_base_flags.get(word_index))
            {
            num_words++;
            }
         }
      return(num_words);
      }

   public int get_num_alias_words()
      {
      int num_words = 0;
      int word_index;
      for (word_index = 0;
           word_index < words.size();
           word_index++)
         {
         if (!word_base_flags.get(word_index))
            {
            num_words++;
            }
         }
      return(num_words);
      }

   public ArrayList<String> get_words()
      {
      return(words);
      }

   public ArrayList<String> get_base_words()
      {
      ArrayList<String> base_words = new ArrayList<String>();
      int word_index;
      for (word_index = 0;
           word_index < words.size();
           word_index++)
         {
         if (word_base_flags.get(word_index))
            {
            base_words.add(words.get(word_index));
            }
         }
      return(base_words);
      }

   public ArrayList<Integer> get_base_word_indices()
      {
      ArrayList<Integer> base_word_indices = new ArrayList<Integer>();
      int word_index;
      for (word_index = 0;
           word_index < words.size();
           word_index++)
         {
         if (word_base_flags.get(word_index))
            {
            base_word_indices.add(word_index);
            }
         }
      return(base_word_indices);
      }

   public ArrayList<String> get_alias_words()
      {
      ArrayList<String> alias_words = new ArrayList<String>();
      int word_index;
      for (word_index = 0;
           word_index < words.size();
           word_index++)
         {
         if (!word_base_flags.get(word_index))
            {
            alias_words.add(words.get(word_index));
            }
         }
      return(alias_words);
      }

   public int get_index_from_word(String word)
      {
      return(words_hash.get(word));
      }

   public String get_word_from_index(int word_index)
      {
      return(words.get(word_index));
      }

   public ArrayList<Integer> get_alias_indices(String base_word)
      {
      ArrayList<Integer> alias_indices = new ArrayList<Integer>();
      int word_index = words_hash.get(base_word);
      ArrayList<Integer> word_alias_indices = word_aliases.get(word_index);
      ListIterator entry_iter = word_alias_indices.listIterator();
      while (entry_iter.hasNext())
         {
         int alias_index = (Integer) entry_iter.next();
         alias_indices.add(alias_index);
         }
      return(alias_indices);
      }

   public ArrayList<Integer> get_alias_indices_from_index(int word_index)
      {
      ArrayList<Integer> alias_indices = new ArrayList<Integer>();
      ArrayList<Integer> word_alias_indices = word_aliases.get(word_index);
      ListIterator entry_iter = word_alias_indices.listIterator();
      while (entry_iter.hasNext())
         {
         int alias_index = (Integer) entry_iter.next();
         alias_indices.add(alias_index);
         }
      return(alias_indices);
      }

   public ArrayList<String> get_aliases(String base_word)
      {
      ArrayList<String> aliases = new ArrayList<String>();
      int word_index = words_hash.get(base_word);
      ArrayList<Integer> word_alias_indices = word_aliases.get(word_index);
      ListIterator entry_iter = word_alias_indices.listIterator();
      while (entry_iter.hasNext())
         {
         int alias_index = (Integer) entry_iter.next();
         String alias = words.get(alias_index);
         aliases.add(alias);
         }
      return(aliases);
      }

   public String toString()
      {
      String out_string = new String("");

      int word_index;
      for (word_index = 0;
           word_index < words.size();
           word_index++)
         {
         /*
         *  If it is not a base word do NOT print a line for it.
         */
         if (Boolean.valueOf(word_base_flags.get(word_index)) ||
             print_all_flag)
            {
            /*
            *  Write a line for each base word.
            */
            String word = words.get(word_index);
            if (word.matches(".*(?:\"|\\s).*"))
               {
               String find = new String("\\\"");
               String replace = new String("\\\\\\\"");
               /*
               *System.out.println("Find and replace: ("
               *                 + find
               *                 + ") ("
               *                 + replace
               *                 + ")");
               */
            
               String out_word = word.replaceAll(find, replace);
               out_string += "\"" + out_word + "\"";
               }
            else
               {
               out_string += word;
               }
            Float word_weight = word_weights.get(word_index);

            /*
            *  The weight may be missing if there is none.
            */
            if (word_weight != null)
               {
               out_string += " " + word_weight;
               }

            ArrayList <Integer> aliases = word_aliases.get(word_index);
            /*
            *  The aliases may be missing if there are none.
            */
            if (aliases != null && aliases.size() > 1)
               {
               /*
               *  The aliases are there, add them onto the end of the
               *  current line.
               */
               out_string += " (";
               String prefix = "";
               int alias_index;
               for (alias_index = 0;
                    alias_index < aliases.size();
                    alias_index++)
                  {
                  /*
                  *  Each alias may include a quoted portion, and if
                  *  so the quotes need escaped, and the string needs
                  *  to be quoted.
                  */
                  Integer alias_word_index = aliases.get(alias_index);
                  String alias = words.get(alias_word_index);
                  if (alias != word)
                     {
                     if (alias.matches(".*(?:\"|\\s).*"))
                        {
                        String find = new String("\\\"");
                        String replace = new String("\\\\\\\"");
                        String out_alias = alias.replaceAll(find, replace);
                        out_string += prefix + "\"" + out_alias + "\"";
                        }
                     else
                        {
                        out_string += prefix + alias;
                        }
                     prefix = ", ";
                     }
                  }
               out_string += ")";
               }
            out_string += "\n";
            }
         }
      return(out_string);
      }

   public boolean ParseLine(String line)
      {
      boolean line_stored = false;
      boolean error_flag = false;

      int parse_state = 0;

      /*
      *   BNF description of input:
      *      <input> := <line>+<EOF>
      *      <line> := <initial string>\w+[<weight>]\w+[<alias list>]\n
      *      <initial string> := <quoted string>|<unquoted string>
      *      <weight> := \d+
      *      <alias list> := (<alias>[,\w*<alias>]*)
      *      <alias> := <quoted string>|<unquoted string>
      *      <quoted string> := "<arbitrary string>"
      *      <arbitrary string> := [<escaped quote>|<non-quote>]+
      *      <unquoted string> := \W+
      *   where
      *      <...> is a named object type described in the definition.
      *      := relates the named object type to its definition.
      *      <EOF> represents the end of file.
      *      * means zero or more of the preceding.
      *      + means one or more of the preceding.
      *      | means the preceding OR the following, but not both.
      *      [...] means treat as one entry.
      *      \w* means zero or more whitespace characters (excludes newlines).
      *      \w+ means one or more whitespace characters (excludes newlines).
      *      \W+ means one or more NON whitespace characters.
      *      \d+ means one or more decimal digits.
      *      all the other characters used (including " , ( ) and \n) represent
      *         themselves (including newline's \n).
      */


      /*
      *  This tokenizes the string components into
      *     string
      *     quoted string (may include embedded quotes)
      *     digits
      *     delimiter ( ) , or \n
      */

      /*
      *  To parse this string we can define a parser state machine.
      *     0 -- Start new line.
      *     1 -- Getting first character of a double-quote-terminated
      *          initial string.
      *     2 -- In a double-quote-terminated initial string.
      *     3 -- Fetching an initial string escaped character.
      *     4 -- In a white-space-terminated initial string.
      *     5 -- Finished with an initial string.
      *     6 -- Getting a weight.
      *     7 -- Finished with a weight.
      *     8 -- Getting a new alias strings.
      *     9 -- Getting first character of a double-quote-terminated alias
      *          string.
      *    10 -- In a double-quote-terminated alias string.
      *    11 -- Fetching an alias string escaped character.
      *    12 -- In a white-space-terminated alias string.
      *    13 -- Find next alias string.
      *    14 -- Return with line done.
      *    15 -- Return with file done.
      *    16 -- Return with ERROR.
      *
      */

      /*
      *  The essential objects for translating this line.
      */
      String word_string = new String("");
      Integer word_index = null;

      String weight_string = new String("");
      Float weight = null;

      String alias_string = new String("");
      /*
      *   Hash (alias word, word's alias index (into list of aliases for the
      *   current word))
      */
      HashMap <String, Integer>  aliases_hash = new HashMap<String, Integer>();

      /*
      *  Array of word indices for aliases for the current word.
      */
      ArrayList<Integer> aliases = new ArrayList<Integer>();

      /*
      *  Start at the beginning of a new line.
      */

      int char_index;

      char curr_char;

      for (char_index = 0;
           char_index <= line.length() && !error_flag && parse_state < 14;
           char_index++)
         {
         /*
         *  Loop until a termination state is reached or we run out of
         *  characters.
         */
         if (char_index == line.length())
            {
            curr_char = '\n';
            }
         else
            {
            curr_char = line.charAt(char_index);
            }

         /*
         *  The current state determines what needs to be done for each input
         *  character.
         */
/*
         System.out.println("In State "
                            + parse_state
                            + " ("
                            + curr_char
                            + ")");
*/
         boolean new_word_done = false;
         boolean weight_done = false;
         boolean new_alias_done = false;
         switch (parse_state)
            {
            /*
            *  Transition table:
            *     From   Input   To     Action to
            *     State  Char   State   be taken:
            */
            case 0:
/*
               System.out.println("Initial state.");
*/
               /*
               *    Start new line --
               *       0      \n    14     Return with Line Done: line empty
               *       0      \s     0     Leading whitespace: Ignore character.
               *       0       "     1     Starting quoted string: Ignore
                                           character.
               *       0    other    4     Store character into initial string.
               */
               if (curr_char == '\n')
                  {
                  parse_state = 14;
                  }
               else if (Character.isWhitespace(curr_char))
                  {
                  parse_state = 0;
                  }
               else if (curr_char == '\"')
                  {
                  parse_state = 1;
                  }
               else
                  {
                  parse_state = 4;
                  word_string = Character.toString(curr_char);
                  }
               break;
            case 1:
/*
               System.out.println("Quoted initial string first char state.");
*/
               /*
               * Getting first character in initial quoted string --
               *    1      \n    16     Return with ERROR: Line done
               *                        prematurely.
               *    1       "    16     Return with ERROR: empty quoted string.
               *    1       \     3     Escaped character coming: Ignore
               *                        character.
               *    1    other    2     Store first character into initial
               *                        string.
               */
               if (curr_char == '\n')
                  {
                  parse_state = 16;
                  }
               else if (curr_char == '\"')
                  {
                  parse_state = 16;
                  }
               else if (curr_char == '\\')
                  {
                  parse_state = 3;
                  }
               else
                  {
                  parse_state = 2;
                  word_string = Character.toString(curr_char);
                  }
               break;
            case 2:
/*
               System.out.println("In quoted initial string state.");
*/
               /*
               * In a double-quote-terminated initial string --
               *    2      \n    14     Return with Line Done: Save initial
               *                        string.
               *    2       "     5     Quoted string done: Save initial
               *                        string.
               *    2       \     3     Escaped character coming: Ignore
               *                        character.
               *    2    other    2     Append character onto initial
               *                        string.
               */
               if (curr_char == '\n')
                  {
                  parse_state = 14;
                  }
               else if (curr_char == '\"')
                  {
                  parse_state = 5;
                  new_word_done = true;
                  }
               else if (curr_char == '\\')
                  {
                  parse_state = 3;
                  }
               else
                  {
                  parse_state = 2;
                  word_string += curr_char;
                  }
               break;
            case 3:
/*
               System.out.println("In quoted initial string esc char state.");
*/
               /*
               * Fetching an initial string escaped character --
               *    3     any     2     Append character onto initial
               *                        string.
               */
               parse_state = 2;
               word_string += Character.toString(curr_char);
               break;
            case 4:
/*
               System.out.println("In white-space-terminated initial string"
                                + " state.");
*/
               /*
               * In a white-space-terminated initial string --
               *    4      \n    14     Return with Line done: Save initial
               *                        string.
               *    4      \s     5     String done: Save initial string.
               *    4    other    4     Append character onto initial string.
               */
               if (curr_char == '\n')
                  {
                  parse_state = 14;
                  new_word_done = true;
                  }
               else if (Character.isWhitespace(curr_char))
                  {
                  parse_state = 5;
                  new_word_done = true;
                  }
               else
                  {
                  parse_state = 4;
                  word_string += curr_char;
                  }
               break;
            case 5:
/*
               System.out.println("Finished initial string state.");
*/
               /*
               * Finished with an initial string --
               *    5      \n    14     Return with Line done: Save initial
               *                        string.
               *    5      \s     5     White space gap: Ignore character.
               *    5       (     8     Loading aliases: Ignore character.
               *    5      \d     6     Loading weight: Store weight digit.
               *    5   other    16     ERROR: Bad character sequence.
               */
               if (curr_char == '\n')
                  {
                  parse_state = 14;
                  }
               else if (Character.isWhitespace(curr_char))
                  {
                  parse_state = 5;
                  }
               else if (Character.isDigit(curr_char))
                  {
                  parse_state = 6;
                  weight_string = Character.toString(curr_char);
                  }
               else if (curr_char == '.')
                  {
                  parse_state = 6;
                  weight_string = Character.toString(curr_char);
                  }
               else if (curr_char == '(')
                  {
                  parse_state = 8;
                  }
               else
                  {
                  parse_state = 16;
                  }
               break;
            case 6:
/*
               System.out.println("In weight string state.");
*/
               /*
               * Getting a weight --
               *    6      \n    14     Return with Line done: Save weight
               *                        number.
               *    5      \s     7     White space gap: Ignore character.
               *    5       (     8     Loading aliases: Ignore character.
               *    6      \d     6     More digits: Append weight digit.
               *    6    other    7     Number done: Save weight number.
               */
               if (curr_char == '\n')
                  {
                  parse_state = 14;
                  weight_done = true;
                  }
               else if (Character.isDigit(curr_char))
                  {
                  parse_state = 6;
                  weight_string += curr_char;
                  }
               else if (curr_char == '.')
                  {
                  parse_state = 6;
                  weight_string += curr_char;
                  }
               else if (Character.isWhitespace(curr_char))
                  {
                  parse_state = 7;
                  weight_done = true;
                  }
               else if (curr_char == '(')
                  {
                  parse_state = 8;
                  weight_done = true;
                  }
               else
                  {
                  parse_state = 7;
                  weight_done = true;
                  }
               break;
            case 7:
/*
               System.out.println("Finished weight string state.");
*/
               /*
               * Finished with a weight --
               *    7      \n    14     Return with Line done: Ignore
               *                        character, save weight number.
               *    7       (     8     Ignore character.
               *    7    other    7     Ignore character.
               */
               if (curr_char == '\n')
                  {
                  parse_state = 14;
                  new_alias_done = true;
                  }
               else if (curr_char == '(')
                  {
                  parse_state = 8;
                  }
               else
                  {
                  parse_state = 7;
                  }
               break;
            case 8:
/*
               System.out.println("Start new alias string state.");
*/
               /*
               * Getting a new alias string --
               *    8      \n    14     Return with Line done: Ignore
               *                        character.
               *    8      \s     8     Leading whitespace: Ignore
               *                        character.
               *    8       "     9     Starting quoted string: Ignore
               *                        character.
               *    8    other   12     Unquoted String: Store character
               *                        into next alias string.
               */
               if (curr_char == '\n')
                  {
                  parse_state = 14;
                  }
               else if (Character.isWhitespace(curr_char))
                  {
                  parse_state = 8;
                  }
               else if (curr_char == '"')
                  {
                  parse_state = 9;
                  }
               else
                  {
                  parse_state = 12;
                  alias_string = Character.toString(curr_char);
                  }
               break;
            case 9:
/*
               System.out.println("Quoted alias string first char state.");
*/
               /*
               * Getting first character in alias quoted string --
               *    9      \n    16     Return with ERROR: Line done
               *                        prematurely.
               *    9       "    16     Return with ERROR: empty quoted string.
               *    9    other   10     Store first character into next
               *                        alias string.
               */
               if (curr_char == '\n')
                  {
                  parse_state = 16;
                  }
               else if (curr_char == '"')
                  {
                  parse_state = 16;
                  }
               else
                  {
                  parse_state = 10;
                  alias_string = Character.toString(curr_char);
                  }
               break;
            case 10:
/*
               System.out.println("In quoted alias string state.");
*/
               /*
               * In a double-quote-terminated alias string --
               *   10      \n    14     Return with Line done: Save alias
               *                        string.
               *   10       "    12     Quoted string done: Save alias string.
               *   10       \    11     Escaped character coming: Ignore
               *                        character.
               *   10    other   10     Append character onto alias string.
               */
               if (curr_char == '\n')
                  {
                  parse_state = 14;
                  }
               else if (curr_char == '"')
                  {
                  parse_state = 13;
                  new_alias_done = true;
                  }
               else if (curr_char == '\\')
                  {
                  parse_state = 11;
                  }
               else
                  {
                  parse_state = 10;
                  alias_string += curr_char;
                  }
               break;
            case 11:
/*
               System.out.println("Quoted alias string esc char state.");
*/
               /*
               * Fetching an alias string escaped character --
               *   11     any    10     Append character onto initial string.
               */
               parse_state = 10;
               alias_string += curr_char;
               break;
            case 12:
/*
               System.out.println("White-space-terminated alias string state.");
*/
               /*
               * In a white-space-terminated alias string --
               *   12      \n    14     Return with Line done: Save alias
               *                        string.
               *   12       ,     8     String done: Save alias string.
               *   12       )    14     Return with Line done: Save alias
               *                        string.
               *   12      \s    13     String done: Save alias string.
               *   12    other   12     Append character onto alias string.
               */
               if (curr_char == '\n')
                  {
                  parse_state = 14;
                  }
               else if (curr_char == ',')
                  {
                  parse_state = 8;
                  new_alias_done = true;
                  }
               else if (curr_char == ')')
                  {
                  parse_state = 14;
                  new_alias_done = true;
                  }
               else if (Character.isWhitespace(curr_char))
                  {
                  parse_state = 13;
                  new_alias_done = true;
                  }
               else
                  {
                  parse_state = 12;
                  alias_string += curr_char;
                  }
               break;
            case 13:
/*
               System.out.println("Find next alias string state.");
*/
               /*
               * Find next alias state --
               *   13      \n    14     Return with Line done: unterminated
               *                        line.
               *   13       ,     8     Comma separator: ignore character.
               *   13       )    15     Return with Line done: okay
               *   13      \s    13     White space: ignore character.
               *   13   other    12     Start of next alias: Store to next
               *                        alias.
               */
               if (curr_char == '\n')
                  {
                  parse_state = 14;
                  }
               else if (curr_char == ',')
                  {
                  parse_state = 8;
                  }
               else if (curr_char == ')')
                  {
                  parse_state = 14;
                  }
               else if (Character.isWhitespace(curr_char))
                  {
                  parse_state = 13;
                  }
               else
                  {
                  parse_state = 12;
                  alias_string = Character.toString(curr_char);
                  }
               break;
            default:
/*
               System.out.println("Default (other) states: " + parse_state);
*/
               /*
               * Line done state --
               *   14    Return with line done.
               */
               /*
               * File done state --
               *   15    Return with file done.
               */
               /*
               * ERROR state --
               *   16    Return with ERROR.
               */
            }

         if (new_word_done && word_string.length() > 0)
            {
            /*
            *  This pass finished inputting the characters for a new word.
            *  See if it has been stored already.
            */
            word_index = words_hash.get(word_string);
            if (word_index == null)
               {
               /*
               *  It is a new word that has not been stored.
               *     1. Get its index and add it to the array of words.
               *     2. Put it and its index into the words hash.
               */
               word_index = words.size();
               words.add(word_string);
               words_hash.put(word_string, new Integer(word_index));

               /*
               *  Every word is its own alias, so add its index to the
               *  current line's alias list.
               */
               aliases.add(word_index);
               word_aliases.add(aliases);
               word_weights.add(null);
               word_base_flags.add(true);
               }
            else
               {
               /*
               *  It is an existing word.  It may be an alias in an
               *  earlier entry, or it may be a duplicate entry in the file.
               *     1. Get its current alias_word_index list.
               *     2. Put all its aliases onto the current word's
               *        aliases_hash.
               */
               aliases = word_aliases.get(word_index);
               int alias_index;
               for (alias_index = 0;
                    alias_index < aliases.size();
                    alias_index++)
                  {
                  Integer alias_word_index = aliases.get(alias_index);
                  String alias = words.get(alias_word_index);
                  aliases_hash.put(alias, alias_index);
                  }
               /*
               *  Whatever it was it is now a base word.
               */
               word_base_flags.set(word_index, true);
               }
            }

         if (weight_done)
            {
            /*
            *  We can test to see if a weight was stored or not,
            *  and if it was, store its value, otherwise store null.
            */
            String regex = new String("(?:\\.\\d+)|(?:\\d+\\.[\\d]*)|(?:\\d+)");
/*
            System.out.println("weight_string = ("
                             + weight_string
                             + ") ("
                             + regex
                             + ").\n");
*/
            weight = null;
            if (weight_string.equals(""))
               {
/*
               System.out.println("No weight string.\n");
*/
               }
            else if (weight_string.matches(regex))
               {
               weight = new Float(weight_string);
/*
               System.out.println("No weight match ("
                                 + weight
                                 + ").\n");
*/
               }
            else
               {
/*
               System.out.println("No weight match.\n");
*/
               }
            word_weights.set(word_index, weight);
            }

         if (new_alias_done)
            {
            if (alias_string.length() > 0)
               {
               /*
               *  See if it was already listed in the list of aliases.
               */
               Integer alias_index = aliases_hash.get(alias_string);
               /*
               *  See if it has already been stored as a word.
               */
               Integer alias_word_index = words_hash.get(alias_string);
               if (alias_word_index == null)
                  {
                  /*
                  *  It is a new word, give it the next index position.
                  */
                  alias_word_index = words.size();
                  words.add(alias_string);
                  words_hash.put(alias_string, alias_word_index);

                  /*
                  *  Give the word the current weight.
                  */
                  word_weights.add(weight);

                  /*
                  *  This is an alias, not a base word.
                  */
                  word_base_flags.add(false);

                  /*
                  *  Add it to the list of aliases for the current word.
                  */
                  aliases.add(alias_word_index);
                  word_aliases.add(aliases);
                  }
               else if (alias_index != null)
                  {
                  /*
                  *  It is already in the list of aliases for this word,
                  *  and therefore already processed, so ignore it.
                  */
                  }
               else
                  {
                  /*
                  *  It is not a duplicate alias. Store it as an alias.
                  */
                  alias_index = aliases.size();
                  aliases.add(alias_word_index);
                  aliases_hash.put(alias_string, alias_index);

                  /*
                  *  The word does already exist and was not in the current
                  *  word's list of aliases, so any aliases it has
                  *  should also be in the current list of aliases also.
                  *     1. update the list of aliases.
                  *     2. store the updated list as the common aliases list
                  *        for the alias word and each of its aliases.
                  *
                  *      This is a "chasing the transitivity" problem that
                  *      needs handled and is not trivial.
                  *
                  *         DUMMIED -- Chasing alias transitivity.
                  */
                  }

               /*
               *  Reset the alias string.
               */
               alias_string = new String();
               }
            }
         }

      if (parse_state == 16)
         {
         System.out.println("Line processing error.\n");
         error_flag = true;
         }
      else if (parse_state == 14)
         {
/*
         System.out.println("Line processed:\n");

         System.out.println("word_string = (" + word_string + ")\n");

         if (weight == null)
            {
            System.out.println("weight = ( NONE )\n");
            }
         else
            {
            System.out.println("weight = (" + weight + ")\n");
            }

         if (aliases.size() == 1)
            {
            System.out.println("There is 1 alias.\n");
            }
         else
            {
            System.out.println("There are " + aliases.size() + " aliases.\n");
            }
*/

   
         int alias_index;
         for (alias_index = 0;
              alias_index < aliases.size();
              alias_index++)
            {
            Integer alias_word_index = aliases.get(alias_index);
            String alias = words.get(alias_word_index);
/*
            System.out.println("   alias["
                             + alias_index
                             + "] ("
                             + alias_word_index
                             + ") = ("
                             + alias
                             + ")\n");
*/
            }
         }
      return(error_flag);
      }

   public boolean LoadFile(String file_path)
      {
      boolean error_flag = false;

      BufferedReader in_file;

      try
         {
         in_file = new BufferedReader(new FileReader(file_path));
         String in_line;
         /*
         *  Process all the lines in the file.
         */
         while ((in_line = in_file.readLine()) != null)
            {
            ParseLine(in_line);
            }
         /*
         *System.out.println("File word_hash:" + words_hash);
         */
         in_file.close();
         }
      catch (FileNotFoundException e)
         {
         System.err.println("Could not open " + file_path);
         error_flag = true;
         }
      catch (Exception e)
         {
         System.err.println("Error opening " + file_path + "\n" + e);
         error_flag = true;
         }

      return(error_flag);
      }

   public boolean ParseSimpleLine(String line)
      {
      boolean word_stored = false;

      /*
      *  Get rid of leading and trailing blanks.
      */
      String new_word = line.trim();
      if (new_word != null && new_word.length() > 0)
         {
         Integer word_index = words_hash.get(new_word);
         if (word_index == null)
            {
            /*
            *System.out.println("word ("
            *                 + new_word.length()
            *                 + "):\n"
            *                 + "\""
            *                 + new_word
            *                 + "\"");
            */
            word_index = words.size();
            words_hash.put(new_word, word_index);
            words.add(new_word);

            word_weights.add(null);
            word_aliases.add(new ArrayList<Integer>());
            word_base_flags.add(true);

            word_stored = true;
            }
         }

      return(word_stored);
      }

   public boolean LoadSimpleFile(String file_path)
      {
      boolean error_flag = false;

      BufferedReader in_file;

      try
         {
         in_file = new BufferedReader(new FileReader(file_path));
         String in_line;
         /*
         *  Process all the lines in the file.
         */
         while ((in_line = in_file.readLine()) != null)
            {
            ParseSimpleLine(in_line);
            }
         /*
         *System.out.println("File word_hash:" + words_hash);
         */
         in_file.close();
         }
      catch (FileNotFoundException e)
         {
         System.err.println("Could not open " + file_path);
         error_flag = true;
         }
      catch (Exception e)
         {
         System.err.println("Error opening " + file_path + "\n" + e);
         error_flag = true;
         }

      return(error_flag);
      }
   }
