
package util;

import java.util.*;
import java.io.*;

import util.*;

/**
*  This class allows storing 3-D vector components.
*/
public class Vector3D
   {

   public double x = 0.0;
   public double y = 0.0;
   public double z = 0.0;

   /**
   *  Creates a Vector3D object with (0, 0, 0) as the component values.
   */
   public Vector3D()
      {
      this.x = 0.0;
      this.y = 0.0;
      this.z = 0.0;
      }

   /**
   *  Creates a Vector3D object with (x, y, and z) component values.
   *  @param x the x-component.
   *  @param y the y-component.
   *  @param z the z-component.
   */
   public Vector3D(double x, double y, double z)
      {
      this.x = x;
      this.y = y;
      this.z = z;
      }

   /**
   *  Creates a Vector3D object from a Vector3D argument's component values.
   *  @param vector the other vector.
   */
   public Vector3D(Vector3D vector)
      {
      this.x = vector.x;
      this.y = vector.y;
      this.z = vector.z;
      }

   /**
   *  Creates a Vector3D object from a Point3D argument's component values.
   *  @param point the point.
   */
   public Vector3D(Point3D point)
      {
      this.x = point.x;
      this.y = point.y;
      this.z = point.z;
      }

   /**
   * Creates a new Vector3D object from the Vector3D's component values.
   * @return the new Vector3D object reference.
   */
   public Vector3D clone()
      {
      Vector3D vector = new Vector3D(this);
      return(vector);
      }

   /**
   * Returns the X component of this Vector3D in double precision.
   * @return the X component value.
   */
   public double getX()
      {
      return(this.x);
      }

   /**
   * Returns the Y component of this Vector3D in double precision.
   * @return the Y component value.
   */
   public double getY()
      {
      return(this.y);
      }

   /**
   * Returns the Z component of this Vector3D in double precision.
   * @return the Z component value.
   */
   public double getZ()
      {
      return(this.z);
      }

   /**
   *  Sets the Vector3D component values to the x, y, and Z component values.
   *  @param x the x-component.
   *  @param y the y-component.
   *  @param z the z-component.
   */
   public void setComponents(double x, double y, double z)
      {
      this.x = x;
      this.y = y;
      this.z = z;
      }

   /**
   *  Sets the Vector3D object to the x, y, and Z component values of
   *  another vector.
   *  @param vector the other vector.
   */
   public void setLocation(Vector3D vector)
      {
      this.x = vector.x;
      this.y = vector.y;
      this.z = vector.z;
      }

   /**
   *  Returns the Euclidean magnitude of these vector components
   *  @param x the X component
   *  @param y the Y component
   *  @param z the Z component
   */
   public static double magnitude(double x, double y, double z)
      {
      double sumsq = x * x + y * y + z * z;
      double mag = Math.sqrt(sumsq);
      
      return(mag);
      }

   /**
   *  Returns the Euclidian magnitude of the Vector3D.
   */
   public double magnitude()
      {
      double sumsq = this.x * this.x +
                     this.y * this.y +
                     this.z * this.z;
      double mag = Math.sqrt(sumsq);
      
      return(mag);
      }


   /**
   *  Returns the scaled vector.
   *  @param scale the scale multiplier.
   *  @return a scaled vector.
   */
   public Vector3D scale(double scale)
      {
      double x_val = this.x * scale;
      double y_val = this.y * scale;
      double z_val = this.z * scale;

      Vector3D vector = new Vector3D(x_val, y_val, z_val);
      
      return(vector);
      }


   /**
   *  Returns the difference between the two Vector3D vectors.
   *  @param vector1 the first vector
   *  @param vector2 the second vector
   */
   public static Vector3D diff(Vector3D vector1, Vector3D vector2)
      {
      double diff_x = vector1.x - vector2.x;
      double diff_y = vector1.y - vector2.y;
      double diff_z = vector1.z - vector2.z;

      Vector3D vector = new Vector3D(diff_x, diff_y, diff_z);
      
      return(vector);
      }

   /**
   *  Returns the difference between the two component vectors.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param z1 the first location's Z component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @param z2 the second location's Z component
   *  @return the difference Vector3D.
   */
   public static Vector3D diff(double x1, double y1, double z1,
                               double x2, double y2, double z2)
      {
      Vector3D diff_vector = new Vector3D(x1 - x2, y1 - y2, z1 - z2);
      
      return(diff_vector);
      }

   /**
   *  Returns the difference between the two Vector3D vectors.
   *  @param vector the other vector
   *  @return the difference Vector3D.
   */
   public Vector3D diff(Vector3D vector)
      {
      double diff_x = this.x - vector.x;
      double diff_y = this.y - vector.y;
      double diff_z = this.z - vector.z;

      Vector3D diff_vector = new Vector3D(diff_x, diff_y, diff_z);
      
      return(diff_vector);
      }

   /**
   *  Returns the difference between the two component vectors.
   *  @param x the vector's X component
   *  @param y the vector's Y component
   *  @param z the vector's Z component
   *  @return the difference Vector3D.
   */
   public Vector3D diff(double x, double y, double z)
      {
      Vector3D diff_vector = new Vector3D(this.x - x, this.y - y, this.z - z);
      
      return(diff_vector);
      }


   /**
   *  Returns the sum of the two Vector3D vectors.
   *  @param vector1 the first vector
   *  @param vector2 the second vector
   *  @return the difference Vector3D.
   */
   public static Vector3D sum(Vector3D vector1, Vector3D vector2)
      {
      double sum_x = vector1.x + vector2.x;
      double sum_y = vector1.y + vector2.y;
      double sum_z = vector1.z + vector2.z;

      Vector3D diff_vector = new Vector3D(sum_x, sum_y, sum_z);
      
      return(diff_vector);
      }


   /**
   *  Returns the sum of the two component vectors.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param z1 the first location's Z component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @param z2 the second location's Z component
   *  @return the sum Vector3D.
   */
   public static Vector3D sum(double x1, double y1, double z1,
                              double x2, double y2, double z2)
      {
      Vector3D sum_vector = new Vector3D(x1 + x2, y1 + y2, z1 + z2);
      
      return(sum_vector);
      }

   /**
   *  Returns the sum of the two Vector3D vectors.
   *  @param vector the other vector
   *  @return the sum Vector3D.
   */
   public Vector3D sum(Vector3D vector)
      {
      double sum_x = this.x + vector.x;
      double sum_y = this.y + vector.y;
      double sum_z = this.z + vector.z;

      Vector3D sum_vector = new Vector3D(sum_x, sum_y, sum_z);
      
      return(sum_vector);
      }

   /**
   *  Returns the sum of the two component vectors.
   *  @param x the vector's X component
   *  @param y the vector's Y component
   *  @param z the vector's Z component
   *  @return the sum Vector3D.
   */
   public Vector3D sum(double x, double y, double z)
      {
      Vector3D sum_vector = new Vector3D(this.x + x, this.y + y, this.z + z);
      
      return(sum_vector);
      }


   /**
   *  Returns the vector dot product between the given Vector3D vectors.
   *  @param vector1 the first vector.
   *  @param vector2 the second vector.
   *  @return the dot product scalar.
   */
   public static double dotProd(Vector3D vector1, Vector3D vector2)
      {
      double dot_prod = vector1.x * vector2.x +
                        vector1.y * vector2.y +
                        vector1.z * vector2.z;
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given
   *  vector components.
   *  @param x1 the first location's X component
   *  @param y1 the first location's Y component
   *  @param z1 the first location's Z component
   *  @param x2 the second location's X component
   *  @param y2 the second location's Y component
   *  @param z2 the second location's Z component
   *  @return the dot product scalar.
   */
   public static double dotProd(double x1, double y1, double z1,
                                double x2, double y2, double z2)
      {
      double dot_prod = x1 * x2 + y1 * y2 + z1 * z2;
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given Vector3D.
   *  @param vector the other vector.
   *  @return the dot product scalar.
   */
   public double dotProd(Vector3D vector)
      {
      double dot_prod = this.x * vector.x +
                        this.y * vector.y +
                        this.z * vector.z;
      
      return(dot_prod);
      }

   /**
   *  Returns the scalar value of the dot product with the given
   *  vector components.
   *  @param x the X component
   *  @param y the Y component
   *  @param z the Z component
   *  @return the dot product scalar.
   */
   public double dotProd(double x, double y, double z)
      {
      double dot_prod = this.x * x + this.y * y + this.z * z;
      
      return(dot_prod);
      }


   /**
   *  Returns the vector from the cross product of the two vectors.
   *  @param vector1 the first vector.
   *  @param vector2 the second vector.
   *  @return the cross product Vector3D.
   */
   public static Vector3D crossProd(Vector3D vector1, Vector3D vector2)
      {
      double prod_x = vector1.y * vector2.z - vector1.z * vector2.y;
      double prod_y = vector1.z * vector2.x - vector1.x * vector2.z;
      double prod_z = vector1.x * vector2.y - vector1.y * vector2.x;

      Vector3D cross_prod = new Vector3D(prod_x, prod_y, prod_z);
      
      return(cross_prod);
      }

   /**
   *  Returns the vector from the cross product of the two vector
   *  component triplets.
   *  @param x1 the first vector's X component
   *  @param y1 the first vector's Y component
   *  @param z1 the first vector's Z component
   *  @param x2 the second vector's X component
   *  @param y2 the second vector's Y component
   *  @param z2 the second vector's Z component
   *  @return the cross product Vector3D.
   */
   public static Vector3D crossProd(double x1, double y1, double z1,
                                    double x2, double y2, double z2)
      {
      double prod_x = y1 * z2 - z1 * y2;
      double prod_y = z1 * x2 - x1 * z2;
      double prod_z = x1 * y2 - y1 * x2;

      Vector3D cross_prod = new Vector3D(prod_x, prod_y, prod_z);
      
      return(cross_prod);
      }

   /**
   *  Returns the vector from the cross product with the given Vector3D.
   *  @param vector the other vector.
   *  @return the cross product Vector3D.
   */
   public Vector3D crossProd(Vector3D vector)
      {
      double prod_x = this.y * vector.z - this.z * vector.y;
      double prod_y = this.z * vector.x - this.x * vector.z;
      double prod_z = this.x * vector.y - this.y * vector.x;

      Vector3D cross_prod = new Vector3D(prod_x, prod_y, prod_z);
      
      return(cross_prod);
      }

   /**
   *  Returns the vector from the cross product with the given vector
   *  components.
   *  @param x the X component
   *  @param y the Y component
   *  @param z the Z component
   *  @return the cross product Vector3D.
   */
   public Vector3D crossProd(double x, double y, double z)
      {
      double prod_x = this.y * z - this.z * y;
      double prod_y = this.z * x - this.x * z;
      double prod_z = this.x * y - this.y * x;

      Vector3D cross_prod = new Vector3D(prod_x, prod_y, prod_z);
      
      return(cross_prod);
      }


   /**
   *  Normalizes the vector.
   *  @return true if a the magnitude was 0 so it couldn't be normalized.
   */
   public boolean normalize()
      {
      boolean error_flag = true;
      double sumsq = this.x * this.x +
                     this.y * this.y +
                     this.z * this.z;
      double mag = Math.sqrt(sumsq);
      if (mag != 0.0)
         {
         double mag_i = 1.0 / mag;

         this.x *= mag_i;
         this.y *= mag_i;
         this.z *= mag_i;
         error_flag = false;
         }
      
      return(error_flag);
      }


   /**
   *  Returns the distance from this Point3D to the specified coordinate.
   *  @param x the X coordinate
   *  @param y the Y coordinate
   *  @param z the Z coordinate
   */
   public double distance(double x, double y, double z)
      {
      double distance = 0.0;
      double x_offset = this.x - x;
      double y_offset = this.y - y;
      double z_offset = this.z - z;

      double sumsq = x_offset * x_offset +
                     y_offset * y_offset +
                     z_offset * z_offset;
      
      distance = Math.sqrt(sumsq);
      
      return(distance);
      }

   /**
   *  Returns the distance from this Point3D to the specified Point3D.
   *  @param point the other point.
   */
   public double distance(Point3D point)
      {
      double distance = 0.0;
      double x_offset = this.x - point.x;
      double y_offset = this.y - point.y;
      double z_offset = this.z - point.z;

      double sumsq = x_offset * x_offset +
                     y_offset * y_offset +
                     z_offset * z_offset;
      
      distance = Math.sqrt(sumsq);
      
      return(distance);
      }


   /**
   *  Returns the distance squared from this Point3D to the specified
   *  coordinate.
   *  @param x the X coordinate
   *  @param y the Y coordinate
   *  @param z the Z coordinate
   */
   public double distanceSQ(double x, double y, double z)
      {
      double x_offset = this.x - x;
      double y_offset = this.y - y;
      double z_offset = this.z - z;

      double sumsq = x_offset * x_offset +
                     y_offset * y_offset +
                     z_offset * z_offset;
      
      return(sumsq);
      }

   /**
   *  Returns the distance squared from this Point3D to the specified Point3D.
   *  @param point the other point.
   */
   public double distanceSQ(Point3D point)
      {
      double x_offset = this.x - point.x;
      double y_offset = this.y - point.y;
      double z_offset = this.z - point.z;

      double sumsq = x_offset * x_offset +
                     y_offset * y_offset +
                     z_offset * z_offset;
      
      return(sumsq);
      }

   /**
   *  Returns the distance squared between the two coordinate locations.
   *  @param x1 the first location's X coordinate
   *  @param y1 the first location's Y coordinate
   *  @param z1 the first location's Z coordinate
   *  @param x2 the second location's X coordinate
   *  @param y2 the second location's Y coordinate
   *  @param z2 the second location's Z coordinate
   */
   public static double distanceSQ(double x1, double y1, double z1,
                                   double x2, double y2, double z2)
      {
      double x_offset = x1 - x2;
      double y_offset = y1 - y2;
      double z_offset = z1 - z2;

      double sumsq = x_offset * x_offset +
                     y_offset * y_offset +
                     z_offset * z_offset;
      
      return(sumsq);
      }

   /**
   * Returns a String that represents the value of this Vector3D.
   * @return the string representation of the vector.
   */
   public String toString()
      {
      String out_string = new String("");

      out_string = "(" + this.x + ", " + this.y + ", " + this.z + ")";
      return(out_string);
      }

   /**
   * Prints the representation the value of this Vector3D.
   */
   public void Print()
      {
      System.out.print("(" + this.x + ", " + this.y + ", " + this.z + ")");
      }
   }

